'use strict';

var _extends = Object.assign || function (target) {
    for (var i = 1; i < arguments.length; i++) {
        var source = arguments[i];for (var key in source) {
            if (Object.prototype.hasOwnProperty.call(source, key)) {
                target[key] = source[key];
            }
        }
    }return target;
};

function _objectWithoutProperties(obj, keys) {
    var target = {};for (var i in obj) {
        if (keys.indexOf(i) >= 0) continue;if (!Object.prototype.hasOwnProperty.call(obj, i)) continue;target[i] = obj[i];
    }return target;
}

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Default render for options
 * @type {ReactClass}
 */
var DefaultComponent = React.createClass({
    displayName: 'DefaultComponent',

    propTypes: {
        option: itemShape
    },

    render: function render() {
        var option = this.props.option;

        return React.createElement('span', null, option);
    }
});

/**
 * Component to render a Selection option
 * @type {ReactClass}
 */
var SelectOption = React.createClass({
    displayName: 'SelectOption',

    propTypes: {
        item: itemShape.isRequired,
        index: React.PropTypes.number.isRequired,
        // Function to render the option to a string or element
        component: React.PropTypes.func.isRequired,
        // Function to check if option is in selected values
        hasValue: React.PropTypes.func.isRequired,
        // Toggle an option in main Select state
        onToggleOption: React.PropTypes.func.isRequired,
        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            isOptionDisabled: function isOptionDisabled() {
                return false;
            }
        };
    },
    render: function render() {
        var _props = this.props,
            item = _props.item,
            index = _props.index,
            isOptionDisabled = _props.isOptionDisabled,
            hasValue = _props.hasValue,
            onToggleOption = _props.onToggleOption,
            Component = _props.component;

        // Check if item should be displayed but marked as disabled

        var isDisabled = isOptionDisabled(item);

        var className = classNames('SelectOption', {
            active: hasValue(item),
            disabled: isDisabled
        });

        return React.createElement('div', { className: className,
            onClick: function onClick(e) {
                if (!isDisabled) {
                    onToggleOption(item);
                }
            } }, React.createElement(Component, { option: item, index: index }));
    }
});

/**
 * Component to render a Select option group
 * @type {ReactClass}
 */
var SelectOptGroup = React.createClass({
    displayName: 'SelectOptGroup',

    propTypes: {
        group: groupShape
    },

    render: function render() {
        var _props2 = this.props,
            group = _props2.group,
            props = _objectWithoutProperties(_props2, ['group']);

        return React.createElement('div', { className: 'SelectOptGroup' }, group.label ? React.createElement('div', { className: 'GroupLabel' }, group.label) : null, React.createElement('div', { className: 'GroupOptions' }, group.options.map(function (item, i) {
            return React.createElement(SelectOption, _extends({}, props, { key: i, item: item, index: i }));
        })));
    }
});

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render a message when search did not return any results
        searchEmptyComponent: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool,

        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: DefaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select',
            searchEmptyComponent: null
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },

    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var options = props.options,
            groups = props.groups;

        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },

    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },

    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },

    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },

    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },

    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },

    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },

    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;

        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },

    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },

    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },

    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props3 = this.props,
            disabled = _props3.disabled,
            block = _props3.block,
            multiple = _props3.multiple,
            placeholder = _props3.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement('span', { className: 'SelectSelections' }, values.map(function (val, i) {
                return React.createElement('span', { key: i, className: 'SelectSelection' }, React.createElement(ComponentSelection, { option: val, index: i }));
            }));
        } else {
            inner = React.createElement('span', { className: 'SelectPlaceholder' }, placeholder);
        }

        return React.createElement(Button, { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle }, inner, ' ', React.createElement(Button.Caret, null));
    },

    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _this = this;

        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups,
            query = _state3.query;

        var _props4 = this.props,
            search = _props4.search,
            searchPlaceholder = _props4.searchPlaceholder,
            filter = _props4.filter,
            SearchEmptyComponent = _props4.searchEmptyComponent,
            props = _objectWithoutProperties(_props4, ['search', 'searchPlaceholder', 'filter', 'searchEmptyComponent']);

        var className = classNames('SelectContainer', {
            'open': opened
        });

        // Filter empty groups based on search query
        var filteredGroups = groups.map(function (group) {
            return _extends({}, group, {
                options: group.options.filter(function (item, i) {
                    return filter(query, item, i);
                })
            });
        }).filter(function (group) {
            return group.options.length > 0;
        });

        return React.createElement('div', { className: className }, search ? React.createElement('div', { className: 'SelectSearch' }, React.createElement(Input, { ref: 'searchInput',
            value: query,
            onChange: this.onSearchChanged,
            placeholder: searchPlaceholder
        })) : null, Boolean(filteredGroups.length) ? React.createElement('div', { className: 'SelectGroups' }, filteredGroups.map(function (group, i) {
            return React.createElement(SelectOptGroup, _extends({}, props, { key: i, group: group, hasValue: _this.hasValue, onToggleOption: _this.onToggleOption }));
        })) : null, search && !filteredGroups.length ? React.createElement('div', { className: 'SearchEmpty' }, React.createElement(SearchEmptyComponent, { query: query })) : null);
    },
    render: function render() {
        var _props5 = this.props,
            name = _props5.name,
            block = _props5.block;
        var opened = this.state.opened;

        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement('div', { className: className, onClick: function onClick(e) {
                return e.stopPropagation();
            } }, React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }), this.renderButton(), opened ? React.createElement(Backdrop, { onClose: this.close }, this.renderGroups()) : null);
    }
});

module.exports = Select;