'use strict';

var React = require('react');
var classNames = require('classnames');

var Input = require('./Input');
var Spinner = require('./Spinner');

var KEYCODE_ENTER = 13;
var KEYCODE_UP = 38;
var KEYCODE_DOWN = 40;

var Autocomplete = React.createClass({
    displayName: 'Autocomplete',

    propTypes: {
        onFetch: React.PropTypes.func.isRequired,
        renderResult: React.PropTypes.func.isRequired,
        // Called when onEnter on the input (no result selected)
        onEnter: React.PropTypes.func.isRequired,
        onPaste: React.PropTypes.func,
        // Focus events
        onFocus: React.PropTypes.func,
        onBlur: React.PropTypes.func,
        // Control the event
        value: React.PropTypes.string,
        // Called when typing
        onChange: React.PropTypes.func.isRequired,
        // Called when selecting an entry
        onSelect: React.PropTypes.func.isRequired,
        // Render options
        placeholder: React.PropTypes.string,
        size: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        var _props = this.props,
            value = _props.value,
            onChange = _props.onChange;


        if (typeof value == 'string' && !onChange) {
            throw new Error('onChange should be passed to Autocomplete when value is passed');
        }

        return {
            value: value || '',
            cursor: null,
            loading: false,
            focused: false,
            results: []
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
        var value = nextProps.value;

        this.updateValue(value);
    },


    /**
     * Update the value.
     */
    updateValue: function updateValue(value) {
        var _this = this;

        var prevValue = this.state.value;
        var onFetch = this.props.onFetch;


        if (prevValue == value) {
            return;
        }

        this.setState({
            value: value,
            loading: true
        });

        onFetch(value, function (results) {
            _this.setState({
                loading: false,
                results: results
            });
        });
    },


    /**
     * Typed value changed, we fetch the new autocomplete result
     */
    onInputChanged: function onInputChanged(e) {
        var onChange = this.props.onChange;
        var value = e.target.value;


        this.updateValue(value);

        if (onChange) {
            onChange(value);
        }
    },


    /**
     * User is focusing/blur the input
     */
    onFocusChanged: function onFocusChanged(isFocused) {
        var _props2 = this.props,
            onFocus = _props2.onFocus,
            onBlur = _props2.onBlur;


        if (isFocused && onFocus) {
            onFocus();
        } else if (onBlur) {
            onBlur();
        }

        this.setState({
            focused: isFocused
        });
    },


    /**
     * Submit value at cursor
     */
    onEnter: function onEnter() {
        var _state = this.state,
            cursor = _state.cursor,
            value = _state.value;
        var onEnter = this.props.onEnter;


        if (cursor >= 0) {
            this.onSelect(cursor);
        } else if (onEnter) {
            onEnter(value);
            this.setState({
                focused: false,
                cursor: null,
                results: [],
                value: ''
            });
        }
    },


    /**
     * Submit a value
     */
    onSelect: function onSelect(index) {
        var onSelect = this.props.onSelect;
        var _state2 = this.state,
            value = _state2.value,
            results = _state2.results;

        var selected = results[index];

        this.setState({
            cursor: null,
            results: [],
            value: ''
        });

        onSelect(value, selected);
    },


    /**
     * User pressed a key in text input
     */
    onKeyDown: function onKeyDown(e) {
        var _state3 = this.state,
            cursor = _state3.cursor,
            results = _state3.results;


        if (e.keyCode === KEYCODE_ENTER) {
            e.preventDefault();
            this.onEnter();
        } else if (e.keyCode === KEYCODE_DOWN) {
            e.preventDefault();
            cursor++;
        } else if (e.keyCode === KEYCODE_UP) {
            e.preventDefault();
            cursor--;
        }

        if (cursor >= results.length) {
            cursor = results.length - 1;
        }
        if (cursor < -1) {
            cursor = -1;
        }

        this.setState({
            cursor: cursor
        });
    },


    /**
     * Render the suggestions
     */
    renderResults: function renderResults() {
        var _this2 = this;

        var _state4 = this.state,
            results = _state4.results,
            value = _state4.value,
            cursor = _state4.cursor;

        var ResultComponent = this.props.renderResult;

        return React.createElement(
            'div',
            { className: 'AutocompleteResults' },
            results.map(function (result, i) {
                var isActive = i === cursor;

                return React.createElement(
                    AutocompleteResult,
                    { key: value + '-' + i, active: isActive,
                        onClick: function onClick(e) {
                            return _this2.onSelect(i);
                        } },
                    React.createElement(ResultComponent, { result: result, index: i, active: isActive })
                );
            })
        );
    },


    /**
     * Focus or blur the autocomplete
     */
    focus: function focus() {
        var input = this.refs.input;

        input.focus();
    },
    blur: function blur() {
        var input = this.refs.input;

        input.blur();
    },
    render: function render() {
        var _this3 = this;

        var _props3 = this.props,
            onPaste = _props3.onPaste,
            size = _props3.size,
            placeholder = _props3.placeholder;
        var _state5 = this.state,
            value = _state5.value,
            focused = _state5.focused,
            loading = _state5.loading,
            results = _state5.results;


        return React.createElement(
            'div',
            { className: 'Autocomplete' },
            React.createElement(Input, {
                ref: 'input',
                value: value,
                placeholder: placeholder,
                size: size,
                onChange: this.onInputChanged,
                onFocus: function onFocus(e) {
                    return _this3.onFocusChanged(true);
                },
                onBlur: function onBlur(e) {
                    return _this3.onFocusChanged(false);
                },
                onPaste: onPaste,
                onKeyDown: this.onKeyDown
            }),
            loading ? React.createElement(Spinner, { size: 'sm', centered: false }) : '',
            focused && results.length > 0 ? this.renderResults() : ''
        );
    }
});

/**
 * Container for the results.
 * @type {ReactClass}
 */
var AutocompleteResult = React.createClass({
    displayName: 'AutocompleteResult',

    propTypes: {
        active: React.PropTypes.bool,
        onClick: React.PropTypes.func,
        children: React.PropTypes.node
    },

    render: function render() {
        var _props4 = this.props,
            active = _props4.active,
            children = _props4.children,
            onClick = _props4.onClick;

        return React.createElement(
            'div',
            { className: classNames('AutocompleteResult', { active: active }),
                onMouseDown: onClick },
            children
        );
    }
});

module.exports = Autocomplete;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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