'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Default render for options
 * @type {ReactClass}
 */
var DefaultComponent = React.createClass({
    displayName: 'DefaultComponent',

    propTypes: {
        option: itemShape
    },

    render: function render() {
        var option = this.props.option;

        return React.createElement(
            'span',
            null,
            option
        );
    }
});

/**
 * Component to render a Selection option
 * @type {ReactClass}
 */
var SelectOption = React.createClass({
    displayName: 'SelectOption',

    propTypes: {
        item: itemShape.isRequired,
        index: React.PropTypes.number.isRequired,
        // Function to render the option to a string or element
        component: React.PropTypes.func.isRequired,
        // Function to check if option is in selected values
        hasValue: React.PropTypes.func.isRequired,
        // Toggle an option in main Select state
        onToggleOption: React.PropTypes.func.isRequired,
        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            isOptionDisabled: function isOptionDisabled() {
                return false;
            }
        };
    },
    render: function render() {
        var _props = this.props,
            item = _props.item,
            index = _props.index,
            isOptionDisabled = _props.isOptionDisabled,
            hasValue = _props.hasValue,
            onToggleOption = _props.onToggleOption,
            Component = _props.component;

        // Check if item should be displayed but marked as disabled

        var isDisabled = isOptionDisabled(item);

        var className = classNames('SelectOption', {
            active: hasValue(item),
            disabled: isDisabled
        });

        return React.createElement(
            'div',
            { className: className,
                onClick: function onClick(e) {
                    if (!isDisabled) {
                        onToggleOption(item);
                    }
                } },
            React.createElement(Component, { option: item, index: index })
        );
    }
});

/**
 * Component to render a Select option group
 * @type {ReactClass}
 */
var SelectOptGroup = React.createClass({
    displayName: 'SelectOptGroup',

    propTypes: {
        group: groupShape
    },

    render: function render() {
        var _props2 = this.props,
            group = _props2.group,
            props = _objectWithoutProperties(_props2, ['group']);

        return React.createElement(
            'div',
            { className: 'SelectOptGroup' },
            group.label ? React.createElement(
                'div',
                { className: 'GroupLabel' },
                group.label
            ) : null,
            React.createElement(
                'div',
                { className: 'GroupOptions' },
                group.options.map(function (item, i) {
                    return React.createElement(SelectOption, _extends({}, props, { key: i, item: item, index: i }));
                })
            )
        );
    }
});

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render a message when search did not return any results
        searchEmptyComponent: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool,

        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: DefaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select',
            searchEmptyComponent: null
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },


    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var options = props.options,
            groups = props.groups;


        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },


    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },


    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },


    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },


    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },


    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },


    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },


    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;


        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },


    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },


    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },


    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props3 = this.props,
            disabled = _props3.disabled,
            block = _props3.block,
            multiple = _props3.multiple,
            placeholder = _props3.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement(
                'span',
                { className: 'SelectSelections' },
                values.map(function (val, i) {
                    return React.createElement(
                        'span',
                        { key: i, className: 'SelectSelection' },
                        React.createElement(ComponentSelection, { option: val, index: i })
                    );
                })
            );
        } else {
            inner = React.createElement(
                'span',
                { className: 'SelectPlaceholder' },
                placeholder
            );
        }

        return React.createElement(
            Button,
            { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle },
            inner,
            ' ',
            React.createElement(Button.Caret, null)
        );
    },


    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _this = this;

        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups,
            query = _state3.query;

        var _props4 = this.props,
            search = _props4.search,
            searchPlaceholder = _props4.searchPlaceholder,
            filter = _props4.filter,
            SearchEmptyComponent = _props4.searchEmptyComponent,
            props = _objectWithoutProperties(_props4, ['search', 'searchPlaceholder', 'filter', 'searchEmptyComponent']);

        var className = classNames('SelectContainer', {
            'open': opened
        });

        // Filter empty groups based on search query
        var filteredGroups = groups.map(function (group) {
            return _extends({}, group, {
                options: group.options.filter(function (item, i) {
                    return filter(query, item, i);
                })
            });
        }).filter(function (group) {
            return group.options.length > 0;
        });

        return React.createElement(
            'div',
            { className: className },
            search ? React.createElement(
                'div',
                { className: 'SelectSearch' },
                React.createElement(Input, { ref: 'searchInput',
                    value: query,
                    onChange: this.onSearchChanged,
                    placeholder: searchPlaceholder
                })
            ) : null,
            Boolean(filteredGroups.length) ? React.createElement(
                'div',
                { className: 'SelectGroups' },
                filteredGroups.map(function (group, i) {
                    return React.createElement(SelectOptGroup, _extends({}, props, { key: i, group: group, hasValue: _this.hasValue, onToggleOption: _this.onToggleOption }));
                })
            ) : null,
            search && !filteredGroups.length ? React.createElement(
                'div',
                { className: 'SearchEmpty' },
                React.createElement(SearchEmptyComponent, { query: query })
            ) : null
        );
    },
    render: function render() {
        var _props5 = this.props,
            name = _props5.name,
            block = _props5.block;
        var opened = this.state.opened;


        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement(
            'div',
            { className: className, onClick: function onClick(e) {
                    return e.stopPropagation();
                } },
            React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }),
            this.renderButton(),
            opened ? React.createElement(
                Backdrop,
                { onClose: this.close },
                this.renderGroups()
            ) : null
        );
    }
});

module.exports = Select;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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