'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var SIZES = require('./SIZES');
var Button = require('./Button');
var Input = require('./Input');
var Backdrop = require('./Backdrop');

var DEFAULT_SEARCH_PLACEHOLDER = 'Search';

var itemShape = React.PropTypes.oneOfType([React.PropTypes.string, React.PropTypes.object]);

var groupShape = React.PropTypes.shape({
    label: React.PropTypes.string,
    options: React.PropTypes.arrayOf(itemShape)
});

/**
 * Default filter for select
 */
function defaultFilter(query, item, i) {
    return true;
}

/**
 * Default render to string for input
 */
function defaultRenderToString(item, i) {
    return String(item);
}

/**
 * Default render for options
 * @type {ReactClass}
 */
var DefaultComponent = React.createClass({
    displayName: 'DefaultComponent',

    propTypes: {
        option: itemShape
    },

    render: function render() {
        var option = this.props.option;

        return React.createElement(
            'span',
            null,
            option
        );
    }
});

/**
 * Component to render a Selection option
 * @type {ReactClass}
 */
var SelectOption = React.createClass({
    displayName: 'SelectOption',

    propTypes: {
        item: itemShape.isRequired,
        index: React.PropTypes.number.isRequired,
        // Function to render the option to a string or element
        component: React.PropTypes.func.isRequired,
        // Function to check if option is in selected values
        hasValue: React.PropTypes.func.isRequired,
        // Toggle an option in main Select state
        onToggleOption: React.PropTypes.func.isRequired,
        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            isOptionDisabled: function isOptionDisabled() {
                return false;
            }
        };
    },
    render: function render() {
        var _props = this.props,
            item = _props.item,
            index = _props.index,
            isOptionDisabled = _props.isOptionDisabled,
            hasValue = _props.hasValue,
            onToggleOption = _props.onToggleOption,
            Component = _props.component;

        // Check if item should be displayed but marked as disabled

        var isDisabled = isOptionDisabled(item);

        var className = classNames('SelectOption', {
            active: hasValue(item),
            disabled: isDisabled
        });

        return React.createElement(
            'div',
            { className: className,
                onClick: function onClick(e) {
                    if (!isDisabled) {
                        onToggleOption(item);
                    }
                } },
            React.createElement(Component, { option: item, index: index })
        );
    }
});

/**
 * Component to render a Select option group
 * @type {ReactClass}
 */
var SelectOptGroup = React.createClass({
    displayName: 'SelectOptGroup',

    propTypes: {
        group: groupShape
    },

    render: function render() {
        var _props2 = this.props,
            group = _props2.group,
            props = _objectWithoutProperties(_props2, ['group']);

        return React.createElement(
            'div',
            { className: 'SelectOptGroup' },
            group.label ? React.createElement(
                'div',
                { className: 'GroupLabel' },
                group.label
            ) : null,
            React.createElement(
                'div',
                { className: 'GroupOptions' },
                group.options.map(function (item, i) {
                    return React.createElement(SelectOption, _extends({}, props, { key: i, item: item, index: i }));
                })
            )
        );
    }
});

/**
 * Interractive select for forms
 *
 * It renders as a normal select on server and has a custom UI on browser (with search, images support).
 *
 * <Select name="test" options=[] />
 */
var Select = React.createClass({
    displayName: 'Select',

    propTypes: {
        // Current value of the select
        value: React.PropTypes.oneOfType([itemShape, React.PropTypes.arrayOf(itemShape)]),

        // List of items to display
        groups: React.PropTypes.arrayOf(groupShape),
        options: React.PropTypes.arrayOf(itemShape),

        // Function to render the option to a string or element
        component: React.PropTypes.func,

        // Function to render a message when search did not return any results
        searchEmptyComponent: React.PropTypes.func,

        // Function to render the selected option in the button
        // Defaults to "renderOption"
        componentSelection: React.PropTypes.func,

        // Function to output an option as a string
        // Defaults to a string representation, you have to provide your own value
        // when using a custom option renderer
        renderToString: React.PropTypes.func,

        // Function to filter an element
        filter: React.PropTypes.func,

        // Optional callback when value changed
        onChange: React.PropTypes.func,

        // Name when using server posting
        name: React.PropTypes.string,

        // Text to display when no value is set
        placeholder: React.PropTypes.string,
        searchPlaceholder: React.PropTypes.string,

        // Delimiter for multiple values
        delimiter: React.PropTypes.string,

        // Prevent selection
        disabled: React.PropTypes.bool,

        // Display the search filter?
        search: React.PropTypes.bool,

        // Accept multiple values
        multiple: React.PropTypes.bool,

        // Size of the select to display
        size: React.PropTypes.oneOf(SIZES),

        // Take the whole width
        block: React.PropTypes.bool,

        // Should an option be marked as disabled
        isOptionDisabled: React.PropTypes.func
    },

    getDefaultProps: function getDefaultProps() {
        return {
            disabled: false,
            search: true,
            delimiter: ',',
            size: SIZES[0],
            multiple: false,
            block: false,
            filter: defaultFilter,
            component: DefaultComponent,
            renderToString: defaultRenderToString,
            searchPlaceholder: DEFAULT_SEARCH_PLACEHOLDER,
            placeholder: 'Select',
            searchEmptyComponent: null
        };
    },
    getInitialState: function getInitialState() {
        return {
            value: this.props.value,
            query: '',
            opened: false,
            groups: this.propsToGroups(this.props)
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(newProps) {
        this.setState({
            value: newProps.value,
            groups: this.propsToGroups(newProps),
            opened: newProps.disabled ? false : this.state.opened
        });
    },


    /**
     * Create list of groups from props
     * @param {Object} props
     * @return {Array<groupShape>}
     */
    propsToGroups: function propsToGroups(props) {
        var options = props.options,
            groups = props.groups;


        if (groups) {
            return groups;
        }

        return [{ options: options }];
    },


    /**
     * Search query changed
     */
    onSearchChanged: function onSearchChanged(e) {
        this.setState({
            query: e.target.value
        });
    },


    /**
     * Toggle (close/open) the select
     */
    onToggle: function onToggle() {
        this.setState({
            opened: !this.state.opened
        });
    },


    /**
     * Close the select
     */
    close: function close() {
        this.setState({
            opened: false
        });
    },


    /**
     * Open the select
     */
    open: function open() {
        this.setState({
            opened: false
        });
    },


    /**
     * Focus the search if open
     */
    focusOnOpen: function focusOnOpen() {
        if (this.state.opened) {
            this.focusSearch();
        }
    },
    componentDidUpdate: function componentDidUpdate() {
        this.focusOnOpen();
    },
    componentDidMount: function componentDidMount() {
        this.focusOnOpen();
    },


    /**
     * Toggle an option
     */
    onToggleOption: function onToggleOption(addValue, e) {
        if (e) {
            e.preventDefault();
        }

        var _state = this.state,
            value = _state.value,
            multiple = _state.multiple;
        var onChange = this.props.onChange;

        var newState = void 0,
            newValue = void 0;

        if (multiple) {
            newValue = value;

            // Add to selection if not yet selected
            if (!this.hasValue(addValue)) {
                newValue = value.concat([addValue]);
            } else if (value.length > 1) {
                // Unselect if many options are selected
                newValue.splice(newValue.indexOf(addValue), 1);
            }

            newState = {
                value: newValue
            };
        } else {
            newValue = addValue;

            newState = {
                value: addValue,
                opened: false
            };
        }

        this.setState(newState, function () {
            if (onChange) {
                onChange(newValue);
            }
        });
    },


    /**
     * Get current value as a string (for hidden input)
     * @return {String}
     */
    getStringValue: function getStringValue() {
        var renderToString = this.props.renderToString;
        var value = this.state.value;


        if (!value) {
            return '';
        }

        if (!this.props.multiple) {
            return renderToString(value);
        } else {
            return value.map(renderToString).join(this.props.delimiter);
        }
    },


    /**
     * Check if a value is selected
     * @param {String} value
     * @return {Boolean}
     */
    hasValue: function hasValue(value) {
        var currentValue = this.state.value;

        if (!this.props.multiple) {
            return currentValue === value;
        } else {
            return currentValue.indexOf(value) >= 0;
        }
    },


    /**
     * Focus the search input
     */
    focusSearch: function focusSearch() {
        var searchInput = this.refs.searchInput;

        if (!searchInput) {
            return;
        }

        searchInput.focus();
    },


    /**
     * Render button to open select
     */
    renderButton: function renderButton() {
        var _props3 = this.props,
            disabled = _props3.disabled,
            block = _props3.block,
            multiple = _props3.multiple,
            placeholder = _props3.placeholder;
        var _state2 = this.state,
            value = _state2.value,
            opened = _state2.opened;

        var ComponentSelection = this.props.componentSelection || this.props.component;

        var inner = void 0;

        if (value) {
            var values = multiple ? value : [value];
            inner = React.createElement(
                'span',
                { className: 'SelectSelections' },
                values.map(function (val, i) {
                    return React.createElement(
                        'span',
                        { key: i, className: 'SelectSelection' },
                        React.createElement(ComponentSelection, { option: val, index: i })
                    );
                })
            );
        } else {
            inner = React.createElement(
                'span',
                { className: 'SelectPlaceholder' },
                placeholder
            );
        }

        return React.createElement(
            Button,
            { size: this.props.size, block: block, disabled: disabled, active: opened, onClick: this.onToggle },
            inner,
            ' ',
            React.createElement(Button.Caret, null)
        );
    },


    /**
     * Render the groups
     */
    renderGroups: function renderGroups() {
        var _this = this;

        var _state3 = this.state,
            opened = _state3.opened,
            groups = _state3.groups,
            query = _state3.query;

        var _props4 = this.props,
            search = _props4.search,
            searchPlaceholder = _props4.searchPlaceholder,
            filter = _props4.filter,
            SearchEmptyComponent = _props4.searchEmptyComponent,
            props = _objectWithoutProperties(_props4, ['search', 'searchPlaceholder', 'filter', 'searchEmptyComponent']);

        var className = classNames('SelectContainer', {
            'open': opened
        });

        // Filter empty groups based on search query
        var filteredGroups = groups.map(function (group) {
            return _extends({}, group, {
                options: group.options.filter(function (item, i) {
                    return filter(query, item, i);
                })
            });
        }).filter(function (group) {
            return group.options.length > 0;
        });

        return React.createElement(
            'div',
            { className: className },
            search ? React.createElement(
                'div',
                { className: 'SelectSearch' },
                React.createElement(Input, { ref: 'searchInput',
                    value: query,
                    onChange: this.onSearchChanged,
                    placeholder: searchPlaceholder
                })
            ) : null,
            Boolean(filteredGroups.length) ? React.createElement(
                'div',
                { className: 'SelectGroups' },
                filteredGroups.map(function (group, i) {
                    return React.createElement(SelectOptGroup, _extends({}, props, { key: i, group: group, hasValue: _this.hasValue, onToggleOption: _this.onToggleOption }));
                })
            ) : null,
            search && !filteredGroups.length ? React.createElement(
                'div',
                { className: 'SearchEmpty' },
                React.createElement(SearchEmptyComponent, { query: query })
            ) : null
        );
    },
    render: function render() {
        var _props5 = this.props,
            name = _props5.name,
            block = _props5.block;
        var opened = this.state.opened;


        var className = classNames('SelectFormControl', {
            block: block
        });

        return React.createElement(
            'div',
            { className: className, onClick: function onClick(e) {
                    return e.stopPropagation();
                } },
            React.createElement('input', { type: 'hidden', name: name, value: this.getStringValue() }),
            this.renderButton(),
            opened ? React.createElement(
                Backdrop,
                { onClose: this.close },
                this.renderGroups()
            ) : null
        );
    }
});

module.exports = Select;
//# sourceMappingURL=data:application/json;base64,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