'use strict';

var React = require('react');
var classNames = require('classnames');

var Input = require('./Input');
var Spinner = require('./Spinner');

var KEYCODE_ENTER = 13;
var KEYCODE_UP = 38;
var KEYCODE_DOWN = 40;

var Autocomplete = React.createClass({
    displayName: 'Autocomplete',

    propTypes: {
        onFetch: React.PropTypes.func.isRequired,
        renderResult: React.PropTypes.func.isRequired,
        // Called when onEnter on the input (no result selected)
        onEnter: React.PropTypes.func.isRequired,
        onPaste: React.PropTypes.func,
        // Focus events
        onFocus: React.PropTypes.func,
        onBlur: React.PropTypes.func,
        // Control the event
        value: React.PropTypes.string,
        // Called when typing
        onChange: React.PropTypes.func.isRequired,
        // Called when selecting an entry
        onSelect: React.PropTypes.func.isRequired,
        // Render options
        placeholder: React.PropTypes.string,
        size: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        var _props = this.props,
            value = _props.value,
            onChange = _props.onChange;


        if (typeof value == 'string' && !onChange) {
            throw new Error('onChange should be passed to Autocomplete when value is passed');
        }

        return {
            value: value || '',
            cursor: null,
            loading: false,
            focused: false,
            results: []
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
        var value = nextProps.value || '';
        this.updateValue(value);
    },


    /**
     * Update the value.
     */
    updateValue: function updateValue(value) {
        var _this = this;

        var prevValue = this.state.value;
        var onFetch = this.props.onFetch;


        if (prevValue == value) {
            return;
        }

        this.setState({
            value: value,
            loading: true
        });

        onFetch(value, function (results) {
            _this.setState({
                loading: false,
                results: results
            });
        });
    },


    /**
     * Typed value changed, we fetch the new autocomplete result
     */
    onInputChanged: function onInputChanged(e) {
        var onChange = this.props.onChange;
        var value = e.target.value;


        this.updateValue(value);

        if (onChange) {
            onChange(value);
        }
    },


    /**
     * User is focusing/blur the input
     */
    onFocusChanged: function onFocusChanged(isFocused) {
        var _props2 = this.props,
            onFocus = _props2.onFocus,
            onBlur = _props2.onBlur;


        if (isFocused && onFocus) {
            onFocus();
        } else if (onBlur) {
            onBlur();
        }

        this.setState({
            focused: isFocused
        });
    },


    /**
     * Submit value at cursor
     */
    onEnter: function onEnter() {
        var _state = this.state,
            cursor = _state.cursor,
            value = _state.value;
        var onEnter = this.props.onEnter;


        if (cursor >= 0) {
            this.onSelect(cursor);
        } else if (onEnter) {
            onEnter(value);
            this.setState({
                focused: false,
                cursor: null,
                results: [],
                value: ''
            });
        }
    },


    /**
     * Submit a value
     */
    onSelect: function onSelect(index) {
        var onSelect = this.props.onSelect;
        var _state2 = this.state,
            value = _state2.value,
            results = _state2.results;

        var selected = results[index];

        this.setState({
            cursor: null,
            results: [],
            value: ''
        });

        onSelect(value, selected);
    },


    /**
     * User pressed a key in text input
     */
    onKeyDown: function onKeyDown(e) {
        var _state3 = this.state,
            cursor = _state3.cursor,
            results = _state3.results;


        if (e.keyCode === KEYCODE_ENTER) {
            e.preventDefault();
            this.onEnter();
        } else if (e.keyCode === KEYCODE_DOWN) {
            e.preventDefault();
            cursor++;
        } else if (e.keyCode === KEYCODE_UP) {
            e.preventDefault();
            cursor--;
        }

        if (cursor >= results.length) {
            cursor = results.length - 1;
        }
        if (cursor < -1) {
            cursor = -1;
        }

        this.setState({
            cursor: cursor
        });
    },


    /**
     * Render the suggestions
     */
    renderResults: function renderResults() {
        var _this2 = this;

        var _state4 = this.state,
            results = _state4.results,
            value = _state4.value,
            cursor = _state4.cursor;

        var ResultComponent = this.props.renderResult;

        return React.createElement(
            'div',
            { className: 'AutocompleteResults' },
            results.map(function (result, i) {
                var isActive = i === cursor;

                return React.createElement(
                    AutocompleteResult,
                    { key: value + '-' + i, active: isActive,
                        onClick: function onClick(e) {
                            return _this2.onSelect(i);
                        } },
                    React.createElement(ResultComponent, { result: result, index: i, active: isActive })
                );
            })
        );
    },


    /**
     * Focus or blur the autocomplete
     */
    focus: function focus() {
        var input = this.refs.input;

        input.focus();
    },
    blur: function blur() {
        var input = this.refs.input;

        input.blur();
    },
    render: function render() {
        var _this3 = this;

        var _props3 = this.props,
            onPaste = _props3.onPaste,
            size = _props3.size,
            placeholder = _props3.placeholder;
        var _state5 = this.state,
            value = _state5.value,
            focused = _state5.focused,
            loading = _state5.loading,
            results = _state5.results;


        return React.createElement(
            'div',
            { className: 'Autocomplete' },
            React.createElement(Input, {
                ref: 'input',
                value: value,
                placeholder: placeholder,
                size: size,
                onChange: this.onInputChanged,
                onFocus: function onFocus(e) {
                    return _this3.onFocusChanged(true);
                },
                onBlur: function onBlur(e) {
                    return _this3.onFocusChanged(false);
                },
                onPaste: onPaste,
                onKeyDown: this.onKeyDown
            }),
            loading ? React.createElement(Spinner, { size: 'sm', centered: false }) : '',
            focused && results.length > 0 ? this.renderResults() : ''
        );
    }
});

/**
 * Container for the results.
 * @type {ReactClass}
 */
var AutocompleteResult = React.createClass({
    displayName: 'AutocompleteResult',

    propTypes: {
        active: React.PropTypes.bool,
        onClick: React.PropTypes.func,
        children: React.PropTypes.node
    },

    render: function render() {
        var _props4 = this.props,
            active = _props4.active,
            children = _props4.children,
            onClick = _props4.onClick;

        return React.createElement(
            'div',
            { className: classNames('AutocompleteResult', { active: active }),
                onMouseDown: onClick },
            children
        );
    }
});

module.exports = Autocomplete;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9BdXRvY29tcGxldGUuanMiXSwibmFtZXMiOlsiUmVhY3QiLCJyZXF1aXJlIiwiY2xhc3NOYW1lcyIsIklucHV0IiwiU3Bpbm5lciIsIktFWUNPREVfRU5URVIiLCJLRVlDT0RFX1VQIiwiS0VZQ09ERV9ET1dOIiwiQXV0b2NvbXBsZXRlIiwiY3JlYXRlQ2xhc3MiLCJwcm9wVHlwZXMiLCJvbkZldGNoIiwiUHJvcFR5cGVzIiwiZnVuYyIsImlzUmVxdWlyZWQiLCJyZW5kZXJSZXN1bHQiLCJvbkVudGVyIiwib25QYXN0ZSIsIm9uRm9jdXMiLCJvbkJsdXIiLCJ2YWx1ZSIsInN0cmluZyIsIm9uQ2hhbmdlIiwib25TZWxlY3QiLCJwbGFjZWhvbGRlciIsInNpemUiLCJnZXRJbml0aWFsU3RhdGUiLCJwcm9wcyIsIkVycm9yIiwiY3Vyc29yIiwibG9hZGluZyIsImZvY3VzZWQiLCJyZXN1bHRzIiwiY29tcG9uZW50V2lsbFJlY2VpdmVQcm9wcyIsIm5leHRQcm9wcyIsInVwZGF0ZVZhbHVlIiwicHJldlZhbHVlIiwic3RhdGUiLCJzZXRTdGF0ZSIsIm9uSW5wdXRDaGFuZ2VkIiwiZSIsInRhcmdldCIsIm9uRm9jdXNDaGFuZ2VkIiwiaXNGb2N1c2VkIiwiaW5kZXgiLCJzZWxlY3RlZCIsIm9uS2V5RG93biIsImtleUNvZGUiLCJwcmV2ZW50RGVmYXVsdCIsImxlbmd0aCIsInJlbmRlclJlc3VsdHMiLCJSZXN1bHRDb21wb25lbnQiLCJtYXAiLCJyZXN1bHQiLCJpIiwiaXNBY3RpdmUiLCJmb2N1cyIsImlucHV0IiwicmVmcyIsImJsdXIiLCJyZW5kZXIiLCJBdXRvY29tcGxldGVSZXN1bHQiLCJhY3RpdmUiLCJib29sIiwib25DbGljayIsImNoaWxkcmVuIiwibm9kZSIsIm1vZHVsZSIsImV4cG9ydHMiXSwibWFwcGluZ3MiOiI7O0FBQUEsSUFBTUEsUUFBUUMsUUFBUSxPQUFSLENBQWQ7QUFDQSxJQUFNQyxhQUFhRCxRQUFRLFlBQVIsQ0FBbkI7O0FBRUEsSUFBTUUsUUFBUUYsUUFBUSxTQUFSLENBQWQ7QUFDQSxJQUFNRyxVQUFVSCxRQUFRLFdBQVIsQ0FBaEI7O0FBRUEsSUFBTUksZ0JBQWdCLEVBQXRCO0FBQ0EsSUFBTUMsYUFBZ0IsRUFBdEI7QUFDQSxJQUFNQyxlQUFnQixFQUF0Qjs7QUFFQSxJQUFNQyxlQUFlUixNQUFNUyxXQUFOLENBQWtCO0FBQUE7O0FBQ25DQyxlQUFXO0FBQ1BDLGlCQUFjWCxNQUFNWSxTQUFOLENBQWdCQyxJQUFoQixDQUFxQkMsVUFENUI7QUFFUEMsc0JBQWNmLE1BQU1ZLFNBQU4sQ0FBZ0JDLElBQWhCLENBQXFCQyxVQUY1QjtBQUdQO0FBQ0FFLGlCQUFjaEIsTUFBTVksU0FBTixDQUFnQkMsSUFBaEIsQ0FBcUJDLFVBSjVCO0FBS1BHLGlCQUFjakIsTUFBTVksU0FBTixDQUFnQkMsSUFMdkI7QUFNUDtBQUNBSyxpQkFBY2xCLE1BQU1ZLFNBQU4sQ0FBZ0JDLElBUHZCO0FBUVBNLGdCQUFjbkIsTUFBTVksU0FBTixDQUFnQkMsSUFSdkI7QUFTUDtBQUNBTyxlQUFjcEIsTUFBTVksU0FBTixDQUFnQlMsTUFWdkI7QUFXUDtBQUNBQyxrQkFBY3RCLE1BQU1ZLFNBQU4sQ0FBZ0JDLElBQWhCLENBQXFCQyxVQVo1QjtBQWFQO0FBQ0FTLGtCQUFjdkIsTUFBTVksU0FBTixDQUFnQkMsSUFBaEIsQ0FBcUJDLFVBZDVCO0FBZVA7QUFDQVUscUJBQWN4QixNQUFNWSxTQUFOLENBQWdCUyxNQWhCdkI7QUFpQlBJLGNBQWN6QixNQUFNWSxTQUFOLENBQWdCUztBQWpCdkIsS0FEd0I7O0FBcUJuQ0ssbUJBckJtQyw2QkFxQmpCO0FBQUEscUJBQ2MsS0FBS0MsS0FEbkI7QUFBQSxZQUNOUCxLQURNLFVBQ05BLEtBRE07QUFBQSxZQUNDRSxRQURELFVBQ0NBLFFBREQ7OztBQUdkLFlBQUssT0FBT0YsS0FBUCxJQUFnQixRQUFqQixJQUE4QixDQUFDRSxRQUFuQyxFQUE2QztBQUN6QyxrQkFBTSxJQUFJTSxLQUFKLENBQVUsZ0VBQVYsQ0FBTjtBQUNIOztBQUVELGVBQU87QUFDSFIsbUJBQVNBLFNBQVMsRUFEZjtBQUVIUyxvQkFBUyxJQUZOO0FBR0hDLHFCQUFTLEtBSE47QUFJSEMscUJBQVMsS0FKTjtBQUtIQyxxQkFBUztBQUxOLFNBQVA7QUFPSCxLQW5Da0M7QUFxQ25DQyw2QkFyQ21DLHFDQXFDVEMsU0FyQ1MsRUFxQ0U7QUFDakMsWUFBTWQsUUFBUWMsVUFBVWQsS0FBVixJQUFtQixFQUFqQztBQUNBLGFBQUtlLFdBQUwsQ0FBaUJmLEtBQWpCO0FBQ0gsS0F4Q2tDOzs7QUEwQ25DOzs7QUFHQWUsZUE3Q21DLHVCQTZDdkJmLEtBN0N1QixFQTZDaEI7QUFBQTs7QUFBQSxZQUNBZ0IsU0FEQSxHQUNjLEtBQUtDLEtBRG5CLENBQ1BqQixLQURPO0FBQUEsWUFFUFQsT0FGTyxHQUVLLEtBQUtnQixLQUZWLENBRVBoQixPQUZPOzs7QUFJZixZQUFJeUIsYUFBYWhCLEtBQWpCLEVBQXdCO0FBQ3BCO0FBQ0g7O0FBRUQsYUFBS2tCLFFBQUwsQ0FBYztBQUNWbEIsd0JBRFU7QUFFVlUscUJBQVM7QUFGQyxTQUFkOztBQUtBbkIsZ0JBQVFTLEtBQVIsRUFBZSxVQUFDWSxPQUFELEVBQWE7QUFDeEIsa0JBQUtNLFFBQUwsQ0FBYztBQUNWUix5QkFBUyxLQURDO0FBRVZFO0FBRlUsYUFBZDtBQUlILFNBTEQ7QUFNSCxLQWhFa0M7OztBQWtFbkM7OztBQUdBTyxrQkFyRW1DLDBCQXFFcEJDLENBckVvQixFQXFFakI7QUFBQSxZQUNObEIsUUFETSxHQUNPLEtBQUtLLEtBRFosQ0FDTkwsUUFETTtBQUFBLFlBRU5GLEtBRk0sR0FFSW9CLEVBQUVDLE1BRk4sQ0FFTnJCLEtBRk07OztBQUlkLGFBQUtlLFdBQUwsQ0FBaUJmLEtBQWpCOztBQUVBLFlBQUlFLFFBQUosRUFBYztBQUNWQSxxQkFBU0YsS0FBVDtBQUNIO0FBQ0osS0E5RWtDOzs7QUFnRm5DOzs7QUFHQXNCLGtCQW5GbUMsMEJBbUZwQkMsU0FuRm9CLEVBbUZUO0FBQUEsc0JBQ00sS0FBS2hCLEtBRFg7QUFBQSxZQUNkVCxPQURjLFdBQ2RBLE9BRGM7QUFBQSxZQUNMQyxNQURLLFdBQ0xBLE1BREs7OztBQUd0QixZQUFJd0IsYUFBYXpCLE9BQWpCLEVBQTBCO0FBQ3RCQTtBQUNILFNBRkQsTUFFTyxJQUFJQyxNQUFKLEVBQVk7QUFDZkE7QUFDSDs7QUFFRCxhQUFLbUIsUUFBTCxDQUFjO0FBQ1ZQLHFCQUFTWTtBQURDLFNBQWQ7QUFHSCxLQS9Ga0M7OztBQWlHbkM7OztBQUdBM0IsV0FwR21DLHFCQW9HekI7QUFBQSxxQkFDb0IsS0FBS3FCLEtBRHpCO0FBQUEsWUFDRVIsTUFERixVQUNFQSxNQURGO0FBQUEsWUFDVVQsS0FEVixVQUNVQSxLQURWO0FBQUEsWUFFRUosT0FGRixHQUVjLEtBQUtXLEtBRm5CLENBRUVYLE9BRkY7OztBQUlOLFlBQUlhLFVBQVUsQ0FBZCxFQUFpQjtBQUNiLGlCQUFLTixRQUFMLENBQWNNLE1BQWQ7QUFDSCxTQUZELE1BRU8sSUFBSWIsT0FBSixFQUFhO0FBQ2hCQSxvQkFBUUksS0FBUjtBQUNBLGlCQUFLa0IsUUFBTCxDQUFjO0FBQ1ZQLHlCQUFTLEtBREM7QUFFVkYsd0JBQVEsSUFGRTtBQUdWRyx5QkFBUyxFQUhDO0FBSVZaLHVCQUFPO0FBSkcsYUFBZDtBQU1IO0FBQ0osS0FuSGtDOzs7QUFxSG5DOzs7QUFHQUcsWUF4SG1DLG9CQXdIMUJxQixLQXhIMEIsRUF3SG5CO0FBQUEsWUFDSnJCLFFBREksR0FDUyxLQUFLSSxLQURkLENBQ0pKLFFBREk7QUFBQSxzQkFFZSxLQUFLYyxLQUZwQjtBQUFBLFlBRUpqQixLQUZJLFdBRUpBLEtBRkk7QUFBQSxZQUVHWSxPQUZILFdBRUdBLE9BRkg7O0FBR1osWUFBTWEsV0FBV2IsUUFBUVksS0FBUixDQUFqQjs7QUFFQSxhQUFLTixRQUFMLENBQWM7QUFDVlQsb0JBQVEsSUFERTtBQUVWRyxxQkFBUyxFQUZDO0FBR1ZaLG1CQUFPO0FBSEcsU0FBZDs7QUFNQUcsaUJBQVNILEtBQVQsRUFBZ0J5QixRQUFoQjtBQUNILEtBcElrQzs7O0FBc0luQzs7O0FBR0FDLGFBekltQyxxQkF5SXpCTixDQXpJeUIsRUF5SXRCO0FBQUEsc0JBQ2lCLEtBQUtILEtBRHRCO0FBQUEsWUFDSFIsTUFERyxXQUNIQSxNQURHO0FBQUEsWUFDS0csT0FETCxXQUNLQSxPQURMOzs7QUFHVCxZQUFJUSxFQUFFTyxPQUFGLEtBQWMxQyxhQUFsQixFQUFpQztBQUM3Qm1DLGNBQUVRLGNBQUY7QUFDQSxpQkFBS2hDLE9BQUw7QUFDSCxTQUhELE1BSUssSUFBSXdCLEVBQUVPLE9BQUYsS0FBY3hDLFlBQWxCLEVBQWdDO0FBQ2pDaUMsY0FBRVEsY0FBRjtBQUNBbkI7QUFDSCxTQUhJLE1BSUEsSUFBSVcsRUFBRU8sT0FBRixLQUFjekMsVUFBbEIsRUFBOEI7QUFDL0JrQyxjQUFFUSxjQUFGO0FBQ0FuQjtBQUNIOztBQUVELFlBQUlBLFVBQVVHLFFBQVFpQixNQUF0QixFQUE4QjtBQUMxQnBCLHFCQUFVRyxRQUFRaUIsTUFBUixHQUFpQixDQUEzQjtBQUNIO0FBQ0QsWUFBSXBCLFNBQVMsQ0FBQyxDQUFkLEVBQWlCO0FBQ2JBLHFCQUFTLENBQUMsQ0FBVjtBQUNIOztBQUVELGFBQUtTLFFBQUwsQ0FBYztBQUNWVDtBQURVLFNBQWQ7QUFHSCxLQW5La0M7OztBQXFLbkM7OztBQUdBcUIsaUJBeEttQywyQkF3S25CO0FBQUE7O0FBQUEsc0JBQ3VCLEtBQUtiLEtBRDVCO0FBQUEsWUFDSkwsT0FESSxXQUNKQSxPQURJO0FBQUEsWUFDS1osS0FETCxXQUNLQSxLQURMO0FBQUEsWUFDWVMsTUFEWixXQUNZQSxNQURaOztBQUVaLFlBQU1zQixrQkFBa0IsS0FBS3hCLEtBQUwsQ0FBV1osWUFBbkM7O0FBRUEsZUFDSTtBQUFBO0FBQUEsY0FBSyxXQUFVLHFCQUFmO0FBQ0tpQixvQkFBUW9CLEdBQVIsQ0FBWSxVQUFDQyxNQUFELEVBQVNDLENBQVQsRUFBZTtBQUN4QixvQkFBTUMsV0FBWUQsTUFBTXpCLE1BQXhCOztBQUVBLHVCQUFPO0FBQUMsc0NBQUQ7QUFBQSxzQkFBb0IsS0FBS1QsUUFBUSxHQUFSLEdBQWNrQyxDQUF2QyxFQUEwQyxRQUFRQyxRQUFsRDtBQUNxQixpQ0FBUztBQUFBLG1DQUFLLE9BQUtoQyxRQUFMLENBQWMrQixDQUFkLENBQUw7QUFBQSx5QkFEOUI7QUFFSCx3Q0FBQyxlQUFELElBQWlCLFFBQVFELE1BQXpCLEVBQWlDLE9BQU9DLENBQXhDLEVBQTJDLFFBQVFDLFFBQW5EO0FBRkcsaUJBQVA7QUFJSCxhQVBBO0FBREwsU0FESjtBQVlILEtBeExrQzs7O0FBMExuQzs7O0FBR0FDLFNBN0xtQyxtQkE2TDNCO0FBQUEsWUFDSUMsS0FESixHQUNjLEtBQUtDLElBRG5CLENBQ0lELEtBREo7O0FBRUpBLGNBQU1ELEtBQU47QUFDSCxLQWhNa0M7QUFpTW5DRyxRQWpNbUMsa0JBaU01QjtBQUFBLFlBQ0tGLEtBREwsR0FDZSxLQUFLQyxJQURwQixDQUNLRCxLQURMOztBQUVIQSxjQUFNRSxJQUFOO0FBQ0gsS0FwTWtDO0FBc01uQ0MsVUF0TW1DLG9CQXNNMUI7QUFBQTs7QUFBQSxzQkFDa0MsS0FBS2pDLEtBRHZDO0FBQUEsWUFDR1YsT0FESCxXQUNHQSxPQURIO0FBQUEsWUFDWVEsSUFEWixXQUNZQSxJQURaO0FBQUEsWUFDa0JELFdBRGxCLFdBQ2tCQSxXQURsQjtBQUFBLHNCQUV3QyxLQUFLYSxLQUY3QztBQUFBLFlBRUdqQixLQUZILFdBRUdBLEtBRkg7QUFBQSxZQUVVVyxPQUZWLFdBRVVBLE9BRlY7QUFBQSxZQUVtQkQsT0FGbkIsV0FFbUJBLE9BRm5CO0FBQUEsWUFFNEJFLE9BRjVCLFdBRTRCQSxPQUY1Qjs7O0FBSUwsZUFDSTtBQUFBO0FBQUEsY0FBSyxXQUFVLGNBQWY7QUFDSSxnQ0FBQyxLQUFEO0FBQ0kscUJBQUksT0FEUjtBQUVJLHVCQUFPWixLQUZYO0FBR0ksNkJBQWFJLFdBSGpCO0FBSUksc0JBQU1DLElBSlY7QUFLSSwwQkFBVSxLQUFLYyxjQUxuQjtBQU1JLHlCQUFTO0FBQUEsMkJBQUssT0FBS0csY0FBTCxDQUFvQixJQUFwQixDQUFMO0FBQUEsaUJBTmI7QUFPSSx3QkFBUTtBQUFBLDJCQUFLLE9BQUtBLGNBQUwsQ0FBb0IsS0FBcEIsQ0FBTDtBQUFBLGlCQVBaO0FBUUkseUJBQVN6QixPQVJiO0FBU0ksMkJBQVcsS0FBSzZCO0FBVHBCLGNBREo7QUFZS2hCLHNCQUFVLG9CQUFDLE9BQUQsSUFBUyxNQUFLLElBQWQsRUFBbUIsVUFBVSxLQUE3QixHQUFWLEdBQW1ELEVBWnhEO0FBYUtDLHVCQUFXQyxRQUFRaUIsTUFBUixHQUFpQixDQUE1QixHQUFnQyxLQUFLQyxhQUFMLEVBQWhDLEdBQXVEO0FBYjVELFNBREo7QUFpQkg7QUEzTmtDLENBQWxCLENBQXJCOztBQThOQTs7OztBQUlBLElBQU1XLHFCQUFxQjdELE1BQU1TLFdBQU4sQ0FBa0I7QUFBQTs7QUFDekNDLGVBQVc7QUFDUG9ELGdCQUFROUQsTUFBTVksU0FBTixDQUFnQm1ELElBRGpCO0FBRVBDLGlCQUFTaEUsTUFBTVksU0FBTixDQUFnQkMsSUFGbEI7QUFHUG9ELGtCQUFVakUsTUFBTVksU0FBTixDQUFnQnNEO0FBSG5CLEtBRDhCOztBQU96Q04sVUFQeUMsb0JBT2hDO0FBQUEsc0JBQ2lDLEtBQUtqQyxLQUR0QztBQUFBLFlBQ0dtQyxNQURILFdBQ0dBLE1BREg7QUFBQSxZQUNXRyxRQURYLFdBQ1dBLFFBRFg7QUFBQSxZQUNxQkQsT0FEckIsV0FDcUJBLE9BRHJCOztBQUVMLGVBQ0k7QUFBQTtBQUFBLGNBQUssV0FBVzlELFdBQVcsb0JBQVgsRUFBaUMsRUFBRTRELGNBQUYsRUFBakMsQ0FBaEI7QUFDSyw2QkFBYUUsT0FEbEI7QUFFS0M7QUFGTCxTQURKO0FBTUg7QUFmd0MsQ0FBbEIsQ0FBM0I7O0FBa0JBRSxPQUFPQyxPQUFQLEdBQWlCNUQsWUFBakIiLCJmaWxlIjoiQXV0b2NvbXBsZXRlLmpzIiwic291cmNlc0NvbnRlbnQiOlsiY29uc3QgUmVhY3QgPSByZXF1aXJlKCdyZWFjdCcpO1xuY29uc3QgY2xhc3NOYW1lcyA9IHJlcXVpcmUoJ2NsYXNzbmFtZXMnKTtcblxuY29uc3QgSW5wdXQgPSByZXF1aXJlKCcuL0lucHV0Jyk7XG5jb25zdCBTcGlubmVyID0gcmVxdWlyZSgnLi9TcGlubmVyJyk7XG5cbmNvbnN0IEtFWUNPREVfRU5URVIgPSAxMztcbmNvbnN0IEtFWUNPREVfVVAgICAgPSAzODtcbmNvbnN0IEtFWUNPREVfRE9XTiAgPSA0MDtcblxuY29uc3QgQXV0b2NvbXBsZXRlID0gUmVhY3QuY3JlYXRlQ2xhc3Moe1xuICAgIHByb3BUeXBlczoge1xuICAgICAgICBvbkZldGNoOiAgICAgIFJlYWN0LlByb3BUeXBlcy5mdW5jLmlzUmVxdWlyZWQsXG4gICAgICAgIHJlbmRlclJlc3VsdDogUmVhY3QuUHJvcFR5cGVzLmZ1bmMuaXNSZXF1aXJlZCxcbiAgICAgICAgLy8gQ2FsbGVkIHdoZW4gb25FbnRlciBvbiB0aGUgaW5wdXQgKG5vIHJlc3VsdCBzZWxlY3RlZClcbiAgICAgICAgb25FbnRlcjogICAgICBSZWFjdC5Qcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgICAgICBvblBhc3RlOiAgICAgIFJlYWN0LlByb3BUeXBlcy5mdW5jLFxuICAgICAgICAvLyBGb2N1cyBldmVudHNcbiAgICAgICAgb25Gb2N1czogICAgICBSZWFjdC5Qcm9wVHlwZXMuZnVuYyxcbiAgICAgICAgb25CbHVyOiAgICAgICBSZWFjdC5Qcm9wVHlwZXMuZnVuYyxcbiAgICAgICAgLy8gQ29udHJvbCB0aGUgZXZlbnRcbiAgICAgICAgdmFsdWU6ICAgICAgICBSZWFjdC5Qcm9wVHlwZXMuc3RyaW5nLFxuICAgICAgICAvLyBDYWxsZWQgd2hlbiB0eXBpbmdcbiAgICAgICAgb25DaGFuZ2U6ICAgICBSZWFjdC5Qcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgICAgICAvLyBDYWxsZWQgd2hlbiBzZWxlY3RpbmcgYW4gZW50cnlcbiAgICAgICAgb25TZWxlY3Q6ICAgICBSZWFjdC5Qcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuICAgICAgICAvLyBSZW5kZXIgb3B0aW9uc1xuICAgICAgICBwbGFjZWhvbGRlcjogIFJlYWN0LlByb3BUeXBlcy5zdHJpbmcsXG4gICAgICAgIHNpemU6ICAgICAgICAgUmVhY3QuUHJvcFR5cGVzLnN0cmluZ1xuICAgIH0sXG5cbiAgICBnZXRJbml0aWFsU3RhdGUoKSB7XG4gICAgICAgIGNvbnN0IHsgdmFsdWUsIG9uQ2hhbmdlIH0gPSB0aGlzLnByb3BzO1xuXG4gICAgICAgIGlmICgodHlwZW9mIHZhbHVlID09ICdzdHJpbmcnKSAmJiAhb25DaGFuZ2UpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignb25DaGFuZ2Ugc2hvdWxkIGJlIHBhc3NlZCB0byBBdXRvY29tcGxldGUgd2hlbiB2YWx1ZSBpcyBwYXNzZWQnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICB2YWx1ZTogICB2YWx1ZSB8fCAnJyxcbiAgICAgICAgICAgIGN1cnNvcjogIG51bGwsXG4gICAgICAgICAgICBsb2FkaW5nOiBmYWxzZSxcbiAgICAgICAgICAgIGZvY3VzZWQ6IGZhbHNlLFxuICAgICAgICAgICAgcmVzdWx0czogW11cbiAgICAgICAgfTtcbiAgICB9LFxuXG4gICAgY29tcG9uZW50V2lsbFJlY2VpdmVQcm9wcyhuZXh0UHJvcHMpIHtcbiAgICAgICAgY29uc3QgdmFsdWUgPSBuZXh0UHJvcHMudmFsdWUgfHwgJyc7XG4gICAgICAgIHRoaXMudXBkYXRlVmFsdWUodmFsdWUpO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBVcGRhdGUgdGhlIHZhbHVlLlxuICAgICAqL1xuICAgIHVwZGF0ZVZhbHVlKHZhbHVlKSB7XG4gICAgICAgIGNvbnN0IHsgdmFsdWU6IHByZXZWYWx1ZSB9ID0gdGhpcy5zdGF0ZTtcbiAgICAgICAgY29uc3QgeyBvbkZldGNoIH0gPSB0aGlzLnByb3BzO1xuXG4gICAgICAgIGlmIChwcmV2VmFsdWUgPT0gdmFsdWUpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuc2V0U3RhdGUoe1xuICAgICAgICAgICAgdmFsdWUsXG4gICAgICAgICAgICBsb2FkaW5nOiB0cnVlXG4gICAgICAgIH0pO1xuXG4gICAgICAgIG9uRmV0Y2godmFsdWUsIChyZXN1bHRzKSA9PiB7XG4gICAgICAgICAgICB0aGlzLnNldFN0YXRlKHtcbiAgICAgICAgICAgICAgICBsb2FkaW5nOiBmYWxzZSxcbiAgICAgICAgICAgICAgICByZXN1bHRzXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIFR5cGVkIHZhbHVlIGNoYW5nZWQsIHdlIGZldGNoIHRoZSBuZXcgYXV0b2NvbXBsZXRlIHJlc3VsdFxuICAgICAqL1xuICAgIG9uSW5wdXRDaGFuZ2VkKGUpIHtcbiAgICAgICAgY29uc3QgeyBvbkNoYW5nZSB9ID0gdGhpcy5wcm9wcztcbiAgICAgICAgY29uc3QgeyB2YWx1ZSB9ID0gZS50YXJnZXQ7XG5cbiAgICAgICAgdGhpcy51cGRhdGVWYWx1ZSh2YWx1ZSk7XG5cbiAgICAgICAgaWYgKG9uQ2hhbmdlKSB7XG4gICAgICAgICAgICBvbkNoYW5nZSh2YWx1ZSk7XG4gICAgICAgIH1cbiAgICB9LFxuXG4gICAgLyoqXG4gICAgICogVXNlciBpcyBmb2N1c2luZy9ibHVyIHRoZSBpbnB1dFxuICAgICAqL1xuICAgIG9uRm9jdXNDaGFuZ2VkKGlzRm9jdXNlZCkge1xuICAgICAgICBjb25zdCB7IG9uRm9jdXMsIG9uQmx1ciB9ID0gdGhpcy5wcm9wcztcblxuICAgICAgICBpZiAoaXNGb2N1c2VkICYmIG9uRm9jdXMpIHtcbiAgICAgICAgICAgIG9uRm9jdXMoKTtcbiAgICAgICAgfSBlbHNlIGlmIChvbkJsdXIpIHtcbiAgICAgICAgICAgIG9uQmx1cigpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5zZXRTdGF0ZSh7XG4gICAgICAgICAgICBmb2N1c2VkOiBpc0ZvY3VzZWRcbiAgICAgICAgfSk7XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIFN1Ym1pdCB2YWx1ZSBhdCBjdXJzb3JcbiAgICAgKi9cbiAgICBvbkVudGVyKCkge1xuICAgICAgICBjb25zdCB7IGN1cnNvciwgdmFsdWUgfSA9IHRoaXMuc3RhdGU7XG4gICAgICAgIGNvbnN0IHsgb25FbnRlciB9ID0gdGhpcy5wcm9wcztcblxuICAgICAgICBpZiAoY3Vyc29yID49IDApIHtcbiAgICAgICAgICAgIHRoaXMub25TZWxlY3QoY3Vyc29yKTtcbiAgICAgICAgfSBlbHNlIGlmIChvbkVudGVyKSB7XG4gICAgICAgICAgICBvbkVudGVyKHZhbHVlKTtcbiAgICAgICAgICAgIHRoaXMuc2V0U3RhdGUoe1xuICAgICAgICAgICAgICAgIGZvY3VzZWQ6IGZhbHNlLFxuICAgICAgICAgICAgICAgIGN1cnNvcjogbnVsbCxcbiAgICAgICAgICAgICAgICByZXN1bHRzOiBbXSxcbiAgICAgICAgICAgICAgICB2YWx1ZTogJydcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIFN1Ym1pdCBhIHZhbHVlXG4gICAgICovXG4gICAgb25TZWxlY3QoaW5kZXgpIHtcbiAgICAgICAgY29uc3QgeyBvblNlbGVjdCB9ID0gdGhpcy5wcm9wcztcbiAgICAgICAgY29uc3QgeyB2YWx1ZSwgcmVzdWx0cyB9ID0gdGhpcy5zdGF0ZTtcbiAgICAgICAgY29uc3Qgc2VsZWN0ZWQgPSByZXN1bHRzW2luZGV4XTtcblxuICAgICAgICB0aGlzLnNldFN0YXRlKHtcbiAgICAgICAgICAgIGN1cnNvcjogbnVsbCxcbiAgICAgICAgICAgIHJlc3VsdHM6IFtdLFxuICAgICAgICAgICAgdmFsdWU6ICcnXG4gICAgICAgIH0pO1xuXG4gICAgICAgIG9uU2VsZWN0KHZhbHVlLCBzZWxlY3RlZCk7XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIFVzZXIgcHJlc3NlZCBhIGtleSBpbiB0ZXh0IGlucHV0XG4gICAgICovXG4gICAgb25LZXlEb3duKGUpIHtcbiAgICAgICAgbGV0IHsgY3Vyc29yLCByZXN1bHRzIH0gPSB0aGlzLnN0YXRlO1xuXG4gICAgICAgIGlmIChlLmtleUNvZGUgPT09IEtFWUNPREVfRU5URVIpIHtcbiAgICAgICAgICAgIGUucHJldmVudERlZmF1bHQoKTtcbiAgICAgICAgICAgIHRoaXMub25FbnRlcigpO1xuICAgICAgICB9XG4gICAgICAgIGVsc2UgaWYgKGUua2V5Q29kZSA9PT0gS0VZQ09ERV9ET1dOKSB7XG4gICAgICAgICAgICBlLnByZXZlbnREZWZhdWx0KCk7XG4gICAgICAgICAgICBjdXJzb3IrKztcbiAgICAgICAgfVxuICAgICAgICBlbHNlIGlmIChlLmtleUNvZGUgPT09IEtFWUNPREVfVVApIHtcbiAgICAgICAgICAgIGUucHJldmVudERlZmF1bHQoKTtcbiAgICAgICAgICAgIGN1cnNvci0tO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGN1cnNvciA+PSByZXN1bHRzLmxlbmd0aCkge1xuICAgICAgICAgICAgY3Vyc29yID0gKHJlc3VsdHMubGVuZ3RoIC0gMSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKGN1cnNvciA8IC0xKSB7XG4gICAgICAgICAgICBjdXJzb3IgPSAtMTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuc2V0U3RhdGUoe1xuICAgICAgICAgICAgY3Vyc29yXG4gICAgICAgIH0pO1xuICAgIH0sXG5cbiAgICAvKipcbiAgICAgKiBSZW5kZXIgdGhlIHN1Z2dlc3Rpb25zXG4gICAgICovXG4gICAgcmVuZGVyUmVzdWx0cygpIHtcbiAgICAgICAgY29uc3QgeyByZXN1bHRzLCB2YWx1ZSwgY3Vyc29yIH0gPSB0aGlzLnN0YXRlO1xuICAgICAgICBjb25zdCBSZXN1bHRDb21wb25lbnQgPSB0aGlzLnByb3BzLnJlbmRlclJlc3VsdDtcblxuICAgICAgICByZXR1cm4gKFxuICAgICAgICAgICAgPGRpdiBjbGFzc05hbWU9XCJBdXRvY29tcGxldGVSZXN1bHRzXCI+XG4gICAgICAgICAgICAgICAge3Jlc3VsdHMubWFwKChyZXN1bHQsIGkpID0+IHtcbiAgICAgICAgICAgICAgICAgICAgY29uc3QgaXNBY3RpdmUgPSAoaSA9PT0gY3Vyc29yKTtcblxuICAgICAgICAgICAgICAgICAgICByZXR1cm4gPEF1dG9jb21wbGV0ZVJlc3VsdCBrZXk9e3ZhbHVlICsgJy0nICsgaX0gYWN0aXZlPXtpc0FjdGl2ZX1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIG9uQ2xpY2s9e2UgPT4gdGhpcy5vblNlbGVjdChpKX0+XG4gICAgICAgICAgICAgICAgICAgICAgICA8UmVzdWx0Q29tcG9uZW50IHJlc3VsdD17cmVzdWx0fSBpbmRleD17aX0gYWN0aXZlPXtpc0FjdGl2ZX0gLz5cbiAgICAgICAgICAgICAgICAgICAgPC9BdXRvY29tcGxldGVSZXN1bHQ+O1xuICAgICAgICAgICAgICAgIH0pfVxuICAgICAgICAgICAgPC9kaXY+XG4gICAgICAgICk7XG4gICAgfSxcblxuICAgIC8qKlxuICAgICAqIEZvY3VzIG9yIGJsdXIgdGhlIGF1dG9jb21wbGV0ZVxuICAgICAqL1xuICAgIGZvY3VzKCkge1xuICAgICAgICBjb25zdCB7IGlucHV0IH0gPSB0aGlzLnJlZnM7XG4gICAgICAgIGlucHV0LmZvY3VzKCk7XG4gICAgfSxcbiAgICBibHVyKCkge1xuICAgICAgICBjb25zdCB7IGlucHV0IH0gPSB0aGlzLnJlZnM7XG4gICAgICAgIGlucHV0LmJsdXIoKTtcbiAgICB9LFxuXG4gICAgcmVuZGVyKCkge1xuICAgICAgICBjb25zdCB7IG9uUGFzdGUsIHNpemUsIHBsYWNlaG9sZGVyIH0gPSB0aGlzLnByb3BzO1xuICAgICAgICBjb25zdCB7IHZhbHVlLCBmb2N1c2VkLCBsb2FkaW5nLCByZXN1bHRzIH0gPSB0aGlzLnN0YXRlO1xuXG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8ZGl2IGNsYXNzTmFtZT1cIkF1dG9jb21wbGV0ZVwiPlxuICAgICAgICAgICAgICAgIDxJbnB1dFxuICAgICAgICAgICAgICAgICAgICByZWY9XCJpbnB1dFwiXG4gICAgICAgICAgICAgICAgICAgIHZhbHVlPXt2YWx1ZX1cbiAgICAgICAgICAgICAgICAgICAgcGxhY2Vob2xkZXI9e3BsYWNlaG9sZGVyfVxuICAgICAgICAgICAgICAgICAgICBzaXplPXtzaXplfVxuICAgICAgICAgICAgICAgICAgICBvbkNoYW5nZT17dGhpcy5vbklucHV0Q2hhbmdlZH1cbiAgICAgICAgICAgICAgICAgICAgb25Gb2N1cz17ZSA9PiB0aGlzLm9uRm9jdXNDaGFuZ2VkKHRydWUpfVxuICAgICAgICAgICAgICAgICAgICBvbkJsdXI9e2UgPT4gdGhpcy5vbkZvY3VzQ2hhbmdlZChmYWxzZSl9XG4gICAgICAgICAgICAgICAgICAgIG9uUGFzdGU9e29uUGFzdGV9XG4gICAgICAgICAgICAgICAgICAgIG9uS2V5RG93bj17dGhpcy5vbktleURvd259XG4gICAgICAgICAgICAgICAgLz5cbiAgICAgICAgICAgICAgICB7bG9hZGluZyA/IDxTcGlubmVyIHNpemU9XCJzbVwiIGNlbnRlcmVkPXtmYWxzZX0gLz4gOiAnJ31cbiAgICAgICAgICAgICAgICB7Zm9jdXNlZCAmJiByZXN1bHRzLmxlbmd0aCA+IDAgPyB0aGlzLnJlbmRlclJlc3VsdHMoKSA6ICcnfVxuICAgICAgICAgICAgPC9kaXY+XG4gICAgICAgICk7XG4gICAgfVxufSk7XG5cbi8qKlxuICogQ29udGFpbmVyIGZvciB0aGUgcmVzdWx0cy5cbiAqIEB0eXBlIHtSZWFjdENsYXNzfVxuICovXG5jb25zdCBBdXRvY29tcGxldGVSZXN1bHQgPSBSZWFjdC5jcmVhdGVDbGFzcyh7XG4gICAgcHJvcFR5cGVzOiB7XG4gICAgICAgIGFjdGl2ZTogUmVhY3QuUHJvcFR5cGVzLmJvb2wsXG4gICAgICAgIG9uQ2xpY2s6IFJlYWN0LlByb3BUeXBlcy5mdW5jLFxuICAgICAgICBjaGlsZHJlbjogUmVhY3QuUHJvcFR5cGVzLm5vZGVcbiAgICB9LFxuXG4gICAgcmVuZGVyKCkge1xuICAgICAgICBjb25zdCB7IGFjdGl2ZSwgY2hpbGRyZW4sIG9uQ2xpY2sgfSA9IHRoaXMucHJvcHM7XG4gICAgICAgIHJldHVybiAoXG4gICAgICAgICAgICA8ZGl2IGNsYXNzTmFtZT17Y2xhc3NOYW1lcygnQXV0b2NvbXBsZXRlUmVzdWx0JywgeyBhY3RpdmUgfSl9XG4gICAgICAgICAgICAgICAgIG9uTW91c2VEb3duPXtvbkNsaWNrfT5cbiAgICAgICAgICAgICAgICB7Y2hpbGRyZW59XG4gICAgICAgICAgICA8L2Rpdj5cbiAgICAgICAgKTtcbiAgICB9XG59KTtcblxubW9kdWxlLmV4cG9ydHMgPSBBdXRvY29tcGxldGU7XG4iXX0=