'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Item header>Category 1</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Item divider />
 *     <Dropdown.Item header>Category 2</Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        size: React.PropTypes.string,
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            scroll: false
        };
    },
    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },


    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },


    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            size = _props.size,
            scroll = _props.scroll,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'size', 'scroll']);

        var open = this.state.open;


        className = classNames('dropdown', className, {
            'dropup': up
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            if (child && child.type && (child.type.displayName == 'DropdownItem' || child.type.displayName == 'DropdownDivider')) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(
            Button.Group,
            _extends({}, otherProps, { className: className }),
            inner
        );

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(
                Backdrop,
                { wrapper: content, onClose: this.close, scroll: true },
                React.createElement(
                    DropdownMenu,
                    { size: size, scroll: scroll },
                    items
                )
            );
        } else {
            return content;
        }
    }
});

/**
 * An item in the dropdown.
 * @type {[type]}
 */
var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(event) {
        var _props2 = this.props,
            href = _props2.href,
            onClick = _props2.onClick;


        if (href) {
            return;
        }

        event.preventDefault();
        event.stopPropagation();

        if (onClick) {
            onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props3 = this.props,
            children = _props3.children,
            divider = _props3.divider,
            disabled = _props3.disabled,
            header = _props3.header,
            checked = _props3.checked,
            href = _props3.href;


        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            warning('Prop "header" on Dropdown.Item is deprecated, use Dropdown.Header instead');
            return React.createElement(
                DropdownHeader,
                null,
                children
            );
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement(
            'li',
            { className: disabled ? 'disabled' : '' },
            React.createElement(
                'a',
                _extends({}, this.props, { href: href || '#', onClick: disabled ? null : this.onClick }),
                checked ? React.createElement(
                    'div',
                    { className: 'dropdown-icon pull-left' },
                    React.createElement(Icon, { id: 'check' })
                ) : '',
                inner
            ),
            outer
        );
    }
});

/**
 * A divider in the dropdown items.
 * @type {ReactClass}
 */
var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',
    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

/**
 * An header in the dropdown items
 * @type {ReactClass}
 */
var DropdownHeader = React.createClass({
    displayName: 'DropdownHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        var children = this.props.children;

        return React.createElement(
            'li',
            { className: 'dropdown-header' },
            children
        );
    }
});

/**
 * Container for the dropdown menu.
 * @type {ReactClass}
 */
var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        size: React.PropTypes.string,
        // Auto-scroll in the dropdown ?
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            size: 'sm',
            scroll: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            maxHeight: null
        };
    },


    /**
     * Detect the max height for this dropdown according to position on screen.
     */
    detectMaxSize: function detectMaxSize() {
        var scroll = this.props.scroll;
        var container = this.refs.container;


        if (!scroll) {
            return;
        }

        var rect = container.getBoundingClientRect();
        var maxHeight = window.innerHeight - rect.top - 30;

        this.setState({
            maxHeight: maxHeight
        });
    },
    componentDidMount: function componentDidMount() {
        this.detectMaxSize();
    },
    componentDidUpdate: function componentDidUpdate(prevProps) {
        var hasChanged = prevProps.children != this.props.children;

        if (hasChanged) {
            this.detectMaxSize();
        }
    },
    render: function render() {
        var _props4 = this.props,
            children = _props4.children,
            size = _props4.size;
        var maxHeight = this.state.maxHeight;

        var className = classNames('dropdown-menu', 'dropdown-' + size);

        return React.createElement(
            'div',
            { ref: 'container', className: className },
            React.createElement(
                'ul',
                { style: { maxHeight: maxHeight } },
                children
            )
        );
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemheader' },
            this.props.children
        );
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemdesc' },
            this.props.children
        );
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Header = DropdownHeader;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;
//# sourceMappingURL=data:application/json;base64,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