'use strict';

var React = require('react');
var classNames = require('classnames');

var Input = require('./Input');
var Spinner = require('./Spinner');

var KEYCODE_ENTER = 13;
var KEYCODE_UP = 38;
var KEYCODE_DOWN = 40;

var Autocomplete = React.createClass({
    displayName: 'Autocomplete',

    propTypes: {
        onFetch: React.PropTypes.func.isRequired,
        renderResult: React.PropTypes.func.isRequired,
        // Called when onEnter on the input (no result selected)
        onEnter: React.PropTypes.func.isRequired,
        onPaste: React.PropTypes.func,
        // Focus events
        onFocus: React.PropTypes.func,
        onBlur: React.PropTypes.func,
        // Control the event
        value: React.PropTypes.string,
        // Called when typing - Required when value is passed
        onChange: React.PropTypes.func,
        // Called when selecting an entry
        onSelect: React.PropTypes.func.isRequired,
        // Render options
        placeholder: React.PropTypes.string,
        size: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        var _props = this.props,
            value = _props.value,
            onChange = _props.onChange;


        if (typeof value == 'string' && !onChange) {
            throw new Error('onChange should be passed to Autocomplete when value is passed');
        }

        return {
            value: value || '',
            cursor: null,
            loading: false,
            focused: false,
            results: []
        };
    },
    componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
        var value = nextProps.value || '';
        this.updateValue(value);
    },


    /**
     * Update the value.
     */
    updateValue: function updateValue(value) {
        var _this = this;

        var prevValue = this.state.value;
        var onFetch = this.props.onFetch;


        if (prevValue == value) {
            return;
        }

        this.setState({
            value: value,
            loading: true
        });

        onFetch(value, function (results) {
            _this.setState({
                loading: false,
                results: results
            });
        });
    },


    /**
     * Typed value changed, we fetch the new autocomplete result
     */
    onInputChanged: function onInputChanged(e) {
        var onChange = this.props.onChange;
        var value = e.target.value;


        this.updateValue(value);

        if (onChange) {
            onChange(value);
        }
    },


    /**
     * User is focusing/blur the input
     */
    onFocusChanged: function onFocusChanged(isFocused) {
        var _props2 = this.props,
            onFocus = _props2.onFocus,
            onBlur = _props2.onBlur;


        if (isFocused && onFocus) {
            onFocus();
        } else if (onBlur) {
            onBlur();
        }

        this.setState({
            focused: isFocused
        });
    },


    /**
     * Submit value at cursor
     */
    onEnter: function onEnter() {
        var _state = this.state,
            cursor = _state.cursor,
            value = _state.value;
        var onEnter = this.props.onEnter;


        if (cursor >= 0) {
            this.onSelect(cursor);
        } else if (onEnter) {
            onEnter(value);
            this.setState({
                focused: false,
                cursor: null,
                results: [],
                value: ''
            });
        }
    },


    /**
     * Submit a value
     */
    onSelect: function onSelect(index) {
        var onSelect = this.props.onSelect;
        var _state2 = this.state,
            value = _state2.value,
            results = _state2.results;

        var selected = results[index];

        this.setState({
            cursor: null,
            results: [],
            value: ''
        });

        onSelect(value, selected);
    },


    /**
     * User pressed a key in text input
     */
    onKeyDown: function onKeyDown(e) {
        var _state3 = this.state,
            cursor = _state3.cursor,
            results = _state3.results;


        if (e.keyCode === KEYCODE_ENTER) {
            e.preventDefault();
            this.onEnter();
        } else if (e.keyCode === KEYCODE_DOWN) {
            e.preventDefault();
            cursor++;
        } else if (e.keyCode === KEYCODE_UP) {
            e.preventDefault();
            cursor--;
        }

        if (cursor >= results.length) {
            cursor = results.length - 1;
        }
        if (cursor < -1) {
            cursor = -1;
        }

        this.setState({
            cursor: cursor
        });
    },


    /**
     * Render the suggestions
     */
    renderResults: function renderResults() {
        var _this2 = this;

        var _state4 = this.state,
            results = _state4.results,
            value = _state4.value,
            cursor = _state4.cursor;

        var ResultComponent = this.props.renderResult;

        return React.createElement(
            'div',
            { className: 'AutocompleteResults' },
            results.map(function (result, i) {
                var isActive = i === cursor;

                return React.createElement(
                    AutocompleteResult,
                    { key: value + '-' + i, active: isActive,
                        onClick: function onClick(e) {
                            return _this2.onSelect(i);
                        } },
                    React.createElement(ResultComponent, { result: result, index: i, active: isActive })
                );
            })
        );
    },


    /**
     * Focus or blur the autocomplete
     */
    focus: function focus() {
        var input = this.refs.input;

        input.focus();
    },
    blur: function blur() {
        var input = this.refs.input;

        input.blur();
    },
    render: function render() {
        var _this3 = this;

        var _props3 = this.props,
            onPaste = _props3.onPaste,
            size = _props3.size,
            placeholder = _props3.placeholder;
        var _state5 = this.state,
            value = _state5.value,
            focused = _state5.focused,
            loading = _state5.loading,
            results = _state5.results;


        return React.createElement(
            'div',
            { className: 'Autocomplete' },
            React.createElement(Input, {
                ref: 'input',
                value: value,
                placeholder: placeholder,
                size: size,
                onChange: this.onInputChanged,
                onFocus: function onFocus(e) {
                    return _this3.onFocusChanged(true);
                },
                onBlur: function onBlur(e) {
                    return _this3.onFocusChanged(false);
                },
                onPaste: onPaste,
                onKeyDown: this.onKeyDown
            }),
            loading ? React.createElement(Spinner, { size: 'sm', centered: false }) : '',
            focused && results.length > 0 ? this.renderResults() : ''
        );
    }
});

/**
 * Container for the results.
 * @type {ReactClass}
 */
var AutocompleteResult = React.createClass({
    displayName: 'AutocompleteResult',

    propTypes: {
        active: React.PropTypes.bool,
        onClick: React.PropTypes.func,
        children: React.PropTypes.node
    },

    render: function render() {
        var _props4 = this.props,
            active = _props4.active,
            children = _props4.children,
            onClick = _props4.onClick;

        return React.createElement(
            'div',
            { className: classNames('AutocompleteResult', { active: active }),
                onMouseDown: onClick },
            children
        );
    }
});

module.exports = Autocomplete;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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