'use strict';

var React = require('react');
var moment = require('moment');
require('moment-duration-format');

var dateShape = React.PropTypes.oneOfType([React.PropTypes.number, React.PropTypes.string, React.PropTypes.instanceOf(Date)]);

/**
 * Render an updated relative date or with a specific format.
 *
 * Relative date: <Date date={new Date(...)} />
 * With format: <Date date={new Date(...)} format="%Y %M" />
 */
var DateSpan = React.createClass({
    displayName: 'DateSpan',

    propTypes: {
        // Date to display
        date: dateShape.isRequired,
        // Interval to refresh the display
        refresh: React.PropTypes.number,
        // Format for output
        format: React.PropTypes.string,
        // Is the date in UTC or Local
        utc: React.PropTypes.bool
    },

    contextTypes: {
        now: dateShape
    },

    getDefaultProps: function getDefaultProps() {
        return {
            format: '',
            refresh: 10 * 1000,
            utc: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            now: 0
        };
    },
    tick: function tick() {
        this.setState({
            now: Date.now()
        });
    },
    componentDidMount: function componentDidMount() {
        var _props = this.props,
            refresh = _props.refresh,
            format = _props.format;

        // We tick only once to update date from utc to local

        if (format) {
            setTimeout(this.tick, 1000);
            return;
        }

        this.interval = setInterval(this.tick, refresh);
    },
    componentWillUnmount: function componentWillUnmount() {
        if (!this.interval) {
            return;
        }

        clearInterval(this.interval);
    },
    render: function render() {
        var now = this.state.now || this.context.now;
        var _props2 = this.props,
            date = _props2.date,
            format = _props2.format,
            utc = _props2.utc;

        var displayDate = void 0;

        // Parse the date
        if (utc) {
            date = moment.utc(date);
        } else {
            date = moment(date);
        }

        // Apply formating if provided
        if (format) {
            // If client-side, we use the real date
            if (this.state.now) {
                date = date.local();
            }

            displayDate = date.format(format);
        } else {
            displayDate = date.from(now);
        }

        return React.createElement(
            'span',
            null,
            displayDate
        );
    }
});

/**
 * Render an updated duration.
 *
 * <Date.Duration duration={6000} />
 */
var DateDuration = React.createClass({
    displayName: 'DateDuration',

    propTypes: {
        duration: React.PropTypes.number,
        format: React.PropTypes.string,
        refresh: React.PropTypes.number
    },

    getDefaultProps: function getDefaultProps() {
        return {
            format: 'h [hrs], m [min], s [sec]',
            refresh: 1000
        };
    },
    getInitialState: function getInitialState() {
        return {
            elapsed: 0
        };
    },
    tick: function tick() {
        var elapsed = this.state.elapsed;
        var refresh = this.props.refresh;


        this.setState({
            elapsed: elapsed + refresh
        });
    },
    componentDidMount: function componentDidMount() {
        var refresh = this.props.refresh;

        if (refresh > 0) {
            this.interval = setInterval(this.tick, refresh);
        }
    },
    componentWillUnmount: function componentWillUnmount() {
        clearInterval(this.interval);
    },
    render: function render() {
        var _props3 = this.props,
            duration = _props3.duration,
            format = _props3.format;
        var elapsed = this.state.elapsed;


        duration = duration + elapsed;

        return React.createElement(
            'span',
            null,
            moment.duration(duration).format(format)
        );
    }
});

var DateContext = React.createClass({
    displayName: 'DateContext',

    propTypes: {
        children: React.PropTypes.node,
        now: dateShape
    },

    childContextTypes: {
        now: dateShape
    },

    getChildContext: function getChildContext() {
        return {
            now: this.props.now
        };
    },
    render: function render() {
        return React.Children.only(this.props.children);
    }
});

module.exports = DateSpan;
module.exports.shape = dateShape;
module.exports.Context = DateContext;
module.exports.Duration = DateDuration;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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