'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

function _objectWithoutProperties(obj, keys) { var target = {}; for (var i in obj) { if (keys.indexOf(i) >= 0) continue; if (!Object.prototype.hasOwnProperty.call(obj, i)) continue; target[i] = obj[i]; } return target; }

var React = require('react');
var classNames = require('classnames');

var Button = require('./Button');
var Icon = require('./Icon');
var Backdrop = require('./Backdrop');
var warning = require('./utils/warning');

/**
 * Dropdown (or up). Automatically bound to child Button.
 * See ButtonDropdown implementation to wrap something else than a button.
 *
 * <Dropdown>
 *
 *     <Button ... />
 *
 *     <Dropdown.Header>Category 1</Dropdown.Header>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 *     <Dropdown.Divider />
 *     <Dropdown.Header>Category 2</Dropdown.Header>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *     <Dropdown.Item href={...}> ... </Dropdown.Item>
 *
 * </Dropdown>
 */

var ButtonDropdown = React.createClass({
    displayName: 'ButtonDropdown',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        up: React.PropTypes.bool,
        center: React.PropTypes.bool,
        size: React.PropTypes.string,
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            scroll: false,
            up: false,
            center: false
        };
    },
    getInitialState: function getInitialState() {
        return {
            open: false
        };
    },


    /**
     * Toggle the dopdown
     * @param  {Event} e?
     */
    toggle: function toggle(e) {
        if (e) {
            e.stopPropagation();
        }

        this.setState({
            open: !this.state.open
        });
    },


    /**
     * Close the dropdown
     */
    close: function close() {
        this.setState({
            open: false
        });
    },
    render: function render() {
        var that = this;
        var inner = [];
        var items = [];

        var _props = this.props,
            className = _props.className,
            children = _props.children,
            up = _props.up,
            center = _props.center,
            size = _props.size,
            scroll = _props.scroll,
            otherProps = _objectWithoutProperties(_props, ['className', 'children', 'up', 'center', 'size', 'scroll']);

        var open = this.state.open;


        className = classNames('dropdown', className, {
            'dropup': up,
            'dropcenter': center
        });

        inner = React.Children.map(children, function (child) {
            // If the Button is connected through Redux.connect, it is
            // renamed to "Connect(Button...)"
            if (child && child.type && child.type.displayName && child.type.displayName.indexOf('Button') !== -1) {
                if (!child.props.onClick && !child.props.href) {
                    return React.cloneElement(child, {
                        onClick: that.toggle,
                        dropdownToggle: true
                    });
                }
                return child;
            }

            return null;
        });

        items = React.Children.map(children, function (child) {
            var acceptedChildren = ['DropdownItem', 'DropdownDivider', 'DropdownHeader'];

            if (child && child.type && acceptedChildren.includes(child.type.displayName)) {
                return React.cloneElement(child, {
                    onClick: function onClick() {
                        if (child.props.onClick) {
                            child.props.onClick();
                        }
                        that.close();
                    }
                });
            }
            return null;
        });

        var content = React.createElement(
            Button.Group,
            _extends({}, otherProps, { className: className }),
            inner
        );

        // Wrap in a backdrop when open
        if (open) {
            return React.createElement(
                Backdrop,
                { wrapper: content, onClose: this.close, scroll: true },
                React.createElement(
                    DropdownMenu,
                    { size: size, scroll: scroll },
                    items
                )
            );
        } else {
            return content;
        }
    }
});

/**
 * An item in the dropdown.
 * @type {[type]}
 */
var DropdownItem = React.createClass({
    displayName: 'DropdownItem',

    propTypes: {
        children: React.PropTypes.node,
        onClick: React.PropTypes.func,
        href: React.PropTypes.string,
        disabled: React.PropTypes.bool,
        divider: React.PropTypes.bool,
        header: React.PropTypes.bool,
        checked: React.PropTypes.bool
    },

    onClick: function onClick(event) {
        var _props2 = this.props,
            href = _props2.href,
            onClick = _props2.onClick;


        if (href) {
            return;
        }

        event.preventDefault();
        event.stopPropagation();

        if (onClick) {
            onClick();
        }
    },
    isInner: function isInner(child) {
        return !child || !child.type || child.type.displayName !== 'DropdownMenu';
    },
    render: function render() {
        var _props3 = this.props,
            children = _props3.children,
            divider = _props3.divider,
            disabled = _props3.disabled,
            header = _props3.header,
            checked = _props3.checked,
            href = _props3.href;


        if (divider) {
            warning('Prop "divider" on Dropdown.Item is deprecated, use Dropdown.Divider instead');
            return React.createElement(DropdownDivider, null);
        }
        if (header) {
            warning('Prop "header" on Dropdown.Item is deprecated, use Dropdown.Header instead');
            return React.createElement(
                DropdownHeader,
                null,
                children
            );
        }

        var inner = [],
            outer = [];

        inner = React.Children.map(children, function (child) {
            if (this.isInner(child)) return child;
            return null;
        }, this);

        outer = React.Children.map(children, function (child) {
            if (!this.isInner(child)) return child;
            return null;
        }, this);

        return React.createElement(
            'li',
            { className: disabled ? 'disabled' : '' },
            React.createElement(
                'a',
                _extends({}, this.props, { href: href || '#', onClick: disabled ? null : this.onClick }),
                checked ? React.createElement(
                    'div',
                    { className: 'dropdown-icon pull-left' },
                    React.createElement(Icon, { id: 'check' })
                ) : '',
                inner
            ),
            outer
        );
    }
});

/**
 * A divider in the dropdown items.
 * @type {ReactClass}
 */
var DropdownDivider = React.createClass({
    displayName: 'DropdownDivider',
    render: function render() {
        return React.createElement('li', { className: 'divider' });
    }
});

/**
 * An header in the dropdown items
 * @type {ReactClass}
 */
var DropdownHeader = React.createClass({
    displayName: 'DropdownHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        var children = this.props.children;

        return React.createElement(
            'li',
            { className: 'dropdown-header' },
            children
        );
    }
});

/**
 * Container for the dropdown menu.
 * @type {ReactClass}
 */
var DropdownMenu = React.createClass({
    displayName: 'DropdownMenu',

    propTypes: {
        className: React.PropTypes.string,
        children: React.PropTypes.node,
        size: React.PropTypes.string,
        // Auto-scroll in the dropdown ?
        scroll: React.PropTypes.bool
    },

    getDefaultProps: function getDefaultProps() {
        return {
            size: 'sm',
            scroll: true
        };
    },
    getInitialState: function getInitialState() {
        return {
            maxHeight: null
        };
    },


    /**
     * Detect the max height for this dropdown according to position on screen.
     */
    detectMaxSize: function detectMaxSize() {
        var scroll = this.props.scroll;
        var container = this.refs.container;


        if (!scroll) {
            return;
        }

        var rect = container.getBoundingClientRect();
        var maxHeight = window.innerHeight - rect.top - 30;

        this.setState({
            maxHeight: maxHeight
        });
    },
    componentDidMount: function componentDidMount() {
        this.detectMaxSize();
    },
    componentDidUpdate: function componentDidUpdate(prevProps) {
        var hasChanged = prevProps.children != this.props.children;

        if (hasChanged) {
            this.detectMaxSize();
        }
    },
    render: function render() {
        var _props4 = this.props,
            children = _props4.children,
            size = _props4.size;
        var maxHeight = this.state.maxHeight;

        var className = classNames('dropdown-menu', 'dropdown-' + size);

        return React.createElement(
            'div',
            { ref: 'container', className: className },
            React.createElement(
                'ul',
                { style: { maxHeight: maxHeight } },
                children
            )
        );
    }
});

var ItemHeader = React.createClass({
    displayName: 'ItemHeader',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemheader' },
            this.props.children
        );
    }
});

var ItemDesc = React.createClass({
    displayName: 'ItemDesc',

    propTypes: {
        children: React.PropTypes.node
    },

    render: function render() {
        return React.createElement(
            'div',
            { className: 'dropdown-itemdesc' },
            this.props.children
        );
    }
});

module.exports = ButtonDropdown;
module.exports.Divider = DropdownDivider;
module.exports.Header = DropdownHeader;
module.exports.Item = DropdownItem;
module.exports.Item.Header = ItemHeader;
module.exports.Item.Desc = ItemDesc;
module.exports.Menu = DropdownMenu;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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