'use strict';

var React = require('react');
var moment = require('moment');

var DateSpan = require('./Date');

var TimeLine = React.createClass({
    displayName: 'TimeLine',

    propTypes: {
        currentX: React.PropTypes.number,
        lineTop: React.PropTypes.number,
        lineBottom: React.PropTypes.number,
        lastX: React.PropTypes.number,
        lastPoints: React.PropTypes.array,
        points: React.PropTypes.array,
        setHovered: React.PropTypes.func
    },

    contextTypes: {
        pointRadius: React.PropTypes.number,
        lineWidth: React.PropTypes.number,
        axeColor: React.PropTypes.string
    },

    getInitialState: function getInitialState() {
        return {
            hovered: false
        };
    },
    setHovered: function setHovered(state, params) {
        // Send global state hovered parameters
        if (!state) {
            params = null;
        }

        this.props.setHovered(params);

        this.setState({
            hovered: state
        });
    },
    renderPathLine: function renderPathLine(index) {
        var lastX = this.props.lastX;
        if (!Boolean(lastX)) {
            return null;
        }

        var currentX = this.props.currentX;

        var currentPoint = this.props.points[index];
        var lastPoint = this.props.lastPoints[index];

        return React.createElement('line', { x1: lastX, y1: lastPoint.y, x2: currentX, y2: currentPoint.y,
            stroke: currentPoint.color, strokeWidth: this.context.lineWidth });
    },
    render: function render() {
        var _this = this;

        var hovered = this.state.hovered;
        var _props = this.props,
            currentX = _props.currentX,
            lineTop = _props.lineTop,
            lineBottom = _props.lineBottom,
            points = _props.points;

        // Increase points radius if hovered

        var pointRadius = this.context.pointRadius;
        if (hovered) {
            pointRadius += 2;
        }

        var hoveredParams = {
            x: currentX,
            points: points
        };

        return React.createElement(
            'g',
            null,
            React.createElement('line', { x1: currentX, x2: currentX, y1: lineTop, y2: lineBottom,
                stroke: this.context.axeColor, strokeWidth: this.context.lineWidth }),
            points.map(function (point, index) {
                var color = point.color;
                // Default serie name to index
                var dataSerie = point.serie || index;

                return React.createElement(
                    'g',
                    { key: index },
                    _this.renderPathLine(index),
                    React.createElement('circle', { key: index,
                        onMouseEnter: _this.setHovered.bind(_this, true, hoveredParams),
                        onMouseLeave: _this.setHovered.bind(_this, false),
                        className: 'serie-point' + (hovered ? ' hovered' : ''),
                        r: pointRadius,
                        fill: color,
                        strokeWidth: _this.context.lineWidth * 2,
                        stroke: '#FFFFFF',
                        cx: currentX,
                        cy: point.y,
                        'data-value': point.value,
                        'data-timestamp': Number(point.date),
                        'data-serie': dataSerie }),
                    '}'
                );
            })
        );
    }
});

var Body = React.createClass({
    displayName: 'Body',

    propTypes: {
        dateMin: React.PropTypes.number,
        dateMax: React.PropTypes.number,
        valueMin: React.PropTypes.number,
        valueMax: React.PropTypes.number,
        width: React.PropTypes.number,
        height: React.PropTypes.number,
        innerX: React.PropTypes.number,
        innerY: React.PropTypes.number,
        yBase: React.PropTypes.number,
        series: React.PropTypes.array,
        setHovered: React.PropTypes.func
    },

    contextTypes: {
        seriesStyle: React.PropTypes.array,
        pointRadius: React.PropTypes.number,
        defaultColor: React.PropTypes.string,
        lineWidth: React.PropTypes.number,
        axeColor: React.PropTypes.string
    },

    // Compute X for a point inside Body
    innerPositionX: function innerPositionX(pX) {
        return this.props.innerX + pX * this.props.width;
    },


    // Compute Y for a point inside Body
    innerPositionY: function innerPositionY(pY) {
        return this.props.yBase + this.props.height * (1 - pY);
    },


    // Return inner X value for a date
    getInnerX: function getInnerX(date) {
        return this.innerPositionX((date - this.props.dateMin) / (this.props.dateMax - this.props.dateMin));
    },


    // Return inner Y value for a value
    getInnerY: function getInnerY(value) {
        return this.innerPositionY((value - this.props.valueMin) / (this.props.valueMax - this.props.valueMin));
    },


    // Return correct mapping for a point to draw in TimeLine
    formatPoint: function formatPoint(time, point, index) {
        var seriesStyle = this.context.seriesStyle;
        var serieStyle = seriesStyle[index];

        return {
            color: point.color || (serieStyle ? serieStyle.color : null) || this.context.defaultColor,
            value: point.value,
            serie: point.serie || (serieStyle ? serieStyle.title : index),
            date: time.date,
            y: this.getInnerY(point.value)
        };
    },


    // Draw this time line
    drawTimeLine: function drawTimeLine(time, i, series) {
        // Current time informations
        var currentX = this.getInnerX(time.date);
        var lineTop = this.props.yBase;
        var lineBottom = this.props.yBase + this.props.height;

        var points = time.points.map(this.formatPoint.bind(this, time));

        // Last time informations
        var lastX = null;
        var lastPoints = null;
        var lastTime = series[i - 1];

        if (Boolean(lastTime)) {
            lastX = this.getInnerX(lastTime.date);
            lastPoints = lastTime.points.map(this.formatPoint.bind(this, lastTime));
        }

        return React.createElement(TimeLine, { key: i,
            currentX: currentX,
            lineTop: lineTop,
            lineBottom: lineBottom,
            points: points,
            lastX: lastX,
            lastPoints: lastPoints,
            setHovered: this.props.setHovered });
    },
    render: function render() {
        var series = this.props.series;


        return React.createElement(
            'g',
            null,
            series.map(this.drawTimeLine, this)
        );
    }
});

var XAxis = React.createClass({
    displayName: 'XAxis',

    propTypes: {
        length: React.PropTypes.number,
        dateMin: React.PropTypes.number,
        dateMax: React.PropTypes.number,
        innerX: React.PropTypes.number,
        yTop: React.PropTypes.number
    },

    contextTypes: {
        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    getValuePerInterval: function getValuePerInterval() {
        var dateMin = this.props.dateMin;
        var dateMax = this.props.dateMax;

        // Get optimal interval for X axis
        return [{
            format: 'SSSS',
            interval: 1
        }, {
            format: 'SSSS',
            interval: 10
        }, {
            format: 'SSSS',
            interval: 100
        }, {
            format: 'H:mm:ss',
            interval: 1 * 1000
        }, {
            format: 'H:mm',
            interval: 60 * 1000
        }, {
            format: 'H',
            interval: 60 * 60 * 1000
        }, {
            format: 'DD/MM',
            interval: 1 * 24 * 60 * 60 * 1000
        }, {
            format: 'DD/MM',
            interval: 7 * 24 * 60 * 60 * 1000
        }, {
            format: 'DD/MM',
            interval: 30 * 24 * 60 * 60 * 1000
        }, {
            format: 'YYYY',
            interval: 365 * 24 * 60 * 60 * 1000
        }, {
            format: 'YYYY',
            interval: 10 * 365 * 24 * 60 * 60 * 1000
        }, {
            format: 'YYYY',
            interval: 100 * 365 * 24 * 60 * 60 * 1000
        }].filter(function (interval) {
            var count = (dateMax - dateMin) / interval.interval;
            return count > 1;
        }).map(function (interval) {
            var count = (dateMax - dateMin) / interval.interval;
            return { count: count, interval: interval };
        }).sort(function (a, b) {
            return a.count > b.count;
        })[0].interval;
    },
    render: function render() {
        var _this2 = this;

        var _props2 = this.props,
            dateMin = _props2.dateMin,
            dateMax = _props2.dateMax,
            length = _props2.length,
            innerX = _props2.innerX,
            yTop = _props2.yTop;

        // Compute intervals for rendering dates

        var valuePerInterval = this.getValuePerInterval();
        var wPerMS = length / (dateMax - dateMin);
        var axeXInterval = (dateMax - dateMin) * wPerMS / (this.context.textFontSize * 10);

        // Construct range of intervals
        var intervalRange = [];
        for (var i = 0; i < axeXInterval; i++) {
            intervalRange.push(i);
        }

        return React.createElement(
            'g',
            null,
            intervalRange.map(function (i) {
                var value = i * valuePerInterval.interval;
                var date = new Date(dateMin + value);

                var x = innerX + value * wPerMS;
                // Don't draw point if too far
                if (x > innerX + length) {
                    return null;
                }

                return React.createElement(
                    'text',
                    { key: i, x: x, y: yTop,
                        fontFamily: _this2.context.textFontFamily, fontSize: _this2.context.textFontSize,
                        fill: _this2.context.textColor, textAnchor: 'middle' },
                    moment(date).format(valuePerInterval.format)
                );
            })
        );
    }
});

var YAxis = React.createClass({
    displayName: 'YAxis',

    propTypes: {
        length: React.PropTypes.number,
        valueMin: React.PropTypes.number,
        valueMax: React.PropTypes.number,
        innerX: React.PropTypes.number,
        innerY: React.PropTypes.number,
        yBase: React.PropTypes.number,
        axeYInterval: React.PropTypes.number
    },

    // Computes the optimal tick step for the Y axis
    // We assume: range = Math.abs(upper - lower)
    // i.e: range should not be negative
    optimalTickStep: function optimalTickStep(maxTicks) {
        var valueMin = this.props.valueMin;
        var valueMax = this.props.valueMax;

        var range = valueMax - valueMin;
        var minimum = range / maxTicks;
        var magnitude = Math.pow(10, Math.floor(Math.log(minimum) / Math.log(10)));
        var residual = minimum / magnitude;

        // Tick is an amplified magnitude
        // depending on the residual
        if (residual > 5) {
            return 10 * magnitude;
        } else if (residual > 2) {
            return 5 * magnitude;
        } else if (residual > 1) {
            return 2 * magnitude;
        }
        return magnitude;
    },


    contextTypes: {
        lineWidth: React.PropTypes.number,
        axeColor: React.PropTypes.string,
        axeMarkerWidth: React.PropTypes.number,
        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    render: function render() {
        var _this3 = this;

        var _props3 = this.props,
            valueMin = _props3.valueMin,
            valueMax = _props3.valueMax,
            length = _props3.length,
            innerX = _props3.innerX,
            innerY = _props3.innerY,
            yBase = _props3.yBase;
        var axeYInterval = this.props.axeYInterval;


        var hPerValue = length / (valueMax - valueMin);
        axeYInterval = (valueMax - valueMin) * hPerValue / (this.context.textFontSize * 4);

        // Calcul perfect value per interval (1, 10, 100, 1000, ...)
        var valuePerInterval = this.optimalTickStep(axeYInterval);
        // Construct range of intervals
        var intervalRange = [];
        for (var i = 0; i < axeYInterval + 1; i++) {
            intervalRange.push(i);
        }

        return React.createElement(
            'g',
            null,
            intervalRange.map(function (i) {
                var value = i * valuePerInterval;
                var y = innerY - value * hPerValue;
                var displayValue = value + valueMin;

                // Don't draw point if is too high
                if (y < yBase) {
                    return null;
                }

                var textX = innerX - 2 * _this3.context.axeMarkerWidth;
                var lineX1 = innerX - _this3.context.axeMarkerWidth;
                var lineX2 = innerX;

                return React.createElement(
                    'g',
                    { key: i },
                    React.createElement(
                        'text',
                        { x: textX, y: y,
                            fontFamily: _this3.context.textFontFamily, fontSize: _this3.context.textFontSize,
                            textAnchor: 'end', fill: _this3.context.textColor },
                        displayValue
                    ),
                    React.createElement('line', { x1: lineX1, x2: lineX2, y1: y, y2: y,
                        stroke: _this3.context.axeColor, strokeWidth: _this3.context.lineWidth })
                );
            })
        );
    }
});

var TimeGraph = React.createClass({
    displayName: 'TimeGraph',

    propTypes: {
        // Series
        series: React.PropTypes.array.isRequired,
        seriesStyle: React.PropTypes.array,
        minValue: React.PropTypes.number,

        // Autofill
        autoFill: React.PropTypes.bool,
        autoFillValue: React.PropTypes.number,
        autoFillStartTime: React.PropTypes.number,
        autoFillEndTime: React.PropTypes.number,

        // Size
        width: React.PropTypes.number,
        height: React.PropTypes.number,
        padding: React.PropTypes.number,

        // Styling
        pointRadius: React.PropTypes.number,
        lineWidth: React.PropTypes.number,
        defaultColor: React.PropTypes.string,

        axeColor: React.PropTypes.string,
        axeMarkerWidth: React.PropTypes.number,

        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    childContextTypes: {
        seriesStyle: React.PropTypes.array,
        pointRadius: React.PropTypes.number,
        defaultColor: React.PropTypes.string,
        lineWidth: React.PropTypes.number,

        axeColor: React.PropTypes.string,
        axeMarkerWidth: React.PropTypes.number,

        textColor: React.PropTypes.string,
        textFontSize: React.PropTypes.number,
        textFontFamily: React.PropTypes.string
    },

    getDefaultProps: function getDefaultProps() {
        return {
            // Autofill
            autoFill: false,
            autoFillValue: 0,
            autoFillInterval: 1000 * 60 * 60 * 24,
            autoFillStartTime: null,
            autoFillEndTime: null,

            // Size
            width: 800,
            height: 400,
            padding: 20,

            // Styling
            pointRadius: 4,
            lineWidth: 1,
            defaultColor: '#26cb7c',

            axeColor: '#eee',
            axeMarkerWidth: 10,

            textColor: '#aaa',
            textFontSize: 10,
            textFontFamily: 'helvetica'
        };
    },
    getChildContext: function getChildContext() {
        return {
            seriesStyle: this.props.seriesStyle,
            pointRadius: this.props.pointRadius,
            defaultColor: this.props.defaultColor,
            lineWidth: this.props.lineWidth,
            axeColor: this.props.axeColor,
            axeMarkerWidth: this.props.axeMarkerWidth,
            textColor: this.props.textColor,
            textFontSize: this.props.textFontSize,
            textFontFamily: this.props.textFontFamily
        };
    },
    getInitialState: function getInitialState() {
        return this.getStateFromProps(this.props);
    },
    componentWillReceiveProps: function componentWillReceiveProps(nextProps) {
        this.setState(this.getStateFromProps(nextProps));
    },
    getStateFromProps: function getStateFromProps(props) {
        var width = props.width;
        var height = props.height;
        var padding = props.padding;
        var minValue = props.minValue;
        var defaultColor = props.defaultColor;
        var seriesStyle = props.seriesStyle;
        var series = props.series;

        // Compute values ranges
        var dateMin = null;
        var dateMax = null;
        var valueMin = null;
        var valueMax = null;

        series = series.map(function (time, i) {
            // Set min/max dates
            var date = new Date(time.date).getTime();
            dateMin = Boolean(dateMin) ? Math.min(dateMin, date) : date;
            dateMax = Boolean(dateMax) ? Math.max(dateMax, date) : date;

            // Set min/max values
            var points = time.points;
            points.forEach(function (point) {
                var value = point.value;
                valueMin = Boolean(valueMin) ? Math.min(valueMin, value) : value;
                valueMax = Boolean(valueMax) ? Math.max(valueMax, value) : value;
            });

            return {
                date: date,
                points: points
            };
        });

        // Sort by date
        series.sort(function (a, b) {
            return a.date - b.date;
        });

        // Set minValue if set
        if (typeof minValue != 'undefined' && minValue < valueMin) {
            valueMin = minValue;
        }

        // Auto-fill
        if (props.autoFill) {
            var autoFillStartTime = props.autoFillStartTime;
            var autoFillEndTime = props.autoFillEndTime;
            var autoFillInterval = props.autoFillInterval;
            var autoFillValue = props.autoFillValue;

            // Set autoFill times to timestamps if provided
            if (Boolean(autoFillStartTime)) {
                autoFillStartTime = new Date(autoFillStartTime).getTime();
            }
            if (Boolean(autoFillEndTime)) {
                autoFillEndTime = new Date(autoFillEndTime).getTime();
            }
            // Set start and end time depending on options
            dateMin = Boolean(autoFillStartTime) ? autoFillStartTime : dateMin;
            dateMax = Boolean(autoFillEndTime) ? autoFillEndTime : dateMax;

            // Set endTime to construct serie
            var serieEndTime = Boolean(autoFillEndTime) ? dateMax : dateMax + autoFillInterval;

            // Set valueMin and valueMax
            valueMin = Math.min(valueMin, autoFillValue);
            valueMax = Math.max(valueMax, autoFillValue);

            // Construct time range
            var timeRange = [];
            for (var t = dateMin; t < serieEndTime; t += autoFillInterval) {
                timeRange.push(t);
            }

            // Fill current serie with existing points or with autoFillValue
            var seriesIndex = 0;

            series = timeRange.map(function (time, i) {
                var data = series[seriesIndex];

                if (Boolean(data) && data.date == time) {
                    seriesIndex++;
                    return data;
                } else {
                    // Construct missing points
                    return {
                        date: time,
                        points: seriesStyle.map(function (style, serieI) {
                            return {
                                serie: style.title || serieI,
                                value: autoFillValue,
                                color: style.color || defaultColor
                            };
                        })
                    };
                }
            });
        }

        if (valueMax == valueMin) {
            valueMax = valueMin + 1;
        }

        // Compute axe sizes
        var axeXHeight = props.textFontSize * 3;
        var axeYWidth = valueMax.toFixed(0).length * props.textFontSize * 1.5 + props.axeMarkerWidth;

        var axeXLength = width - 2 * padding - axeYWidth;
        var axeYLength = height - 2 * padding - axeXHeight;

        // Compute body size
        var innerX = axeYWidth + padding;
        var innerY = height - (axeXHeight + padding);

        var yBase = padding;
        var yTop = height - padding;

        // No tooltip by default
        var hovered = null;

        return {
            width: width,
            height: height,
            series: series,
            dateMin: dateMin,
            dateMax: dateMax,
            valueMin: valueMin,
            valueMax: valueMax,
            axeXLength: axeXLength,
            axeYLength: axeYLength,
            innerX: innerX,
            innerY: innerY,
            yTop: yTop,
            yBase: yBase,
            hovered: hovered
        };
    },
    setHovered: function setHovered(params) {
        this.setState({
            hovered: params
        });
    },
    renderTooltip: function renderTooltip() {
        var width = this.props.width;
        var hovered = this.state.hovered;

        if (!hovered) {
            return null;
        }

        var x = hovered.x,
            points = hovered.points;


        var sumPointsY = points.reduce(function (total, point) {
            return total + point.y;
        }, 0);
        var tooltipY = Math.floor(sumPointsY / points.length);

        // Pretty display date
        var pointsDateStr = new Date(points[0].date).toLocaleString();
        pointsDateStr = pointsDateStr.split(' ')[0].split('-').join('/');

        var tooltipStyle = {
            top: tooltipY - 20
        };
        // Set tooltip to right or left depending on values position in TimeGraph
        if (x > width / 2) {
            tooltipStyle.right = width - x + 15;
        } else {
            tooltipStyle.left = x + 15;
        }

        return React.createElement(
            'div',
            { className: 'time-graph-tooltip', style: tooltipStyle },
            React.createElement(
                'h6',
                { className: 'points-date' },
                React.createElement(DateSpan, { date: new Date(points[0].date), format: 'dddd, MMMM D, YYYY' })
            ),
            React.createElement(
                'table',
                { className: 'points-details' },
                React.createElement(
                    'tbody',
                    null,
                    React.createElement(
                        'tr',
                        { className: 'points-colors' },
                        points.map(function (point, index) {
                            return React.createElement('td', { key: index, style: { backgroundColor: point.color } });
                        })
                    ),
                    React.createElement(
                        'tr',
                        { className: 'points-values' },
                        points.map(function (point, index) {
                            return React.createElement(
                                'td',
                                { key: index },
                                React.createElement(
                                    'b',
                                    null,
                                    point.value
                                ),
                                ' ',
                                point.serie
                            );
                        })
                    )
                )
            )
        );
    },
    render: function render() {
        var _state = this.state,
            width = _state.width,
            height = _state.height,
            series = _state.series,
            dateMin = _state.dateMin,
            dateMax = _state.dateMax,
            valueMin = _state.valueMin,
            valueMax = _state.valueMax,
            axeXLength = _state.axeXLength,
            axeYLength = _state.axeYLength,
            innerX = _state.innerX,
            innerY = _state.innerY,
            yTop = _state.yTop,
            yBase = _state.yBase;


        return React.createElement(
            'div',
            { className: 'time-graph' },
            this.renderTooltip(),
            React.createElement(
                'svg',
                { width: width, height: height, viewBox: '0 0 ' + width + ' ' + height, preserveAspectRatio: 'xMidYMid meet' },
                React.createElement(XAxis, { length: axeXLength,
                    dateMin: dateMin,
                    dateMax: dateMax,
                    innerX: innerX,
                    yTop: yTop }),
                React.createElement(YAxis, { length: axeYLength,
                    valueMin: valueMin,
                    valueMax: valueMax,
                    innerX: innerX,
                    innerY: innerY,
                    yBase: yBase }),
                React.createElement(Body, { series: series,
                    dateMin: dateMin,
                    dateMax: dateMax,
                    valueMin: valueMin,
                    valueMax: valueMax,
                    width: axeXLength,
                    height: axeYLength,
                    innerX: innerX,
                    innerY: innerY,
                    yBase: yBase,
                    setHovered: this.setHovered })
            )
        );
    }
});

module.exports = TimeGraph;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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