/**
 * Hypertable client interface.
 * @example
 * var hypertable = require('hypertable');
 * @module hypertable
 */

var util = require('util');

exports = module.exports = require('./Client_types');

var Hql_types = require('./Hql_types');
exports.HqlResult = Hql_types.HqlResult;
exports.HqlResult2 = Hql_types.HqlResult2;
exports.HqlResultAsArrays = Hql_types.HqlResultAsArrays;
exports.ThriftClient = require('./ThriftClient.js');
exports.Int64 = require('node-int64');
exports.Timestamp = require('./Timestamp');
exports.SerializedCellsVersion = require('./SerializedCellsVersion.js');
exports.SerializedCellsReader = require('./SerializedCellsReader.js');
exports.SerializedCellsWriter = require('./SerializedCellsWriter.js');

/**
 * Enum for SerializedCells constants
 * @readonly
 * @enum {number}
 */
exports.SerializedCellsFlag = {
  EOB:  0x01,
  EOS:  0x02,
  FLUSH: 0x04,
  REV_IS_TS: 0x10,
  AUTO_TIMESTAMP: 0x20,
  HAVE_TIMESTAMP: 0x40,
  HAVE_REVISION:  0x80
};

/** SerializedCells version */
exports.SerializedCellsVersion = 1;

/**
 * Return string representation of a Key.  The following is example output:
 * <pre>
 * Key(row='/foo/bar.html', column_family='count', column_qualifier='2014-06-14 07:31:18', timestamp=1427841277086165001, revision=1427841277086165001, flag=255)
 * </pre>
 * The timestamp and revision fields are excluded from the output if they are
 * null or undefined.
 * @name Key#toString
 * @method
 * @return {String} Printable representation of the Key.
 * @memberof module:hypertable
 */
exports.Key.prototype.toString = function() {
  var tmpTimestamp;
  var timestampField = '';
  if (this.timestamp !== null && this.timestamp !== undefined) {
    tmpTimestamp = new exports.Timestamp(this.timestamp);
    timestampField = ', timestamp=' + tmpTimestamp.toString();
  }
  var revisionField = '';
  if (this.revision !== null && this.revision !== undefined) {
    tmpTimestamp = new exports.Timestamp(this.revision);
    revisionField = ', revision=' + tmpTimestamp.toString();
  }
  return util.format('Key(row=\'%s\', column_family=\'%s\', column_qualifier=\'%s\'%s%s, flag=%d)',
                     this.row, 
                     this.column_family,
                     this.column_qualifier,
                     timestampField,
                     revisionField,
                     this.flag);
};

/**
 * Return string representation of a Cell.  The following is example output:
 * <pre>
 * Cell(key=Key(row='/foo/bar.html', column_family='count', column_qualifier='2014-06-14 07:31:18', timestamp=1427841277086165001, revision=1427841277086165001, flag=255), value='2')
 * </pre>
 * @name Cell#toString
 * @method
 * @return {String} Printable representation of the Cell.
 * @memberof module:hypertable
 */
exports.Cell.prototype.toString = function() {
  return util.format("Cell(key=%s, value='%s')", this.key.toString(), this.value);
};

