/* jshint node: true */

(function () {
  'use strict';

  var layer2 = require('../src/js'),
      assert = require('assert'),
      util = require('util');

  var device = 'en0';
  var opts = {monitor: true, promisc: true}; // Speed up.

  var sampleCapturePath = './test/dat/mesh3.pcap';
  var totalPackets = 3;

  var smallCapture = {
    path: './test/dat/mesh3.pcap',
    length: 3
  };

  var largeCapture = {
    path: './test/dat/mesh780.pcap',
    length: 780
  };

  function checkEqual(pathA, pathB) {

    var replayA = new layer2.capture.Replay(pathA);
    var replayB = new layer2.capture.Replay(pathB);

    var a, b;
    while ((a = replayA.read()) !== null || (b = replayB.read()) !== null) {
      assert.deepEqual(a, b);
    }

  }

  function test1() {
    var nDispatches = 0;
    var nPackets = 0;
    var capture = new layer2.capture.Replay(sampleCapturePath, {
      batchSize: 1
    });
    capture
      .on('dispatch', function () {
        nDispatches++;
      })
      .on('data', function () { nPackets++; })
      .on('end', function () {
        assert.equal(nPackets, totalPackets);
        assert.equal(nDispatches, totalPackets + 1);
      });
  }

  function test2() {
    var capture = new layer2.capture.Replay(sampleCapturePath, {
      batchSize: 2
    });
    capture.once('readable', function () {
      // Note the `once`.
      assert.ok(capture.read() !== null);
      capture.close();
    });
  }

  function test3() {
    var capture = new layer2.capture.Live(device, opts);
    capture.once('readable', function () {
      assert.ok(capture.read() !== null);
      capture.close();
    });
  }

  function test4() {
    var capture = new layer2.capture.Live(device, opts);
    var totalPackets = 10;
    var nPackets = 0;
    var nExtraPackets = 0;
    var stats;

    capture
      .on('data', function (data) {
        assert.ok(data !== null);
        if (nPackets++ > totalPackets) {
          if (!capture.isClosed()) {
            stats = capture.getStats();
            capture.close();
            nExtraPackets = 0; // Reset this.
          } else {
            nExtraPackets++;
          }
        }
      })
      .on('end', function () {
        assert.ok(nPackets >= totalPackets);
        assert.ok(nExtraPackets < 10); // Default batch size.
        assert.ok(stats && stats.psRecv >= nPackets);
      });
  }

  function test5() {
      var capture = new layer2.capture.Replay(sampleCapturePath, {batchSize: 4});

      capture.on('readable', function () {
        assert.ok(capture.read() !== null);
        capture.close();
        console.log('done');
      });

  }

  function test6() {
    var capture = new layer2.capture.Live(device, opts);
    var decoder = new layer2.transform.Decoder('IEEE802_11_RADIO', {stringify: true});
    capture.pipe(decoder).pipe(process.stdout);
  }

  function test7() {
    var capture = new layer2.capture.Replay(sampleCapturePath);
    var decoder = new layer2.transform.Decoder('IEEE802_11_RADIO');
    capture.pipe(decoder).pipe(process.stdout);
  }

  function test8() {
    new layer2.capture.Live(device, opts)
      .on('fetch', function (dispatchUsage, bufUsage) {
        console.log('dispatch: ' + dispatchUsage);
        console.log('buffer: ' + bufUsage);
      })
      .on('data', function (data) {
        var packet;
        try {
          var i = 0;
          while (i++ < 10000) {
            packet = layer2.transform.Decoder.IEEE802_11_RADIO(data);
          }
        } catch (err) {
          console.error('e');
        }
      });
  }

  function test9() {
    var capture = new layer2.capture.Replay('./dat/1.pcap');
    var decoder = layer2.transform.Decoder.IEEE802_11_RADIO;
    var packetN = 0;
    capture
      .on('data', function (data) {
        packetN++;
        try {
          decoder(data);
        } catch (err) {
          console.error(packetN + ': ' + err);
        }
      })
      .on('end', function () { console.log(packetN); });
  }

  function test10() {
    var bssids = {};
    var capture = new layer2.capture.Live(device, opts);

    capture
      .pipe(new layer2.transform.Decoder('IEEE802_11_RADIO'))
      .once('readable', function () {
        setTimeout(function () { capture.close(); }, 10000);
      })
      .on('data', function (packet) {
        var bssid = packet.body.bssid;
        if (bssid) {
          if (!bssids[bssid]) {
            console.log(bssid);
            bssids[bssid] = 1;
          } else {
            bssids[bssid] += 1;
          }
        }
      })
      .on('end', function () {
        console.dir(bssids);
      });
  }

  function test11() {
    var capture = new layer2.capture.Live(device, opts);
    var save = new layer2.capture.Save('dat/foo.pcap', capture.getDatalink());

    capture
      .on('end', save.close.bind(save))
      .pipe(save);

    setTimeout(function () { capture.close(); }, 2000);

  }

  function test12() {

    var savePath = 'dat/write.pcap';
    var replay = new layer2.capture.Replay(sampleCapturePath);
    var save = new layer2.capture.Save(savePath, replay.getDatalink());

    replay
      .on('data', function (buf) { save.write(buf); })
      .on('end', function () {
        save.close();
        checkEqual(savePath, sampleCapturePath);
      });

  }

  function test13() {

      var capture = new layer2.capture.Live(device, opts);
      var ended = false;
      var finished = false;

      capture
        .on('data', function () {})
        .on('finish', function () { console.log('f');finished = true; })
        .on('end', function () { console.log('e');ended = true; })
        .on('close', function () {
        console.log('c');
          // assert.ok(ended);
          // assert.ok(finished);
        });

      setTimeout(function () { capture.push(null); }, 500);

  }

  function test14() {
      var nPackets = 0;
      var nBreaks = 0;

      new layer2.capture.Replay(largeCapture.path, {
        bufferSize: 100000
      })
        .on('break', function () { nBreaks++; })
        .on('data', function () { nPackets++; })
        .on('end', function () {
          assert.equal(nPackets, largeCapture.length);
          assert.ok(nBreaks > 0);
        });
  }

  function test15() {

    var capture = new layer2.capture.Live(null, {promisc: true, monitor: true});
    var packet = new Buffer('000019006f08000066be02f80000000012309e098004d2a400c4006e008438355f8e8a486fb74b', 'hex');

    capture
      .close(1000)
      .write(packet);
  }

  function test16() {

    var capture = new layer2.capture.Live(null, {monitor: true});
    var extractor = new layer2.transform.Extractor();
    var decoder = new layer2.transform.Decoder({stringify: true});
    var save = new layer2.capture.Save('test/dat/mixed.pcap', {
      // linkType: 'IEEE802_11'
    });

    var nInvalid = 0;

    var extracted = capture.pipe(extractor);

    extracted.pipe(save);

    extracted
      // .on('fetch', function (ratio, usage) {
      //   console.log('ratio: ' + ratio + ' usage: ' + usage);
      // })
      .pipe(decoder)
      .on('invalid', function () {
        if (nInvalid++ === 10) {
          capture.close();
        }
      })
      .pipe(process.stdout);
      // .on('data', function () { });

  }

  function test17() {

    var capture = new layer2.capture.Replay('dat/sample.pcap');
    var extractor = new layer2.transform.Extractor();
    var decoder = new layer2.transform.Decoder({stringify: true});

    capture
      .pipe(extractor)
      .pipe(decoder)
      .pipe(process.stdout);

  }

  function test18() {

    var capture = new layer2.capture.Replay('test/dat/mixed.pcap');
    var decoder = new layer2.transform.Decoder({stringify: true});

    capture
      .pipe(decoder)
      .on('invalid', function () { console.log('INVALID'); })
      .on('data', function () { console.log('VALID'); });

  }

  function test19() {

    var capture = new layer2.capture.Live('en0', {monitor: true});
    var decoder = new layer2.decode.Decoder();
    var save = new layer2.capture.Save('dat/all.pcap');

    var nValid = 0;
    var nInvalid = 0;

    capture
      .pipe(save);

    capture
      .pipe(decoder)
      .on('data', function () {
        nValid++;
        process.stderr.write('.');
      })
      .on('invalid', function () {
        nInvalid++;
        process.stderr.write('I');
      });

    process.on('SIGINT', function () {
      capture
        .on('close', function () {
          console.log(util.format(
            '\nTotal: %s (%s%% invalid).',
            (nValid + nInvalid), Math.round(100 * nInvalid / (nValid + nInvalid))
          ));
        })
        .close();
    });

  }

  function test20() {
    var capture = new layer2.capture.Live('en0', {monitor: true});
    var decoder = new layer2.decode.Decoder({stringify: true});
    capture
      .pipe(decoder)
      .pipe(process.stdout);
  }

  function test21() {
    var capture = new layer2.capture.Live('en0', {monitor: true});
    var extractor = new layer2.transform.Extractor();
    var decoder = new layer2.transform.Decoder();
    var nBytesReceived = {};

    capture
      .close(10000)
      .pipe(extractor)
      .pipe(decoder)
      .on('data', function (frame) {
        if (frame.type === 'data' && frame.body) {
          var sa = frame.sa;
          nBytesReceived[sa] = (nBytesReceived[sa] || 0) + frame.body.length;
        }
      })
      .on('end', function () {
        console.dir(nBytesReceived);
      });
  }

  function test22() {

    var capture = new layer2.capture.Live('en0', {monitor: true});
    var decode = layer2.decode.Decoder.decode;
    var save = new layer2.capture.Save('dat/valid.pcap', {
      linkType: capture.getLinkType()
    });

    var nValid = 0;
    var nInvalid = 0;

    capture
      .on('data', function (buf) {
        var frame = decode(this.getLinkType(), buf);
        if (frame !== null) { // Valid frame.
          nValid++;
          process.stderr.write('.');
          save.write(buf);
        } else {
          nInvalid++;
          process.stderr.write('I');
        }
      })
      .on('end', function () { save.end(); });

    process.on('SIGINT', function () {
      capture
        .on('close', function () {
          console.log(util.format(
            '\nTotal: %s (%s%% invalid).',
            (nValid + nInvalid), Math.round(100 * nInvalid / (nValid + nInvalid))
          ));
        })
        .close();
    });

  }

  function test23() {

    var capture = new layer2.capture.Live('en0', {monitor: true});
    var decode = layer2.Decoder.decode;
    var save = new layer2.capture.Save('dat/raw80211.pcap', {
      linkType: 'IEEE802_11'
    });

    var nValid = 0;
    var nInvalid = 0;

    capture
      .on('data', function (buf) {
        var frame = decode(this.getLinkType(), buf);
        if (frame !== null) { // Valid frame.
          var radiocapHeaderLength = frame.headerLength;
          nValid++;
          process.stderr.write('.');
          save.write(buf.slice(radiocapHeaderLength));
        } else {
          nInvalid++;
          process.stderr.write('I');
        }
      })
      .on('end', function () { save.end(); });

    process.on('SIGINT', function () {
      capture
        .on('close', function () {
          console.log(util.format(
            '\nTotal: %s (%s%% invalid).',
            (nValid + nInvalid), Math.round(100 * nInvalid / (nValid + nInvalid))
          ));
        })
        .close();
    });

  }

  test23();

})();
