'use strict';

var express = require('express'),
    router = express.Router(),
    Release = require('app/models/release').Release,
    auth = require('modules/auth'),
    requestRelease = require('app/http/requests/admin/release');

/*
 Method      Route                       Action      Return      Description                         Resource AngularJS
 GET         /article                    index       Html        Articles index.                     ---
 GET         /article/load               load        Action      Load articles.                      query({action: 'load'}... )
 GET	     /article/create             create      Html        Create article.                     ---
 POST        /article	                 store       Action      Store article.                      save(... )
 GET         /article/{article}          show        Html        Preview article.                    ---
 GET	     /article/{article}/edit	 edit        Html        Edit article.                       ---
 GET         /article/{article}/load     load        Action      Load article for show or edit.      get({article: article.id, action: 'load'}... )
 PUT/PATCH   /article/{article}          update      Action      Update article.                     update({article: article.id}... )
 DELETE      /article/{article}          destroy     Action      Delete article.                     delete({article: article.id}... )
 */

// Routes require authentication.
router.use(auth.isAuthenticated);

// Index
router.get('/', function (req, res) {
    res.render('admin/release/index');
});

// List
router.post('/list', function (req, res) {
    var data = req.body,
        conditions = (typeof data.conditions !== ' undefined') ? data.conditions : {},
        options = (typeof data.options !== ' undefined') ? data.options : {};

    if (typeof data.paginate !== 'undefined') {
        // {page: data.page, limit: 5, sort: {version: 'desc'}
        Release.paginate(conditions, data.paginate, function (err, releases) {
            if (err) {
                return res.json(
                    req.response.return('Error loading releases !')
                );
            }

            req.response.setSuccess();
            req.response.setData(releases);
            return res.json(req.response.return());
        });
    } else {
        Release.find(conditions)
            .select(options.select)
            .sort(options.sort)
            .limit(options.limit)
            .exec(function (err, releases) {
                if (err) {
                    return res.json(
                        req.response.return('Error loading releases !')
                    );
                }

                req.response.setSuccess();
                req.response.setData({releases: releases});
                return res.json(req.response.return());
            });
    }
});

// Create
router.get('/create', function (req, res) {
    res.render('admin/release/create');
});

// Store
router.post('/', requestRelease, function (req, res) {
    var newRelease = new Release(req.body);

    newRelease.save(function (err) {
        if (err) {

            console.log(err);
            // Get errors.
            req.response.setMsg(err.errors, 'message');

            return res.json(
                req.response.return()
            );
        }

        req.response.setSuccess();
        return res.json(
            req.response.return('Release registered !')
        );
    });
});

// Show
//router.get('/:release', function (req, res, next) {
//
//});

// Edit
router.get('/:version/edit', function (req, res) {
    return res.render('admin/release/edit');
});

// Load
router.post('/:version/load', function (req, res) {
    Release.findOne({version: req.params.version}, function (err, release) {
        if (err) {
            // Get errors.
            req.response.setMsg(err.errors, 'message');

            return res.json(
                req.response.return()
            );
        }

        if (release) {
            req.response.setSuccess();
            req.response.setData({release: release});
        } else {
            req.response.setMsg("Release not found !");
        }

        return res.json(
            req.response.return()
        );
    });
});

// Update
router.put('/:version', requestRelease, function (req, res) {

    // .findOneAndUpdate( not execute middleware updateAt
    Release.update({version: req.params.version}, req.body, function (err) {
        if (err) {
            // Get errors.
            req.response.setMsg(err.errors, 'message');

            return res.json(
                req.response.return()
            );
        }

        req.response.setSuccess();
        return res.json(
            req.response.return('Release updated !')
        );
    });
});

// Delete
router.delete('/:version', function (req, res) {
    var version = req.params.version;

    Release.findOne({version: version}, function (err, release) {
        if (err) {
            return res.json(
                req.response.return('Error in finding the release.')
            );
        }

        if (!release) {
            return res.json(
                req.response.return('Release not found.')
            );
        }

        /**
         * Note: There is no query hook for remove(), only for documents. If you set a 'remove' hook, it will be fired
         * when you call myDoc.remove(), not when you call MyModel.remove().
         */
        release.remove(function (err) {
            if (err) {
                // Get errors.
                req.response.setMsg(err.errors, 'message');

                return res.json(
                    req.response.return()
                );
            }

            req.response.setSuccess();
            var msg = 'Release ' + version + ' deleted !';

            return res.json(
                req.response.return(msg)
            );
        });
    });
});

module.exports = router;
