'use strict';

var connection = require("./index"),
    mongoose = connection.mongoose,
    Schema = mongoose.Schema,
    uniqueTypes = require('mongoose-unique-validator'),
    paginate = require('modules/mongoose/paginate'),

    /**
     * Schema Documentation.
     *
     * @type {*|Schema}
     */
    documentationSchema = new Schema(
        {
            title: {
                type: String,
                trim: true,
                minlength: [3, 'Title accept min 3 characters !'],
                maxlength: [30, 'Title accept max 30 characters !'],
                required: [true, 'Title is required']
            },
            description: {
                type: String,
                trim: true,
                minlength: [10, 'Description accept max 10 characters !'],
                required: [true, 'Description is required']
            },
            releases: [
                {
                    type: Schema.Types.ObjectId,
                    ref: 'releases'
                }
            ],
            published: {
                type: Boolean,
                required: [true, 'Published is required']
            }
        },
        {
            timestamps: true
        }
    );

/**
 * Create unique index documentation for release.
 */
documentationSchema.index(
    {
        title: 1,
        releases: 1
    },
    {
        unique: ['Documentation for release already registered.']
    }
);

/**
 * Unique types.
 * Documentation:
 *   https://github.com/blakehaswell/mongoose-unique-validator
 */
documentationSchema.plugin(uniqueTypes);

/**
 * Paginate.
 * Documentation:
 *   https://github.com/edwardhotchkiss/mongoose-paginate
 */
documentationSchema.plugin(paginate);

/**
 * Find documentation by release.
 *
 * @param version
 * @param options
 * @param callback
 */
documentationSchema.statics.findByRelease = function (version, options, callback) {
    var query = this;

    var select = options.select || '';

    return require('./release').Release.findOne({version: version})
        .exec().then(function (release) {
            if (!release) {
                return null;
            }

            return query.find({releases: {_id: release._id}})
                .select(select)
                .exec();
        }).asCallback(callback);
};

/**
 * Create model Documentation.
 */
var Documentation = mongoose.model("documentations", documentationSchema);

/**
 * Export model Documentation.
 */
module.exports.Documentation = Documentation;
