'use strict';

var connection = require("./index"),
    mongoose = connection.mongoose,
    Schema = mongoose.Schema,
    uniqueTypes = require('mongoose-unique-validator'),
    paginate = require('modules/mongoose/paginate'),
    random = require('modules/mongoose/random'),
    moment = require('moment'),
    showdown = require('showdown'),
    prettify = require('showdown-prettify'),
    showdownConverter = new showdown.Converter({extensions: ['prettify']}),

    /**
     * Schema Release.
     *
     * @type {*|Schema}
     */
    releaseSchema = new Schema(
        {
            version: {
                type: String,
                trim: true,
                minlength: [3, 'Version accept min 3 characters !'],
                maxlength: [5, 'Version accept max 5 characters !'],
                required: [true, 'Version is required'],
                unique: 'Version already registered.'
            },
            description: {
                type: String,
                trim: true,
                minlength: [10, 'Description accept min 10 characters !'],
                required: [true, 'Description is required']
            },
            published: {
                type: Boolean,
                required: [true, 'Published is required']
            }
        },
        {
            timestamps: true
        }
    );
/**
 * Unique types.
 *
 * Documentation:
 *   https://github.com/blakehaswell/mongoose-unique-validator
 */
releaseSchema.plugin(uniqueTypes);

/**
 * Paginate.
 *
 * Documentation:
 *   https://github.com/edwardhotchkiss/mongoose-paginate
 */
releaseSchema.plugin(paginate);

/**
 * Fetch random document(s) from your mongoose collections.
 *
 * Example:
 *   Model.paginate([query], [options], [callback])
 *
 * Parameters:
 *   [query] {Object} - Query criteria. Documentation: https://docs.mongodb.org/manual/tutorial/query-documents
 *     [options] {Object}
 *       [select] {Object | String} - Fields to return. Documentation: http://mongoosejs.com/docs/api.html#query_Query-select
 *       [limit] {Number} - Range of documents the query will return. Documentation: http://mongoosejs.com/docs/api.html#query_Query-limit
 *       [populate] {Array | Object | String} - Paths which should be populated with other documents. Documentation: http://mongoosejs.com/docs/api.html#query_Query-populate
 *
 */
releaseSchema.plugin(random);

releaseSchema.pre('remove', function(next) {
    var id = this._id;

    require('./documentation').Documentation.find({'releases': {_id: id}})
        .populate('releases')
        .exec(function(err, documentation){

            var error = null;
            if(documentation.length){
                var documentation_title = documentation.map(function(value){
                    return value.title;
                });
                error = new mongoose.Error.ValidationError(this);
                error.errors.relationship = new mongoose.Error.ValidatorError({
                    message: 'Error, Documentation(s): ' + documentation_title.join(', ') + ' - Related to release !'
                });
            }

            return next(error);
    });
});

// Getters
releaseSchema.virtual('createdAtMoment').get(function(){
    return moment.utc(this.createdAt);
});

releaseSchema.virtual('descriptionHtml').get(function(){
    return showdownConverter.makeHtml(this.description)
});

// Setters
//releaseSchema.virtual('createdAt.moment').set(function (date) {
//    this.createdAt = moment.utc(date);
//});

/**
 * Create model Release.
 */
var Release = mongoose.model("releases", releaseSchema);

/**
 * Export model Release.
 */
module.exports.Release = Release;
