'use strict';

var connection = require("./index"),
    mongoose = connection.mongoose,
    Schema = mongoose.Schema,
    uniqueTypes = require('mongoose-unique-validator'),
    hash = require('modules/mongoose/hash'),
    email = require('modules/mongoose/validator/email'),

    /**
     * Schema User.
     *
     * @type {*|Schema}
     */
    UserSchema = new Schema(
        {
            username: {
                type: String,
                trim: true,
                minlength: [3, 'Username accept min 3 characters !'],
                maxlength: [15, 'Username accept max 15 characters !'],
                required: [true, 'Username is required']
            },
            email: {
                type: String,
                trim: true,
                unique: true,
                uniqueCaseInsensitive: true,
                email: 'Email invalid !',
                required: [true, 'Email is required'],
                index: true
            },
            password: {
                type: String,
                minlength: [6, 'Password accept min 6 characters !'],
                hash: 15
            },
            resetPassword: {
                token: {
                    type: String,
                    required: false
                },
                expires: {
                    type: String,
                    required: false
                },
                ip: {
                    type: String,
                    required: false
                }
            },
            oauth: {
                id: {
                    type: String,
                    required: false
                },
                provider: {
                    type: String,
                    required: false
                },
                token: {
                    type: String,
                    required: false
                }
            }
        }
    );

/**
 * Generating a hash password and checking password.
 * 
 * Example Schema:
 *   password: {
 *       type: String,
 *       minlength: [6, 'Password accept min 6 characters !'],
 *       hash: 15
 *   }
 *
 * Method created:
 *   model.comparePaswword('compare'); // return true of false
 *   
 * @param hash
 */
UserSchema.plugin(hash);

/**
 * Add validator email.
 * Example:
 * ...
 *   email: {
 *       type: String,
 *       trim: true,
 *       unique: true,
 *       email: 'Email invalid !',
 *       ...
 *
 * Parameters
 *   email: {
 *       email: {Boolean|String}
 *       ...
 *
 * Default message: 'Field {PATH} {VALUE} not valid !'
 *
 * @param email
 */
UserSchema.plugin(email);

/**
 * Unique types.
 *
 * Documentation:
 *   https://github.com/blakehaswell/mongoose-unique-validator
 *
 * Custom Error Messages
 *   UserSchema.plugin(uniqueTypes, { message: 'Error, expected {PATH} to be unique.' });
 */
UserSchema.plugin(uniqueTypes);

/**
 * Create model User.
 */
var User = mongoose.model("users", UserSchema);

/**
 * Export model User.
 */
module.exports.User = User;
