'use strict';

/**
 * Gulpfile
 *
 * gulp --option:
 *   gulp or gulp watch create files min also not min.
 *   gulp --production create only files min
 */
var argv = require('minimist')(process.argv.slice(2)),
    gulp = require('gulp'),
    settings = require('./config'),
    sourcemaps = require('gulp-sourcemaps'),
    plumber = require('gulp-plumber'),
    rename = require('gulp-rename'),
    autoprefixer = require('gulp-autoprefixer'),
    concat = require('gulp-concat'),
    flatten = require('gulp-flatten'),
    jshint = require('gulp-jshint'),
    uglify = require('gulp-uglify'),
    imagemin = require('gulp-imagemin'),
    cache = require('gulp-cache'),
    minifycss = require('gulp-minify-css'),
    sass = require('gulp-sass'),
    gulpif = require('gulp-if'),
    es = require('event-stream'),
    browserSync = require('browser-sync'),
    runSequence = require('run-sequence'),

    path = {
        bower: './resources/assets/bower/',
        components: './resources/assets/components/',
        dist: './public/',
        src: './resources/assets/',
        angular: './resources/assets/javascripts/angular/'
    },

    files = {
        js: {
            dependencies: {
                jshint: false,
                concat: true,
                path: [
                    path.bower + 'jquery/dist/jquery.js',
                    path.bower + 'moment/moment.js',
                    path.bower + 'showdown/dist/showdown.js',
                    path.bower + 'showdown-prettify/dist/showdown-prettify.js',
                    path.bower + 'code-prettify/src/prettify.js',
                    path.bower + 'bootstrap-sass/assets/javascripts/bootstrap.js',
                    path.bower + 'angular/angular.js',
                    path.bower + 'angular-resource/angular-resource.js',
                    path.bower + 'angular-ui-router/release/angular-ui-router.js',
                    path.bower + 'angular-animate/angular-animate.js',
                    path.bower + 'angular-touch/angular-touch.js',
                    path.bower + 'angular-sanitize/angular-sanitize.js',
                    path.bower + 'oclazyload/dist/ocLazyLoad.js',
                    path.bower + 'ng-showdown/dist/ng-showdown.js',
                    path.bower + 'AngularJS-Toaster/toaster.js',
                    path.bower + 'angular-ui-select/dist/select.js',
                    path.bower + 'angular-bootstrap/ui-bootstrap-tpls.min.js'
                ],
                dist: 'javascripts/'
            },
            angular: {
                jshint: true,
                concat: true,
                path: [
                    path.angular + '/modules/*.js',
                    path.angular + 'app.js',
                    path.angular + 'config.js',
                    path.angular + 'providers.js',
                    path.angular + 'factory/*.js',
                    path.angular + 'services/*.js',
                    path.angular + 'directives/*.js',
                    path.angular + 'routers.js',
                    path.angular + 'run.js',
                    path.angular + 'controllers/app/*.js'
                ],
                dist: 'javascripts/'
            },
            angularControllers: {
                jshint: true,
                concat: false,
                path: [
                    path.angular + 'controllers/partials/**/*.js'
                ],
                dist: 'javascripts/controllers/'
            }
        },
        css: {
            dependencies: [
                // SASS
                path.bower + 'AngularJS-Toaster/toaster.scss',
                path.bower + 'font-awesome/scss/font-awesome.scss',
                // CSS
                path.components + 'code-prettify/theme/github-v2.css',
                path.bower + 'angular-ui-select/dist/select.css'
            ],
            main: [
                path.src + 'stylesheets/sass/main.scss'
            ]
        },
        fonts: [
            path.bower + 'bootstrap-sass/assets/fonts/bootstrap/*.{otf,eot,svg,ttf,woff,woff2}',
            path.bower + 'font-awesome/fonts/*.{otf,eot,svg,ttf,woff,woff2}',
            path.src + 'fonts/**/*.{otf,eot,svg,ttf,woff,woff2}'
        ],
        images: [
            path.src + 'images/**.{png,jpg,jpeg}'
        ],
        icons: [
            path.src + 'icons/*'
        ],
        libs: [
            // Angular Bootstrap
            // {
            //     path: path.bower + 'angular-bootstrap/ui-bootstrap-tpls.min.js',
            //     dist: 'angular-bootstrap/'
            // }
        ]
    };

/**
 * Compress images files.
 *
 * @param files
 * @param dist
 */
var compressImages = function (files, dist, level) {
    var optimizationLevel = (typeof level === 'undefined') ? 3 : level;

    gulp.src(files)
        .pipe(cache(
            imagemin(
                {
                    optimizationLevel: optimizationLevel,
                    progressive: true,
                    interlaced: true
                }
            )
        ))
        .pipe(gulp.dest(dist));
};

/**
 * Compile Scripts
 *
 * @param indexJs
 */
var compileScript = function (indexJs) {

    var indexJsMap = (indexJs instanceof Array) ? indexJs : [indexJs],
        filesJs = files.js;

    return es.merge(indexJsMap.map(function (index) {

        var dist = path.dist + filesJs[index].dist;

        return gulp.src(filesJs[index].path)
            .pipe(sourcemaps.init({loadMaps: true}))
            .pipe(plumber({
                errorHandler: function (error) {
                    console.log(error.message);
                    this.emit('end');
                }
            }))
            .pipe(gulpif(filesJs[index].jshint, jshint()))
            .pipe(jshint.reporter('default'))
            .pipe(gulpif(filesJs[index].concat, concat(index + '.js')))
            .pipe(gulpif(!argv.production, gulp.dest(dist)))
            .pipe(rename({suffix: '.min'}))
            .pipe(uglify())
            .pipe(sourcemaps.write('/maps'))
            .pipe(gulp.dest(dist))
            .pipe(browserSync.reload({stream: true}));
    }));
};

/**
 * Compile CSS and SCSS.
 *
 * @param indexCss
 */
var compileCss = function (indexCss) {

    var indexCssMap = (indexCss instanceof Array) ? indexCss : [indexCss],
        filesCss = files.css;

    return es.merge(indexCssMap.map(function (index) {

        return gulp.src(filesCss[index])
            .pipe(plumber({
                errorHandler: function (error) {
                    console.log(error.message);
                    this.emit('end');
                }
            }))
            .pipe(sass())
            .pipe(concat(index + '.css'))
            .pipe(autoprefixer('last 2 versions'))
            .pipe(gulpif(!argv.production, gulp.dest(path.dist + 'stylesheets/')))
            .pipe(rename({suffix: '.min'}))
            .pipe(minifycss())
            .pipe(gulp.dest(path.dist + 'stylesheets/'))
            .pipe(browserSync.reload({stream: true}));
    }));
};

/**
 * Compress Images
 */
gulp.task('images', function () {
    return compressImages(files.images, path.dist + 'images/');
});

/**
 * Compress Icons
 */
gulp.task('icons', function () {
    return compressImages(files.icons, path.dist + '/');
});

/**
 * Compile scripts dependencies
 */
gulp.task('scripts-dependencies', function () {
    return compileScript('dependencies');
});

/**
 * Compile scripts AngularJs
 */
gulp.task('scripts-angular', function () {
    return compileScript(['angular', 'angularControllers']);
});

/**
 * Compile SASS and compress CSS dependencies
 */
gulp.task('styles-dependencies', function () {
    return compileCss('dependencies');
});

/**
 * Compile SASS and compress CSS main
 */
gulp.task('styles-main', function () {
    return compileCss('main');
});

/**
 * Copy fonts
 */
gulp.task('fonts', function () {
    return gulp.src(files.fonts)
        .pipe(flatten())
        .pipe(gulp.dest(path.dist + 'fonts/'));
});

/**
 * Copy libs
 */
gulp.task('libs', function () {

    // if empty.
    if ((files.libs).length === 0) {
        return this;
    }

    return es.merge(files.libs.map(function (index) {
        return gulp.src(index.path)
            .pipe(flatten({includeParents: 1000}))
            .pipe(gulp.dest(path.dist + 'libs/' + index.dist));
    }));
});

/**
 * Clear directory
 */
gulp.task('clean',
    require('del').bind(null, path.dist)
);

/**
 * Build sequence
 */
gulp.task('build', function (callback) {
    runSequence(
        [
            'libs',
            'fonts',
            'images',
            'icons',
            'styles-dependencies',
            'scripts-dependencies'
        ],
        [
            'styles-main',
            'scripts-angular'
        ],
        callback
    );
});

/**
 * Start Gulp
 */
gulp.task('default', ['clean'], function () {
    gulp.start('build');
});

/**
 * Reload Gulp
 */
gulp.task('bs-reload', function () {
    browserSync.reload();
});

/**
 * Init browserSync
 */
gulp.task('browserSync', function () {
    browserSync.init({
        proxy: settings.server.hostname + ":" + settings.server.port
    });
});

/**
 * Gulp Watch
 */
gulp.task('watch', ['default', 'browserSync'], function () {
    gulp.watch(path.src + '/stylesheets/**', ['styles-main']);
    gulp.watch(path.src + '/javascripts/**', ['scripts-angular']);
    gulp.watch('./resources/views/**', ['bs-reload']);
});
