'use strict';

var express = require('express'),
    router = express.Router(),
    Release = require('app/models/release').Release;

/*
 Method      Route                       Action      Return      Description                         Resource AngularJS
 GET         /article                    index       Html        Articles index.                     ---
 GET         /article/load               load        Action      Load articles.                      query({action: 'load'}... )
 GET	     /article/create             create      Html        Create article.                     ---
 POST        /article	                 store       Action      Store article.                      save(... )
 GET         /article/{article}          show        Html        Preview article.                    ---
 GET	     /article/{article}/edit	 edit        Html        Edit article.                       ---
 GET         /article/{article}/load     load        Action      Load article for show or edit.      get({article: article.id, action: 'load'}... )
 PUT/PATCH   /article/{article}          update      Action      Update article.                     update({article: article.id}... )
 DELETE      /article/{article}          destroy     Action      Delete article.                     delete({article: article.id}... )
 */


// Index
router.get('/', function (req, res) {
    res.render('release/index');
});

// Load releases pagination
router.get('/load/:page', function (req, res) {

    var page = parseInt(req.params.page);

    Release.paginate(
        {
            published: true
        },
        {
            page: page,
            limit: 5,
            select: '-_id version description createdAt',
            sort: {version: 'desc'}
        }, function (err, releases) {
            if (err) {
                return res.status(500).json(
                    req.response.return('Error loading releases !')
                );
            } else if (releases.docs.length === 0) {
                return res.status(404).json(
                    req.response.return('Releases not found !')
                );
            }

            releases.docs = releases.docs.map(function (release) {
                return {
                    version: release.version,
                    description: release.descriptionHtml,
                    createdAt: release.createdAtMoment.format('MMM DD YYYY')
                };
            });

            req.response.setData(releases);
            req.response.setSuccess();
            return res.json(req.response.return());
        });
});

// Load releases
router.get('/load', function (req, res) {

    Release.find({published: true}, '-_id version')
        .sort({version: 'desc'})
        .exec()
        .then(function (releases) {
            if (releases.length === 0) {
                return res.status(404).json(
                    req.response.return('Releases not found !')
                );
            }

            req.response.setData(releases);
            req.response.setSuccess();
            return res.json(req.response.return());
        }).catch(function () {
        return res.status(500).json(
            req.response.return('Error loading releases !')
        );
    });

});

// Load current releases
router.get('/current', function (req, res) {

    Release.findOne({published: true})
        .sort({version: 'desc'})
        .select('-_id version')
        .exec()
        .then(function (releases) {
            if (releases.length === 0) {
                return res.status(404).json(
                    req.response.return('Releases not found !')
                );
            }

            req.response.setData(releases);
            req.response.setSuccess();
            return res.json(req.response.return());
        }).catch(function () {
        return res.status(500).json(
            req.response.return('Error loading releases !')
        );
    });

});

module.exports = router;
