/**
 * Module authentication for AngularJS.
 *
 * Example:
 *
 *   // Auth false
 *   .state('signup', {
 *       url: '/signup',
 *       templateUrl: '/view/login/signup',
 *       ...
 *       // Control access
 *       access: {
 *           // Only access page if not login.
 *           auth: false,
 *           // Optional redirect: URL
 *           redirect: '/myaccount',
 *       }
 *       ...
 *
 *   // Auth true
 *   .state('myaccount', {
 *       url: '/myaccount',
 *       ...
 *       // Control access
 *       access: {
 *           // Only access page if logged.
 *           auth: true,
 *           // Optional setting -> redirect: URL
 *           redirect: 'signin'
 *           // redirect result, domain.com/signin?r=myaccount
 *           // after login will be redirected to myaccount.
 *           // if paramOfRedirect parameter is set to false will not be redirected.
 *           paramOfRedirect: false
 *       }
 *       ...
 *
 * === Config redirect default in file ../config.js ===
 *
 *   ...
 *   'auth': {
 *       redirect: {
 *           notLogged: '/signin',
 *           logged: '/myaccount'
 *           paramOfRedirect: true
 *       }
 *   }
 *   ...
 *
 *
 * === Check user is logged ===
 *
 * return {boolean}
 * service.check()
 *
 */
var AppPath = angular.module('AppAuth', []);

AppPath.provider('auth', [function () {
    var settings_default = {
            auth: {
                redirect: {
                    notLogged: 'signin',
                    paramOfRedirect: true,
                    logged: 'myaccount'
                },
                cookie: {
                    nameFlagLogged: 'login'
                }
            }
        },
        settings = {};

    this.config = function (op_settings) {
        settings = angular.extend(settings_default, op_settings);
    };

    this.$get = ['settings', '$stateParams',
        function (settings, $stateParams) {
            var service = {};

            /**
             * Check user is logged
             *
             * @returns {boolean}
             */
            service.check = function () {
                var cookieName = settings.auth.cookie.nameFlagLogged,
                    getCookieValues = function (cookie) {
                        var cookieArray = cookie.split('=');
                        return cookieArray[1].trim();
                    },
                    getCookieNames = function (cookie) {
                        var cookieArray = cookie.split('=');
                        return cookieArray[0].trim();
                    },
                    cookies = document.cookie.split(';'),
                    cookieValue = cookies.map(getCookieValues)[cookies.map(getCookieNames).indexOf(cookieName)];

                return (((cookieValue === undefined) ? null : cookieValue) === 'true');
            };

            /**
             * Parameters redirection when login.
             * 
             * @returns {{redirect: string, params: {}, defaultRedirect: boolean}}
             */
            service.paramsRedirectionLogin = function () {
                var redirect = (typeof $stateParams.r !== 'undefined') ? decodeURIComponent($stateParams.r) : settings.auth.redirect.logged;

                return {
                    redirect: redirect,
                    params: (typeof $stateParams.p !== 'undefined') ? JSON.parse(decodeURIComponent($stateParams.p)) : {},
                    defaultRedirect: (redirect === settings.auth.redirect.logged)
                };
            };

            /**
             * Route control access
             *
             * @param toState
             */
            service.routeAuth = function (event, $state, toState, toParams) {
                //if route not have control access
                if (typeof toState.access === 'undefined' || typeof toState.access.auth === 'undefined') {
                    return;
                }

                var authRoute = toState.access.auth,
                    redirect = {
                        notLogged: (typeof toState.access.redirect !== 'undefined') ? toState.access.redirect : settings.auth.redirect.notLogged,
                        logged: (typeof toState.access.redirect !== 'undefined') ? toState.access.redirect : settings.auth.redirect.logged
                    };

                //Only access page if logged.
                if (authRoute && !service.check()) {
                    var paramOfRedirect = (typeof toState.access.paramOfRedirect !== 'undefined') ? toState.access.paramOfRedirect : settings.auth.redirect.paramOfRedirect;
                    var params = {};

                    if (paramOfRedirect) {
                        params.r = encodeURIComponent(toState.name);

                        if (Object.keys(toParams).length !== 0) {
                            params.p = encodeURIComponent(JSON.stringify(toParams));
                        }
                    }

                    event.preventDefault();

                    $state.go(redirect.notLogged, params);
                }
                //Only access page if not login.
                else if (!authRoute && service.check()) {

                    event.preventDefault();

                    $state.go(redirect.logged);
                }
            };

            return service;
        }
    ];
}]);

/**
 * Module load file path.
 *   controller,
 *   image,
 *   style,
 *
 * Example:
 *
 *   ...
 *   controller: 'homeController', //Controller page
 *   resolve: {
 *   //Load dependencies
 *   deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
 *       return $ocLazyLoad.load({
 *           //Insert controller before id
 *           insertBefore: '#load_controllers',
 *           files: [
 *               //Return controller path according destination and minify.
 *               path.controller('homeController')
 *           ]
 *       },
 *       ...
 *
 * === Config path directories in file ../config.js ===
 *
 *   ...
 *   path: {
 *       minifyFiles: true,
 *       controller: '/javascripts/',
 *       image: '/images/',
 *       style: '/stylesheets/'
 *   },
 *   ...
 *
 */

var AppPath = angular.module('AppPath', []);

AppPath.provider('path', [function () {
    var settings_default = {
            path: {
                minifyFiles: true,
                scripts: '/javascripts/',
                controllers: '/javascripts/',
                images: '/images/',
                styles: '/stylesheets/',
                libs: '/libs/'
            }
        },
        settings = {};

    this.config = function (op_settings) {
        settings = angular.extend(settings_default, op_settings);
    };

    this.$get = ['settings',
        function (settings) {
            var service = {},
                min = (typeof settings.path.minifyFiles === 'undefined' || settings.path.minifyFiles === true),
                extension = function (type) {
                    return ((min) ? '.min' : '') + '.' + type;
                };

            /**
             * Return controller path according destination and minify.
             *
             * @param controllerName
             * @returns {*}
             */
            service.controller = function (controllerName) {
                return settings.path.controllers + controllerName + extension('js');
            };

            service.script = function (script) {
                return settings.path.scripts + script + extension('js');
            };

            service.style = function (style) {
                return settings.path.styles + style + extension('css');
            };

            service.image = function (image) {
                return settings.path.images + image;
            };

            service.lib = function (lib) {
                return settings.path.libs + lib;
            };

            return service;
        }
    ];
}]);

/**
 * Module Route.
 * Return route with namespace at application.
 *
 * Example:
 *   ['$resource', 'route', function ($resource, route) {
 *      return function (url) {
 *          return $resource(
 *              route.namespace(url),
 *              ...
 *
 *   route.namespace(), return '/api'
 *   route.namespace('/home'), return '/api/home'
 */
var AppRoute = angular.module('AppRoute', []);

AppRoute.provider("route", [function () {
    var settings_default = {
            route: {
                namespace: 'api'
            }
        },
        settings = {};

    this.config = function (op_settings) {
        settings = angular.extend(settings_default, op_settings);
    };

    this.$get = ['settings',
        function (settings) {
            var service = {};

            service.namespace = function (route) {
                if (typeof route === 'undefined') {
                    return '/' + settings.route.namespace;
                }
                return '/' + settings.route.namespace + '/' + route;
            };

            return service;
        }
    ];
}]);

/**
 * Modules for APP AngularJS
 */
var App;
App = angular.module('App', [
    'ui.router',
    'oc.lazyLoad',
    'ngResource',
    'ngAnimate',
    'ngTouch',
    'ngSanitize',
    'ng-showdown',
    'toaster',
    'ui.select',
    'ui.bootstrap',
    'AppPath',
    'AppAuth',
    'AppRoute'
]);

/**
 * Config application
 */
App.constant('settings', {
    path: {
        minifyFiles: true,
        scripts: '/javascripts/',
        controllers: '/javascripts/controllers/',
        images: '/images/',
        styles: '/stylesheets/',
        libs: '/libs/'
    },
    auth: {
        redirect: {
            notLogged: 'signin',
            logged: 'admin',
            paramOfRedirect: true
        },
        cookie: {
            nameFlagLogged: 'login'
        }
    },
    route: {
        namespace: 'api'
    }
});

App.config(['$interpolateProvider', '$locationProvider', 'settings', 'routeProvider', 'pathProvider', 'authProvider', '$ocLazyLoadProvider', '$showdownProvider', '$urlMatcherFactoryProvider',
    function ($interpolateProvider, $locationProvider, settings, routeProvider, pathProvider, authProvider, $ocLazyLoadProvider, $showdownProvider, $urlMatcherFactoryProvider) {

        /**
         * Alter symbol of AngularJS because Handlebars possesses the same symbol.
         *
         * Handlebars:
         *   Alternative delimiters are not supported.
         * https://github.com/wycats/handlebars.js#compatibility
         */
        $interpolateProvider.startSymbol('[[');
        $interpolateProvider.endSymbol(']]');

        /**
         * Config Modules
         *
         * Route, Path, Auth, ocLazyLoad, Location, Showdown, urlMatcherFactoryProvider
         */
        routeProvider.config(settings.route);

        pathProvider.config(settings.path);

        authProvider.config(settings.auth);

        $ocLazyLoadProvider.config({
            debug: true
        });

        $locationProvider.html5Mode({
            enabled: true,
            requireBase: false
        });

        $showdownProvider.loadExtension('prettify');

        // Url contains a trailing slash or not
        $urlMatcherFactoryProvider.strictMode(false);
    }]);

/**
 * Factory service account.
 */
angular.module("App").factory('accountService',
    ['$http', 'route', '$templateCache', '$rootScope', '$state', '$stateParams', 'toaster', 'auth',
        function ($http, route, $templateCache, $rootScope, $state, $stateParams, toaster, auth) {
            return {
                signIn: function (user) {
                    // Clear message.
                    toaster.clear();

                    return $http.post(route.namespace('admin/signin'), user)
                        .success(function (response) {
                            // Clear cache templates.
                            $templateCache.removeAll();

                            // Alter username navbar.
                            $rootScope.$emit('username', response.data.username);

                            // Get parameters for redirect
                            var paramsRedirectionLogin = auth.paramsRedirectionLogin();

                            // Go to page redirect.
                            $state.go(paramsRedirectionLogin.redirect, paramsRedirectionLogin.params);
                        })
                        .error(function (response) {
                            toaster.error("Error", response.msg.join('<br>'));
                        });
                },
                logout: function () {
                    return $http.post(route.namespace('/admin/logout'))
                        .success(function () {
                            // Clear cache templates.
                            $templateCache.removeAll();

                            return $state.go('home');
                        })
                        .error(function () {
                            toaster.error("Error in logout");
                        });
                }
            };
        }]
);

/**
 * Factory service admin release.
 */
angular.module("App").factory('adminReleaseService',
    ['$resource', 'route', function ($resource, route) {
        return $resource(
            route.namespace('admin/release/:release/:action/:page'),
            {
                release: '@release',
                action: '@action',
                page: '@page'
            },
            {
                update: {
                    method: 'PUT'
                },
                query: {
                    isArray: false
                }
            }
        );
    }]
);

/**
 * Factory service Documentation.
 */
angular.module("App").factory('documentationService',
    ['$resource', 'route', function ($resource, route) {
        return $resource(
            route.namespace('docs/:release/:doc/:action'),
            {
                release: '@release',
                doc: '@doc',
                action: '@action'
            },
            {
                update: {
                    method: 'PUT'
                },
                query: {
                    isArray: false
                }
            }
        );
    }]
);

/**
 * Factory service release.
 */
angular.module("App").factory('releaseService',
    ['$resource', 'route', function ($resource, route) {
        return $resource(
            route.namespace('release/:release/:action/:page'),
            {
                release: '@release',
                action: '@action',
                page: '@page'
            },
            {
                update: {
                    method: 'PUT'
                },
                query: {
                    isArray: false
                }
            }
        );
    }]
);

/**
 * Factory request
 */
angular.module("App").factory('request',
    ['$resource', 'route', function ($resource, route) {
        return function (url) {
            return $resource(
                route.namespace(url),
                {},
                {
                    'post': {
                        method: 'POST'
                    },
                    'put': {
                        method: 'PUT'
                    }
                }
            );
        };
    }]
);

/**
 * Listening the directive and add value on the front end.
 * Used to change the user's name on the front end.
 *
 * Example:
 *   <span compile="username">Value</span>
 */
App.directive('compile', ['$compile', function ($compile) {
    return function(scope, element, attrs) {
        scope.$watch(
            function(scope) {
                return scope.$eval(attrs.compile);
            },
            function(value) {
                element.html(value);

                $compile(element.contents())(scope);
            }
        );
    };
}]);

App.config(['$stateProvider', '$urlRouterProvider', '$locationProvider', 'settings', 'routeProvider', 'pathProvider', 'authProvider', '$ocLazyLoadProvider', '$showdownProvider', '$urlMatcherFactoryProvider',
    function ($stateProvider, $urlRouterProvider, $locationProvider, settings, routeProvider, pathProvider, authProvider, $ocLazyLoadProvider, $showdownProvider, $urlMatcherFactoryProvider) {

        /**
         * Config Modules
         *
         * Route, Path, Auth, ocLazyLoad, Location, Showdown
         */
        routeProvider.config(settings.route);

        pathProvider.config(settings.path);

        authProvider.config(settings.auth);

        $ocLazyLoadProvider.config({
            debug: true
        });

        $locationProvider.html5Mode({
            enabled: true,
            requireBase: false
        });

        $showdownProvider.loadExtension('prettify');

        // Url contains a trailing slash or not
        $urlMatcherFactoryProvider.strictMode(false);

        // Namespace used for define route.
        var namespace = '/' + settings.route.namespace;

        $stateProvider
            .state('home', {
                url: "/",
                templateUrl: namespace + "/home",
                data: {
                    title: 'Home',
                    description: 'MEANStack.io bringing together the best of MEAN MongoDB, Express, AngularJS and Node.js'
                },
                controller: "homeController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load({
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('homeController')
                                ]
                            }
                        );
                    }]
                }
            })

            .state('about', {
                url: "/about",
                templateUrl: namespace + "/about",
                data: {title: 'About'},
                controller: "aboutController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load({
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('aboutController')
                                ]
                            }
                        );
                    }]
                }
            })

            .state('releases', {
                url: "/releases/:page",
                params: {
                    page: {
                        value: "1",
                        squash: true
                    }
                },
                templateUrl: namespace + "/release",
                data: {title: 'Releases'},
                controller: "releaseController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load(
                            {
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('releaseController')
                                ]
                            }
                        );
                    }],
                    releasesPageResource: ['releaseService', '$stateParams', function (releaseService, $stateParams) {
                        return releaseService.query(
                            {
                                action: 'load',
                                page: $stateParams.page
                            }
                        ).$promise;
                    }],
                    releasesResource: ['releaseService', function (releaseService) {
                        return releaseService.query(
                            {
                                action: 'load'
                            }
                        ).$promise;
                    }]
                }
            })

            .state('redirectDocs', {
                url: "/docs",
                resolve: {
                    releasesResource: ['releaseService', function (releaseService) {
                        return releaseService.query(
                            {
                                action: 'current'
                            }
                        ).$promise;
                    }]
                },
                controller: ['$state', 'releasesResource', function ($state, releasesResource) {
                    $state.go('docs', {release: releasesResource.data.version});
                }]
            })

            .state('docs', {
                url: "/docs/:release/:doc",
                params: {
                    release: {
                        value: null
                    },
                    doc: {
                        value: null,
                        squash: true
                    }
                },
                templateUrl: namespace + "/docs",
                data: {title: 'Documentation'},
                controller: "documentationController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load(
                            {
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('documentationController')
                                ]
                            }
                        );
                    }],
                    documentationsResource: ['documentationService', '$stateParams', function (documentationService, $stateParams) {
                        return documentationService.query(
                            {
                                action: 'load',
                                release: $stateParams.release,
                                doc: $stateParams.doc
                            }
                        ).$promise;
                    }],
                    releasesResource: ['releaseService', function (releaseService) {
                        return releaseService.query(
                            {
                                action: 'load'
                            }
                        ).$promise;
                    }]
                }
            })

            .state('signin', {
                url: "/admin/signin?r?p",
                templateUrl: namespace + "/admin/signin",
                data: {title: 'Admin - Sign In'},
                controller: "adminSignInController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load({
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/adminSignInController')
                                ]
                            }
                        );
                    }]
                },
                access: {
                    auth: false
                }
            })

            .state('admin', {
                url: "/admin",
                templateUrl: namespace + "/admin",
                data: {title: 'Admin - Dashboard'},
                controller: "adminDashboardController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load({
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/adminDashboardController')
                                ]
                            }
                        );
                    }]
                },
                access: {
                    auth: true,
                    paramOfRedirect: false
                },
                cache: false
            })

            // ======== Release ========
            .state('admin-release', {
                url: "/admin/release",
                templateUrl: namespace + "/admin/release",
                data: {title: 'Admin - Releases'},
                controller: "adminReleaseController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load({
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/release/adminReleaseController')
                                ]
                            }
                        );
                    }]
                },
                access: {
                    auth: true
                }
            })

            .state('admin-release-create', {
                url: "/admin/release/create",
                templateUrl: namespace + "/admin/release/create",
                data: {title: 'Admin - New release'},
                controller: "adminReleaseCreateController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load([
                            {
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/release/adminReleaseCreateController')
                                ]
                            }
                        ]);
                    }]
                },
                access: {
                    auth: true
                }
            })

            .state('admin-release-edit', {
                url: "/admin/release/:release/edit",
                templateUrl: function (params) {
                    return namespace + "/admin/release/" + params.release + "/edit";
                },
                data: {title: 'Admin - Edit release'},
                controller: "adminReleaseEditController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load([
                            {
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/release/adminReleaseEditController')
                                ]
                            }
                        ]);
                    }]
                },
                access: {
                    auth: true
                }
            })

            // ======== Documentation ========
            .state('admin-documentation', {
                url: "/admin/documentation",
                templateUrl: namespace + "/admin/documentation",
                data: {title: 'Admin - Documentations'},
                controller: "adminDocumentationController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load({
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/documentation/adminDocumentationController')
                                ]
                            }
                        );
                    }]
                },
                access: {
                    auth: true
                }
            })

            .state('admin-documentation-create', {
                url: "/admin/documentation/create",
                templateUrl: namespace + "/admin/documentation/create",
                data: {title: 'Admin - New documentation'},
                controller: "adminDocumentationCreateController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load([
                            {
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/documentation/adminDocumentationCreateController')
                                ]
                            }
                        ]);
                    }]
                },
                access: {
                    auth: true
                }
            })

            .state('admin-documentation-edit', {
                url: "/admin/documentation/:documentation/edit",
                templateUrl: function (params) {
                    return namespace + "/admin/documentation/" + params.documentation + "/edit";
                },
                data: {title: 'Admin - Edit documentation'},
                controller: "adminDocumentationEditController",
                resolve: {
                    deps: ['$ocLazyLoad', 'path', function ($ocLazyLoad, path) {
                        return $ocLazyLoad.load([
                            {
                                insertBefore: '#load_js_before',
                                files: [
                                    path.controller('admin/documentation/adminDocumentationEditController')
                                ]
                            }
                        ]);
                    }]
                },
                access: {
                    auth: true
                }
            })

            .state('404', {
                templateUrl: namespace + "/error/404",
                data: {title: 'Page not found'}
            })

            .state('403', {
                templateUrl: namespace + "/error/403",
                data: {title: 'Not authorized'}
            })

            .state('500', {
                templateUrl: namespace + "/error/500",
                data: {title: 'Internal server error'}
            });


        $urlRouterProvider.otherwise(function ($injector, $location) {
            var state = $injector.get('$state');
            state.go('404');
            return $location.path();
        });
    }]);

/**
 *  Init global settings and run the app
 */
App.run(["$rootScope", "settings", "$state", "auth", "path", function ($rootScope, settings, $state, auth, path) {
    // state to be accessed from view
    $rootScope.$state = $state;

    // settings to be accessed from view
    $rootScope.settings = settings;

    // load files in view
    $rootScope.path = path;

    // check auth to be accessed from view
    $rootScope.login = function () {
        return auth.check();
    };

    $rootScope.$on('$stateChangeStart', function (event, toState, toParams, fromState, fromParams) {

        //Route control access
        auth.routeAuth(event, $state, toState, toParams);

    });

    // if error handler go to page error angular.
    $rootScope.$on('$stateChangeError', function(event, toState, toParams, fromState, fromParams, error) {

        event.preventDefault();

        if(error.status === 404) {
            return $state.go('404');
        } else if(error.status === 403) {
            return $state.go('403');
        }

        return $state.go('500');
    });
}]);

App.controller('navbarController', ['$scope', '$rootScope', 'request', '$templateCache', '$state', 'accountService',
    function ($scope, $rootScope, request, $templateCache, $state, accountService) {

    /**
     * Alter username navbar.
     */
    $rootScope.$on('username', function(event, name) {
        $scope.username = name;
    });

    /**
     * Logoff
     */
    $scope.logout = function () {
        accountService.logout();
    };
}]);
