/**
 * Module authentication for AngularJS.
 *
 * Example:
 *
 *   // Auth false
 *   .state('signup', {
 *       url: '/signup',
 *       templateUrl: '/view/login/signup',
 *       ...
 *       // Control access
 *       access: {
 *           // Only access page if not login.
 *           auth: false,
 *           // Optional redirect: URL
 *           redirect: '/myaccount',
 *       }
 *       ...
 *
 *   // Auth true
 *   .state('myaccount', {
 *       url: '/myaccount',
 *       ...
 *       // Control access
 *       access: {
 *           // Only access page if logged.
 *           auth: true,
 *           // Optional setting -> redirect: URL
 *           redirect: 'signin'
 *           // redirect result, domain.com/signin?r=myaccount
 *           // after login will be redirected to myaccount.
 *           // if paramOfRedirect parameter is set to false will not be redirected.
 *           paramOfRedirect: false
 *       }
 *       ...
 *
 * === Config redirect default in file ../config.js ===
 *
 *   ...
 *   'auth': {
 *       redirect: {
 *           notLogged: '/signin',
 *           logged: '/myaccount'
 *           paramOfRedirect: true
 *       }
 *   }
 *   ...
 *
 *
 * === Check user is logged ===
 *
 * return {boolean}
 * service.check()
 *
 */
var AppPath = angular.module('AppAuth', []);

AppPath.provider('auth', [function () {
    var settings_default = {
            auth: {
                redirect: {
                    notLogged: 'signin',
                    paramOfRedirect: true,
                    logged: 'myaccount'
                },
                cookie: {
                    nameFlagLogged: 'login'
                }
            }
        },
        settings = {};

    this.config = function (op_settings) {
        settings = angular.extend(settings_default, op_settings);
    };

    this.$get = ['settings', '$stateParams',
        function (settings, $stateParams) {
            var service = {};

            /**
             * Check user is logged
             *
             * @returns {boolean}
             */
            service.check = function () {
                var cookieName = settings.auth.cookie.nameFlagLogged,
                    getCookieValues = function (cookie) {
                        var cookieArray = cookie.split('=');
                        return cookieArray[1].trim();
                    },
                    getCookieNames = function (cookie) {
                        var cookieArray = cookie.split('=');
                        return cookieArray[0].trim();
                    },
                    cookies = document.cookie.split(';'),
                    cookieValue = cookies.map(getCookieValues)[cookies.map(getCookieNames).indexOf(cookieName)];

                return (((cookieValue === undefined) ? null : cookieValue) === 'true');
            };

            /**
             * Parameters redirection when login.
             * 
             * @returns {{redirect: string, params: {}, defaultRedirect: boolean}}
             */
            service.paramsRedirectionLogin = function () {
                var redirect = (typeof $stateParams.r !== 'undefined') ? decodeURIComponent($stateParams.r) : settings.auth.redirect.logged;

                return {
                    redirect: redirect,
                    params: (typeof $stateParams.p !== 'undefined') ? JSON.parse(decodeURIComponent($stateParams.p)) : {},
                    defaultRedirect: (redirect === settings.auth.redirect.logged)
                };
            };

            /**
             * Route control access
             *
             * @param toState
             */
            service.routeAuth = function (event, $state, toState, toParams) {
                //if route not have control access
                if (typeof toState.access === 'undefined' || typeof toState.access.auth === 'undefined') {
                    return;
                }

                var authRoute = toState.access.auth,
                    redirect = {
                        notLogged: (typeof toState.access.redirect !== 'undefined') ? toState.access.redirect : settings.auth.redirect.notLogged,
                        logged: (typeof toState.access.redirect !== 'undefined') ? toState.access.redirect : settings.auth.redirect.logged
                    };

                //Only access page if logged.
                if (authRoute && !service.check()) {
                    var paramOfRedirect = (typeof toState.access.paramOfRedirect !== 'undefined') ? toState.access.paramOfRedirect : settings.auth.redirect.paramOfRedirect;
                    var params = {};

                    if (paramOfRedirect) {
                        params.r = encodeURIComponent(toState.name);

                        if (Object.keys(toParams).length !== 0) {
                            params.p = encodeURIComponent(JSON.stringify(toParams));
                        }
                    }

                    event.preventDefault();

                    $state.go(redirect.notLogged, params);
                }
                //Only access page if not login.
                else if (!authRoute && service.check()) {

                    event.preventDefault();

                    $state.go(redirect.logged);
                }
            };

            return service;
        }
    ];
}]);
