/* */ 
"format cjs";
import { ListWrapper } from "angular2/src/facade/collection";
export class AST {
    visit(visitor) { return null; }
    toString() { return "AST"; }
}
/**
 * Represents a quoted expression of the form:
 *
 * quote = prefix `:` uninterpretedExpression
 * prefix = identifier
 * uninterpretedExpression = arbitrary string
 *
 * A quoted expression is meant to be pre-processed by an AST transformer that
 * converts it into another AST that no longer contains quoted expressions.
 * It is meant to allow third-party developers to extend Angular template
 * expression language. The `uninterpretedExpression` part of the quote is
 * therefore not interpreted by the Angular's own expression parser.
 */
export class Quote extends AST {
    constructor(prefix, uninterpretedExpression, location) {
        super();
        this.prefix = prefix;
        this.uninterpretedExpression = uninterpretedExpression;
        this.location = location;
    }
    visit(visitor) { return visitor.visitQuote(this); }
    toString() { return "Quote"; }
}
export class EmptyExpr extends AST {
    visit(visitor) {
        // do nothing
    }
}
export class ImplicitReceiver extends AST {
    visit(visitor) { return visitor.visitImplicitReceiver(this); }
}
/**
 * Multiple expressions separated by a semicolon.
 */
export class Chain extends AST {
    constructor(expressions) {
        super();
        this.expressions = expressions;
    }
    visit(visitor) { return visitor.visitChain(this); }
}
export class Conditional extends AST {
    constructor(condition, trueExp, falseExp) {
        super();
        this.condition = condition;
        this.trueExp = trueExp;
        this.falseExp = falseExp;
    }
    visit(visitor) { return visitor.visitConditional(this); }
}
export class PropertyRead extends AST {
    constructor(receiver, name, getter) {
        super();
        this.receiver = receiver;
        this.name = name;
        this.getter = getter;
    }
    visit(visitor) { return visitor.visitPropertyRead(this); }
}
export class PropertyWrite extends AST {
    constructor(receiver, name, setter, value) {
        super();
        this.receiver = receiver;
        this.name = name;
        this.setter = setter;
        this.value = value;
    }
    visit(visitor) { return visitor.visitPropertyWrite(this); }
}
export class SafePropertyRead extends AST {
    constructor(receiver, name, getter) {
        super();
        this.receiver = receiver;
        this.name = name;
        this.getter = getter;
    }
    visit(visitor) { return visitor.visitSafePropertyRead(this); }
}
export class KeyedRead extends AST {
    constructor(obj, key) {
        super();
        this.obj = obj;
        this.key = key;
    }
    visit(visitor) { return visitor.visitKeyedRead(this); }
}
export class KeyedWrite extends AST {
    constructor(obj, key, value) {
        super();
        this.obj = obj;
        this.key = key;
        this.value = value;
    }
    visit(visitor) { return visitor.visitKeyedWrite(this); }
}
export class BindingPipe extends AST {
    constructor(exp, name, args) {
        super();
        this.exp = exp;
        this.name = name;
        this.args = args;
    }
    visit(visitor) { return visitor.visitPipe(this); }
}
export class LiteralPrimitive extends AST {
    constructor(value) {
        super();
        this.value = value;
    }
    visit(visitor) { return visitor.visitLiteralPrimitive(this); }
}
export class LiteralArray extends AST {
    constructor(expressions) {
        super();
        this.expressions = expressions;
    }
    visit(visitor) { return visitor.visitLiteralArray(this); }
}
export class LiteralMap extends AST {
    constructor(keys, values) {
        super();
        this.keys = keys;
        this.values = values;
    }
    visit(visitor) { return visitor.visitLiteralMap(this); }
}
export class Interpolation extends AST {
    constructor(strings, expressions) {
        super();
        this.strings = strings;
        this.expressions = expressions;
    }
    visit(visitor) { return visitor.visitInterpolation(this); }
}
export class Binary extends AST {
    constructor(operation, left, right) {
        super();
        this.operation = operation;
        this.left = left;
        this.right = right;
    }
    visit(visitor) { return visitor.visitBinary(this); }
}
export class PrefixNot extends AST {
    constructor(expression) {
        super();
        this.expression = expression;
    }
    visit(visitor) { return visitor.visitPrefixNot(this); }
}
export class MethodCall extends AST {
    constructor(receiver, name, fn, args) {
        super();
        this.receiver = receiver;
        this.name = name;
        this.fn = fn;
        this.args = args;
    }
    visit(visitor) { return visitor.visitMethodCall(this); }
}
export class SafeMethodCall extends AST {
    constructor(receiver, name, fn, args) {
        super();
        this.receiver = receiver;
        this.name = name;
        this.fn = fn;
        this.args = args;
    }
    visit(visitor) { return visitor.visitSafeMethodCall(this); }
}
export class FunctionCall extends AST {
    constructor(target, args) {
        super();
        this.target = target;
        this.args = args;
    }
    visit(visitor) { return visitor.visitFunctionCall(this); }
}
export class ASTWithSource extends AST {
    constructor(ast, source, location) {
        super();
        this.ast = ast;
        this.source = source;
        this.location = location;
    }
    visit(visitor) { return this.ast.visit(visitor); }
    toString() { return `${this.source} in ${this.location}`; }
}
export class TemplateBinding {
    constructor(key, keyIsVar, name, expression) {
        this.key = key;
        this.keyIsVar = keyIsVar;
        this.name = name;
        this.expression = expression;
    }
}
export class RecursiveAstVisitor {
    visitBinary(ast) {
        ast.left.visit(this);
        ast.right.visit(this);
        return null;
    }
    visitChain(ast) { return this.visitAll(ast.expressions); }
    visitConditional(ast) {
        ast.condition.visit(this);
        ast.trueExp.visit(this);
        ast.falseExp.visit(this);
        return null;
    }
    visitPipe(ast) {
        ast.exp.visit(this);
        this.visitAll(ast.args);
        return null;
    }
    visitFunctionCall(ast) {
        ast.target.visit(this);
        this.visitAll(ast.args);
        return null;
    }
    visitImplicitReceiver(ast) { return null; }
    visitInterpolation(ast) { return this.visitAll(ast.expressions); }
    visitKeyedRead(ast) {
        ast.obj.visit(this);
        ast.key.visit(this);
        return null;
    }
    visitKeyedWrite(ast) {
        ast.obj.visit(this);
        ast.key.visit(this);
        ast.value.visit(this);
        return null;
    }
    visitLiteralArray(ast) { return this.visitAll(ast.expressions); }
    visitLiteralMap(ast) { return this.visitAll(ast.values); }
    visitLiteralPrimitive(ast) { return null; }
    visitMethodCall(ast) {
        ast.receiver.visit(this);
        return this.visitAll(ast.args);
    }
    visitPrefixNot(ast) {
        ast.expression.visit(this);
        return null;
    }
    visitPropertyRead(ast) {
        ast.receiver.visit(this);
        return null;
    }
    visitPropertyWrite(ast) {
        ast.receiver.visit(this);
        ast.value.visit(this);
        return null;
    }
    visitSafePropertyRead(ast) {
        ast.receiver.visit(this);
        return null;
    }
    visitSafeMethodCall(ast) {
        ast.receiver.visit(this);
        return this.visitAll(ast.args);
    }
    visitAll(asts) {
        asts.forEach(ast => ast.visit(this));
        return null;
    }
    visitQuote(ast) { return null; }
}
export class AstTransformer {
    visitImplicitReceiver(ast) { return ast; }
    visitInterpolation(ast) {
        return new Interpolation(ast.strings, this.visitAll(ast.expressions));
    }
    visitLiteralPrimitive(ast) { return new LiteralPrimitive(ast.value); }
    visitPropertyRead(ast) {
        return new PropertyRead(ast.receiver.visit(this), ast.name, ast.getter);
    }
    visitPropertyWrite(ast) {
        return new PropertyWrite(ast.receiver.visit(this), ast.name, ast.setter, ast.value);
    }
    visitSafePropertyRead(ast) {
        return new SafePropertyRead(ast.receiver.visit(this), ast.name, ast.getter);
    }
    visitMethodCall(ast) {
        return new MethodCall(ast.receiver.visit(this), ast.name, ast.fn, this.visitAll(ast.args));
    }
    visitSafeMethodCall(ast) {
        return new SafeMethodCall(ast.receiver.visit(this), ast.name, ast.fn, this.visitAll(ast.args));
    }
    visitFunctionCall(ast) {
        return new FunctionCall(ast.target.visit(this), this.visitAll(ast.args));
    }
    visitLiteralArray(ast) {
        return new LiteralArray(this.visitAll(ast.expressions));
    }
    visitLiteralMap(ast) {
        return new LiteralMap(ast.keys, this.visitAll(ast.values));
    }
    visitBinary(ast) {
        return new Binary(ast.operation, ast.left.visit(this), ast.right.visit(this));
    }
    visitPrefixNot(ast) { return new PrefixNot(ast.expression.visit(this)); }
    visitConditional(ast) {
        return new Conditional(ast.condition.visit(this), ast.trueExp.visit(this), ast.falseExp.visit(this));
    }
    visitPipe(ast) {
        return new BindingPipe(ast.exp.visit(this), ast.name, this.visitAll(ast.args));
    }
    visitKeyedRead(ast) {
        return new KeyedRead(ast.obj.visit(this), ast.key.visit(this));
    }
    visitKeyedWrite(ast) {
        return new KeyedWrite(ast.obj.visit(this), ast.key.visit(this), ast.value.visit(this));
    }
    visitAll(asts) {
        var res = ListWrapper.createFixedSize(asts.length);
        for (var i = 0; i < asts.length; ++i) {
            res[i] = asts[i].visit(this);
        }
        return res;
    }
    visitChain(ast) { return new Chain(this.visitAll(ast.expressions)); }
    visitQuote(ast) {
        return new Quote(ast.prefix, ast.uninterpretedExpression, ast.location);
    }
}
//# sourceMappingURL=data:application/json;base64,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