var ts = require('typescript');
var logger_1 = require('./logger');
var utils_1 = require("./utils");
var compiler_host_1 = require("./compiler-host");
var logger = new logger_1.default({ debug: false });
var TypeChecker = (function () {
    function TypeChecker(host, resolve, fetch) {
        this._host = host;
        this._resolve = resolve;
        this._fetch = fetch;
        this._options = ts.clone(this._host.options);
        this._options.inlineSourceMap = false;
        this._options.sourceMap = false;
        this._options.declaration = false;
        this._options.isolatedModules = false;
        this._options.skipDefaultLibCheck = true;
        this._files = {};
        this._declarationFiles = [];
        this._typings = {};
    }
    TypeChecker.prototype.check = function (sourceName, source) {
        var file = this.registerFile(sourceName);
        this.registerSource(sourceName, source);
        return file.errors;
    };
    TypeChecker.prototype.registerDeclarationFile = function (sourceName, isDefaultLib) {
        var file = this.registerFile(sourceName, isDefaultLib);
        this._declarationFiles.push(file);
    };
    TypeChecker.prototype.registerFile = function (sourceName, isDefaultLib) {
        var _this = this;
        if (isDefaultLib === void 0) { isDefaultLib = false; }
        if (!this._files[sourceName]) {
            var source = new Deferred();
            if (utils_1.isTypescriptDeclaration(sourceName)) {
                this._fetch(sourceName)
                    .then(function (source) {
                    _this._host.addFile(sourceName, source, isDefaultLib);
                    _this.registerSource(sourceName, source);
                })
                    .catch(function (err) {
                    logger.error(err.message);
                });
            }
            var loaded = source.promise
                .then(function (source) { return _this.resolveDependencies(sourceName, source); })
                .then(function (depsMap) {
                _this._host.addResolutionMap(sourceName, depsMap);
                _this._files[sourceName].deps = Object.keys(depsMap)
                    .map(function (key) { return depsMap[key]; })
                    .filter(function (res) { return utils_1.isTypescript(res); })
                    .map(function (res) { return _this.registerFile(res); })
                    .concat(_this._declarationFiles);
            });
            var errors = loaded
                .then(function () { return _this.canEmit(_this._files[sourceName]); })
                .then(function () { return _this.getAllDiagnostics(_this._files[sourceName]); });
            this._files[sourceName] = {
                sourceName: sourceName,
                source: source,
                loaded: loaded,
                errors: errors,
                checked: false,
            };
        }
        return this._files[sourceName];
    };
    TypeChecker.prototype.registerSource = function (sourceName, source) {
        if (!this._files[sourceName])
            throw new Error(sourceName + " has not been registered");
        this._files[sourceName].source.resolve(source);
    };
    TypeChecker.prototype.resolveDependencies = function (sourceName, source) {
        var _this = this;
        var info = ts.preProcessFile(source, true);
        var resolvedReferences = info.referencedFiles
            .map(function (ref) { return _this.resolveReference(ref.fileName, sourceName); });
        var resolvedImports = info.importedFiles
            .map(function (imp) { return _this.resolveImport(imp.fileName, sourceName); });
        var refs = [].concat(info.referencedFiles).concat(info.importedFiles).map(function (pre) { return pre.fileName; });
        var deps = resolvedReferences.concat(resolvedImports);
        return Promise.all(deps)
            .then(function (resolved) {
            return refs.reduce(function (result, ref, idx) {
                result[ref] = resolved[idx];
                return result;
            }, {});
        });
    };
    TypeChecker.prototype.resolveReference = function (referenceName, sourceName) {
        if ((utils_1.isAmbient(referenceName) && !this._options.resolveAmbientRefs) || (referenceName.indexOf("/") === -1))
            referenceName = "./" + referenceName;
        return this._resolve(referenceName, sourceName);
    };
    TypeChecker.prototype.resolveImport = function (importName, sourceName) {
        var _this = this;
        if (utils_1.isRelative(importName) && utils_1.isTypescriptDeclaration(sourceName) && !utils_1.isTypescriptDeclaration(importName))
            importName = importName + ".d.ts";
        return this._resolve(importName, sourceName)
            .then(function (resolvedImport) {
            if (_this._options.resolveTypings && utils_1.isAmbientImport(importName) && utils_1.isJavaScript(resolvedImport) && !utils_1.isTypescriptDeclaration(sourceName)) {
                if (!_this._typings[resolvedImport]) {
                    _this._typings[resolvedImport] = _this.resolveTyping(importName, sourceName)
                        .then(function (resolvedTyping) {
                        return resolvedTyping ? resolvedTyping : resolvedImport;
                    });
                }
                return _this._typings[resolvedImport];
            }
            else {
                return resolvedImport;
            }
        });
    };
    TypeChecker.prototype.resolveTyping = function (importName, sourceName) {
        var _this = this;
        var packageName = importName.split(/\//)[0];
        return this._resolve(packageName, sourceName)
            .then(function (exported) {
            return exported.slice(0, -3) + "/package.json";
        })
            .then(function (address) {
            return _this._fetch(address)
                .then(function (packageText) {
                var typings = JSON.parse(packageText).typings;
                return typings ? _this._resolve("./" + typings, address) : undefined;
            })
                .catch(function (err) {
                logger.warn("unable to resolve typings for " + importName + ", " + address + " could not be found");
                return undefined;
            });
        });
    };
    TypeChecker.prototype.canEmit = function (file, seen) {
        var _this = this;
        seen = seen || [];
        if (seen.indexOf(file) < 0) {
            seen.push(file);
            return file.loaded.then(function () { return Promise.all(file.deps.map(function (dep) { return _this.canEmit(dep, seen); })); });
        }
    };
    TypeChecker.prototype.accumulateDeps = function (file, result) {
        var _this = this;
        result = result || [];
        if (result.indexOf(file) < 0) {
            result.push(file);
            file.deps.forEach(function (dep) { return _this.accumulateDeps(dep, result); });
        }
        return result;
    };
    TypeChecker.prototype.getAllDiagnostics = function (file) {
        var _this = this;
        var deps = this.accumulateDeps(file);
        var filelist = deps.map(function (dep) { return dep.sourceName; }).concat([compiler_host_1.__HTML_MODULE__]);
        var program = ts.createProgram(filelist, this._options, this._host);
        return deps.reduce(function (diags, dep) {
            if (!dep.checked) {
                var sourceFile = _this._host.getSourceFile(dep.sourceName);
                diags = diags
                    .concat(program.getSyntacticDiagnostics(sourceFile))
                    .concat(program.getSemanticDiagnostics(sourceFile));
                dep.checked = true;
            }
            return diags;
        }, program.getGlobalDiagnostics());
    };
    return TypeChecker;
})();
exports.TypeChecker = TypeChecker;
var Deferred = (function () {
    function Deferred() {
        var _this = this;
        this.promise = new Promise(function (resolve, reject) {
            _this.resolve = resolve;
            _this.reject = reject;
        });
    }
    return Deferred;
})();
//# sourceMappingURL=data:application/json;base64,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