var queryDescriptor = require('./base')
  , QueryHub = require('./QueryHub')
  , queryTypes = require('./types')
  , QueryMotifRegistry = require('./QueryMotifRegistry')
  , merge = require('../../util').merge
  , PRIVATE_COLLECTION = require('./types/constants').PRIVATE_COLLECTION
  ;

module.exports = {
  type: 'Store'

, events: {
    init: function (store, opts) {
      store._queryCoordinator = new QueryHub(store);

      // Contains registered query motifs defined via store.query.expose
      var registry = store._queryMotifRegistry = new QueryMotifRegistry;

      /**
       * Exposes a parameterized data set equivalent to the result set of a
       * query generated by the `callback`. The parameterized
       * Behind the scenes, this creates a QueryMotif.
       * Query motifs define a set of data equivalent to the result set of a
       * query, and they provide an easy string name reference to this data. This
       * declares and registers a query motif that will subsequently be available
       * from the Store or Model via Store#_queryMotifRegistry and
       * Model#_queryMotifRegistry respectively.
       *
       * @optional @param {String} ns is the namespace of documents over which to
       * query
       * @param {String} motifName is the name of the query motif
       * @param {Function} callback in which we define the query.
       * callback(queryArgs...) returns a QueryBuilder
       * @return {Object} store.query for method chaining
       * @api public
       */
      store.query.expose = function (ns, motifName, callback) {
        var motifs;
        // store.query.expose(motifs);
        if (arguments.length === 1 && ns.constructor === Object) {
          motifs = ns;

        // store.query.expose(ns, motifs)
        } else if (arguments.length === 2 && motifName.constructor === Object) {
          motifs = motifName;
        }
        if (motifs) {
          for (motifName in motifs) {
            callback = motifs[motifName];
            this.expose(ns, motifName, callback);
          }
        } else {
          registry.add(ns, motifName, callback);
        }
        return this;
      };
    }

  , middleware: function (store, middleware, createMiddleware) {
      middleware.fetchQuery = createMiddleware();
      middleware.fetchQuery.add(function (req, res, next) {
        req.context.guardQuery(req, res, next);
      });
      middleware.fetchQuery.add(function (req, res, next) {
        var query = req.target
          , triplets = [];
        store._fetchAndCompileQueryData(query, {
          each: function (path, datum, ver) {
            triplets.push([path, datum, ver]);
          }
        , done: function (single) {
            res.send(triplets, single);
            next();
          }
        });
      });
    }
  }

, decorate: function (Store) {
    Store.dataDescriptor(queryDescriptor);
  }

, proto: {
    query: function (ns) {
      var store = this;
      return Object.create(this._queryMotifRegistry.queryTupleBuilder(ns), {
        fetch: {value: function (callback) {
          store.fetch(this, callback);
        }}
      , waitFetch: {value: function (callback) {
          store.waitFetch(this, callback);
        }}
      , subscribe: {value: function (callback) {
          store.subscribe(this, callback);
        }}
      , fetchOrCreate: {value: function (newAttrs, callback) {
          store.fetchOrCreate(this, newAttrs, callback);
        }}
      });
    }

    // TODO Test this
  , fetchOrCreate: function (query, newAttrs, callback) {
      var self = this;
      return this.fetch(query, function (err, result) {
        if (err) return callback(err);
        if (typeof newAttrs === 'function') {
          callback = newAttrs;
          newAttrs = null;
        }
        if (typeof resultVal === 'undefined' || resultVal === null) {
          var queryTuple = query.tuple
            , queryJson = self._queryMotifRegistry.queryJSON(queryTuple)
            , ns = queryTuple[0]
            , obj = {}
            , equals = queryJson.equals;
          if (equals) obj = merge(obj, equals);
          if (newAttrs) obj = merge(obj, newAttrs);
          self.add(ns, obj, null, function (err, path, val) {
            callback(err, obj);
          });
        } else {
          callback(err, result);
        }
      });
    }

    /**
     * Fetches data associated with a queryTuple [ns, {queryMotif: queryArgs, ...}].
     *
     * @param {Array} queryTuple represented as
     * [ns, {queryMotif: queryArgs, ...}]
     * @param {Object} opts can have keys:
     * @config {Function} [opts.each] invoked for every matching doc
     * @config {Function} [opts.done] invoked after the query results are
     * fetched and after opts.each has been called on every matching doc.
     * @api private
     */
  , _fetchAndCompileQueryData: function (queryTuple, opts) {
      var eachDatumCb = opts.each
        , done = opts.done
        , queryJson = this._queryMotifRegistry.queryJSON(queryTuple);
      this._fetchQueryData(queryTuple, function (err, result, version) {
        if (err) return done(err);
        var path;
        if (Array.isArray(result)) {
          for (var i = 0, l = result.length; i < l; i++) {
            var doc = result[i];
            path = queryJson.from + '.' + doc.id;
            eachDatumCb(path, doc, version);
          }
        } else if (result) {
          // TODO Replace conditional with polymorphic approach
          if (queryJson.type === 'count') {
            path = PRIVATE_COLLECTION + '.' + queryTuple[3] + '.count';
          } else {
            path = queryJson.from + '.' + result.id;
          }
          eachDatumCb(path, result, version);
        }
        done(queryJson.type === 'findOne' || queryJson.type === 'one');
      });
    }

    /**
     * @param {Array} queryTuple represented as [ns, queryMotif, params...]
     * @param {Function} callback(err, result, version)
     */
  , _fetchQueryData: function (queryTuple, callback) {
      var queryJson = this._queryMotifRegistry.queryJSON(queryTuple);
      // TODO fetch(queryTuple, ...) ?
      this._queryCoordinator.fetch(queryJson, callback);
    }

  }
};
