emitStream = require 'emit-stream'
sinon = require 'sinon'
{EventEmitter} = require 'events'
{BrowserModel: Model} = require '../test/util/model'
transaction = require('../lib/transaction')
expect = require 'expect.js'
hashable = require '../lib/protocol.hashable'
readable = require '../lib/protocol.readable'

describe 'Model fetch', ->
  beforeEach ->
    @model = new Model _clientId: 'x'
    @model.readStream.resume()
    @emitter = @model.broadcaster
    @remoteEmitter = new EventEmitter
    remoteStream = emitStream @remoteEmitter
    remoteStream.pipe @model.writeStream

  afterEach ->
    @emitter.removeAllListeners()
    @remoteEmitter.removeAllListeners()

  describe 'on a path', ->
    describe 'first fetch', ->
      beforeEach ->
        @doc = id: 1, name: 'Bryan', age: 27, _v_: 0

      describe 'minimal result latency', ->
        beforeEach (done) ->
          @model.fetch 'collection.1', (err, @result) =>
            expect(err).to.equal null
            done()
          @remoteEmitter.emit 'ack.fetch',
            docs:
              'collection.1':
                snapshot: @doc
            pointers:
              'collection.1': true

        it 'should callback with a scoped model', ->
          expect(@result).to.be.a Model
          expect(@result.path()).to.equal 'collection.1'

        it 'should initialize the proper documents and versions', ->
          expect(@result.get()).to.eql @doc
          expect(@model.version('collection.1')).to.equal 0

      describe 'non-trivial result latency', ->
        it 'should re-send fetches at intervals until receiving "ack.fetch"', (done) ->
          cb = sinon.spy()
          @emitter.on 'fetch', cb
          @model.fetch 'collection.1'
          setTimeout ->
            expect(cb).to.have.callCount(2)
            done()
          , 600

      describe 'compound fetches', ->
        describe 'who designate same doc, with different whitelist doc fields', ->
          it 'TODO'
        describe 'where one fetch is a subset of another fetch', ->
          it 'TODO'
        describe 'where all fetches designate mutually exclusive docs', ->
          it 'TODO'

    describe 'subsequent fetches', ->
      describe 'whose doc is equivalent to the first fetch, with different whitelist doc field', ->
        it 'should ask the server for both fetches again (to maintain version consistency)', ->
          {id} = @model.fetch 'collection.1.name'
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Bryan'
                  _v_: 0
            pointers:
              'collection.1.name': true

          cb = sinon.spy()
          @emitter.on 'fetch', cb
          {id} = @model.fetch 'collection.1.age'
          expect(cb).to.be.calledWithEql [
            [
              id
              {
                t: 'collection.1.age'
                v: 0
                f: ['name'] # Other fields
              }
            ]
          ]

        it 'should update the result of both fetches, if both modified to a later version', (done) ->
          {id} = @model.fetch 'collection.1.name', (err, $name) =>
            expect($name.get()).to.equal 'Bryan'
            expect(@model.get('collection.1.age')).to.equal undefined
            {id} = @model.fetch 'collection.1.age', (err, $age) ->
              expect($name.get()).to.equal 'Brian'
              expect($age.get()).to.equal 28
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.1':
                  # TODO This should include a snapshot, because of differing
                  # whitelist
                  ops: [
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.name', 'Brian'], ver: 0
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.age', 28], ver: 1
                  ]
              pointers:
                'collection.1.age': true
          expect(@model.get('collection.1.name')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Bryan'
                  _v_: 0
            pointers:
              'collection.1.name': true

      describe 'whose result is a subset of the first fetch', ->
        it 'should ask the server for the first fetch again (to maintain version consistency)', ->
          {id} = @model.fetch 'collection.1'
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Bryan'
                  age: 27
                  _v_: 0
            pointers:
              'collection.1': true

          cb = sinon.spy()
          @emitter.on 'fetch', cb
          {id} = @model.fetch 'collection.1.age'
          expect(cb).to.be.calledWithEql [
            [
              id
              {
                t: 'collection.1.age'
                o: 'collection.1' # 'o' for override
                v: 0
              }
            ]
          ]

        it 'should provide a result for the subsequent fetch', (done) ->
          {id} = @model.fetch 'collection.1', (err, $doc) =>
            expect($doc.get()).to.eql
              id: 1
              name: 'Bryan'
              age: 27
            {id} = @model.fetch 'collection.1.age', (err, $age) =>
              expect($age.get()).to.equal 28
              expect(@model.get('collection.1.age')).to.equal 28
              expect($doc.version()).to.equal 2
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.1':
                  ops: [
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.name', 'Brian'], ver: 0
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.age', 28], ver: 1
                  ]
              pointers:
                'collection.1.age': true
          expect(@model.get('collection.1')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Bryan'
                  age: 27
                  _v_: 0
            pointers:
              'collection.1': true

        it 'should update the result of the first fetch, if it was modified to a later version', (done) ->
          {id} = @model.fetch 'collection.1', (err, $doc) =>
            expect($doc.get()).to.eql
              id: 1
              name: 'Bryan'
              age: 27
            {id} = @model.fetch 'collection.1.age', (err, $age) =>
              expect($doc.get()).to.eql
                id: 1
                name: 'Brian'
                age: 28
              expect($doc.version()).to.equal 2
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.1':
                  ops: [
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.name', 'Brian'], ver: 0
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.age', 28], ver: 1
                  ]
              pointers:
                'collection.1.age': true
          expect(@model.get('collection.1')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Bryan'
                  age: 27
                  _v_: 0
            pointers:
              'collection.1': true

      describe 'whose docs are a superset of the first fetch', ->
        it 'should do only the recent fetch', ->
          {id} = @model.fetch 'collection.1.age'
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  age: 27
                  _v_: 0
            pointers:
              'collection.1.age': true

          cb = sinon.spy()
          @emitter.on 'fetch', cb
          {id} = @model.fetch 'collection.1'
          expect(cb).to.be.calledWithEql [
            [
              id
              {
                t: 'collection.1'
                v: 0
              }
            ]
          ]

        it 'should update the first fetch results based on the recent fetch results', (done) ->
          {id} = @model.fetch 'collection.1.age', (err, $age) =>
            expect(@model.get('collection.1.age')).to.equal 27
            expect($age.get()).to.equal 27
            {id} = @model.fetch 'collection.1', (err, $doc) =>
              expect($age.get()).to.equal 28
              expect(@model.get('collection.1.age')).to.equal 28
              expect($age.version()).to.equal 2
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.1':
                  snapshot:
                    id: 1
                    name: 'Brian'
                    age: 28
                    _v_: 2
                  ops: [
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.name', 'Brian'], ver: 0
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.age', 28], ver: 1
                  ]
              pointers:
                'collection.1': true
          expect(@model.get('collection.1.age')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  age: 27
                  _v_: 0
            pointers:
              'collection.1.age': true

        it 'should provide a result for the recent fetch', (done) ->
          {id} = @model.fetch 'collection.1.age', (err, $age) =>
            expect(@model.get('collection.1')).to.eql
              id: 1
              age: 27
            {id} = @model.fetch 'collection.1', (err, $doc) =>
              expectedDoc = id: 1, name: 'Brian', age: 28
              expect(@model.get('collection.1')).to.eql expectedDoc
              expect($doc.get()).to.eql expectedDoc
              expect($doc.version()).to.equal 2
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.1':
                  snapshot:
                    id: 1
                    name: 'Brian'
                    age: 28
                    _v_: 2
                  ops: [
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.name', 'Brian'], ver: 0
                    transaction.create id: 'other-client.1', method: 'set', args: ['collection.1.age', 28], ver: 1
                  ]
              pointers:
                'collection.1': true
          expect(@model.get('collection.1')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  age: 27
                  _v_: 0
            pointers:
              'collection.1.age': true

      describe 'whose doc is mutually exclusive from the first fetch', ->
        it 'should do only the recent fetch', ->
          {id} = @model.fetch 'collection.1'
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Brian'
                  _v_: 0
            pointers:
              'collection.1': true

          cb = sinon.spy()
          @emitter.on 'fetch', cb
          {id} = @model.fetch 'collection.2'
          expect(cb).to.be.calledWithEql [
            [
              id
              {t: 'collection.2'}
            ]
          ]

        it 'should not update the first fetch results', (done) ->
          {id} = @model.fetch 'collection.1', (err, $docOne) =>
            expectedOne =
              id: 1
              name: 'Brian'
            expect(@model.get('collection.1')).to.eql expectedOne
            expect($docOne.get()).to.eql expectedOne
            {id} = @model.fetch 'collection.2', (err, $docTwo) =>
              expect(@model.get('collection.1')).to.eql expectedOne
              expect($docOne.get()).to.eql expectedOne
              expect($docOne.version()).to.equal 0
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.2':
                  snapshot:
                    id: 2
                    name: 'Nate'
                    _v_: 1
              pointers:
                'collection.2': true
          expect(@model.get('collection.1')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Brian'
                  _v_: 0
            pointers:
              'collection.1': true

        it 'should provide a result for the recent fetch', (done) ->
          {id} = @model.fetch 'collection.1', (err, $docOne) =>
            expect(@model.get('collection.2')).to.equal undefined
            {id} = @model.fetch 'collection.2', (err, $docTwo) =>
              expectedTwo =
                id: 2
                name: 'Nate'
              expect(@model.get('collection.2')).to.eql expectedTwo
              expect($docTwo.get()).to.eql expectedTwo
              expect($docTwo.version()).to.equal 1
              done()
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.2':
                  snapshot:
                    id: 2
                    name: 'Nate'
                    _v_: 1
              pointers:
                'collection.2': true
          expect(@model.get('collection.2')).to.equal undefined
          @remoteEmitter.emit 'ack.fetch',
            id: id
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Brian'
                  _v_: 0
            pointers:
              'collection.1': true

  describe 'acks', ->
    describe 'subsequent fetches', ->
      describe 'without an overlapping subscription', ->
        describe 'when subsequent result includes a later version of a prior doc', ->
          describe 'and there are no field scope differences', ->
            it 'should transform (against local ops) and apply incoming ops'
            it 'should apply incoming ops without xf (if no local ops)', ->
              @model.fetch 'collection.1', (err, $docA) =>
                expect(err).to.equal null
                expect($docA.get()).to.eql {id: 1, name: 'Bryan'}
                expect($docA.version()).to.equal 0
                @model.fetch 'collection.1', (err, $docB) =>
                  expect(err).to.equal null
                  expect($docB.get()).to.eql {id: 1, name: 'Brian'}
                  expect($docB.version()).to.equal 1
                @remoteEmitter.emit 'ack.fetch',
                  docs:
                    'collection.1':
                      ops: [
                        transaction.create id: 'other-client.1', ver: 0, method: 'set', args: ['collection.1.name', 'Brian']
                      ]
                  pointers:
                    'collection.1': true
              @remoteEmitter.emit 'ack.fetch',
                docs:
                  'collection.1':
                    snapshot:
                      id: 1
                      name: 'Bryan'
                      _v_: 0
                pointers:
                  'collection.1': true

          describe 'and there are field scope differences', ->
            it "should serially (1) xf local ops against remote ops, (2) apply the xf'ed local ops to incoming snapshot, (3) set to result of (2)"

    describe 'duplicate fetches', ->
      describe 'without an overlapping subscription', ->
        it 'should include document version data in its outgoing message', (done) ->
          cb = sinon.spy()
          {id: idOne} = @model.fetch 'collection.1', (err, $docA) =>
            @emitter.on 'fetch', (msg) ->
              # nextTick in order to have access to idTwo via fetch returning
              process.nextTick ->
                cb()
                expect(msg).to.eql [
                  idTwo
                  {
                    t: 'collection.1'
                    v: 0
                  }
                ]
                expect(cb).to.be.calledOnce()
                done()
            {id: idTwo} = @model.fetch 'collection.1'
          @remoteEmitter.emit 'ack.fetch',
            id: idOne
            docs:
              'collection.1':
                snapshot:
                  id: 1
                  name: 'Bryan'
                  _v_: 0
            pointers:
              'collection.1': true

      describe 'with an overlapping subscription', ->
        it 'TODO'

  describe 'on a pattern', ->
    describe '*', ->
      describe 'first fetch', ->
        beforeEach (done) ->
          docOne =
            id: 1
            name: 'Brian'
            _v_: 0
          docTwo =
            id: 2
            name: 'Nate'
            _v_: 1

          @model.fetch '*', (err, @result) =>
            expect(err).to.equal null
            done()
          @remoteEmitter.emit 'ack.fetch',
            docs:
              'collection.1':
                snapshot: docOne
              'collection.2':
                snapshot: docTwo
            pointers:
              '*': true

        it 'should callback with a scoped model', ->
          expect(@result).to.be.a Model
          expect(@result.path()).to.equal ''

        it 'should initialize the proper documents and versions', ->
          expect(@result.get()).to.eql
            collection:
              1:
                id: 1
                name: 'Brian'
              2:
                id: 2
                name: 'Nate'
          expect(@model.get('collection.1')).to.eql
            id: 1
            name: 'Brian'
          expect(@model.get('collection.2')).to.eql
            id: 2
            name: 'Nate'
          expect(@model.version('collection.1')).to.equal 0
          expect(@model.version('collection.2')).to.equal 1

    describe 'collection.*...', ->
      describe 'first fetch', ->
        beforeEach (done) ->
          docOne =
            id: 1
            name: 'Brian'
            _v_: 0
          docTwo =
            id: 2
            name: 'Nate'
            _v_: 1

          @model.fetch 'collection.*.name', (err, @result) =>
            expect(err).to.equal null
            done()
          @remoteEmitter.emit 'ack.fetch',
            docs:
              'collection.1':
                snapshot: docOne
              'collection.2':
                snapshot: docTwo
            pointers:
              'collection.*.name': true

        it 'should callback with a scoped model', ->
          expect(@result).to.be.a Model
          expect(@result.path()).to.equal 'collection'

        it 'should initialize the proper documents and versions', ->
          expect(@result.get()).to.eql
            1:
              id: 1
              name: 'Brian'
            2:
              id: 2
              name: 'Nate'
          expect(@model.get('collection.1')).to.eql
            id: 1
            name: 'Brian'
          expect(@model.get('collection.2')).to.eql
            id: 2
            name: 'Nate'
          expect(@model.version('collection.1')).to.equal 0
          expect(@model.version('collection.2')).to.equal 1

      describe 'subsequent fetches', ->
        describe 'with same * position, with different trailing doc field', ->
          it 'should ask the server for both fetches again (to maintain version consistency)', ->
            {id} = @model.fetch 'collection.*.name'
            @remoteEmitter.emit 'ack.fetch',
              id: id
              docs:
                'collection.1':
                  snapshot:
                    id: 1
                    name: 'Bryan'
                    _v_: 0
              pointers:
                'collection.*.name': true

            cb = sinon.spy()
            @emitter.on 'fetch', cb
            {id} = @model.fetch 'collection.*.age'
            expect(cb).to.be.calledWithEql [
              [
                id
                {
                  t: 'collection.*.age'
                  v: # version info here is a hash from ids to versions, because * matches many ids
                    1: 0
                  f: ['name'] # Other fields
                }
              ]
            ]

        describe 'with later * position', ->
          it 'TODO'
        describe 'with earlier * position', ->
          it 'TODO'

    describe 'collection.x...*...', ->
      describe 'first fetch', ->
        beforeEach (done) ->
          doc =
            id: 1
            pets: [
              {name: 'Banana'}
              {name: 'Squeak'}
            ]
            _v_: 0

          @model.fetch 'collection.1.pets.*.name', (err, @result) =>
            expect(err).to.equal null
            done()
          @remoteEmitter.emit 'ack.fetch',
            docs:
              'collection.1':
                snapshot: doc
            pointers:
              'collection.1.pets.*.name': true

        it 'should callback with a scoped model', ->
          expect(@result).to.be.a Model
          expect(@result.path()).to.equal 'collection.1.pets'

        it 'should initialize the proper documents and versions', ->
          expect(@result.get()).to.eql [
            {name: 'Banana'}
            {name: 'Squeak'}
          ]
          expect(@model.get('collection.1')).to.eql
            id: 1
            pets: [
              {name: 'Banana'}
              {name: 'Squeak'}
            ]
          expect(@model.version('collection.1')).to.equal 0

      describe 'subsequent fetches', ->
        describe 'with same * position and same prefix, with different trailing doc field', ->
          it 'TODO'

        describe 'with same * position and overlapping prefixes', ->
          it 'TODO'

        describe 'with same * position and non-overlapping prefixes', ->
          it 'TODO'

        describe 'with later * position and overlapping prefixes', ->
          it 'TODO'

        describe 'with later * position and non-overlapping prefixes', ->
          it 'TODO'

        describe 'with earlier * position and overlapping prefixes', ->
          it 'TODO'

        describe 'with earlier * position and non-overlapping prefixes', ->
          it 'TODO'

  describe 'on a query', ->
    describe 'first fetch', ->
      describe 'minimal result latency xxx', ->
        beforeEach (done) ->
          registry = @model._queryMotifRegistry
          registry.add 'collection', 'olderThan', (age) ->
            @where('age').gt(age)
          @query = @model.query('collection').olderThan(21)
          @model.fetch @query, (err, @result) =>
            expect(err).to.equal null
            done()
          pointers = {}
          pointers[hashable.hash(@query)] =
            ns: 'collection'
            ids: [1, 4]
          @remoteEmitter.emit 'ack.fetch',
            docs:
              'collection.1':
                snapshot: @docOne = {id: 1, age: 22, _v_: 0}
              'collection.4':
                snapshot: @docTwo = {id: 4, age: 30, _v_: 5}
            pointers: pointers

        it 'should callback with a scoped model', ->
          expect(@result).to.be.a Model
          expect(@result.path()).to.equal readable.resultPath(@query, @model)

        it 'should initialize the proper documents and versions', ->
          expect(@result.get()).to.eql [@docOne, @docTwo]
          expect(@model.get('collection.1')).to.eql {id: 1, age: 22}
          expect(@model.get('collection.4')).to.eql {id: 4, age: 30}
          expect(@model.version('collection.1')).to.equal 0
          expect(@model.version('collection.4')).to.equal 5

      describe 'non-trivial result latency', ->
        it 'should re-send fetches at intervals until receiving "ack.fetch"'

    describe 'compound fetches', ->
      it 'TODO'

    describe 'subsequent fetches', ->
      it 'TODO'
