# DONE Be aware of using paths relative to the main doc result vs ones relative
#      to a data field's immediate parent schema
# DONE After receiving a db query result, figure out an efficient way to match
#      the results to the appropriate data fields
# DONE Find an efficient and unobtrusive way to lookup a query of a data field.
# DONE Improve DbQuery
# DONE What about handling split property lookup for Ref's own fields btwn 2
#      data sources and assigning values to the logical result.
# TODO An approach that elegantly handles smart query building for fallback reads
#      and Ref-family fields
# TODO Add in multiple logical results that trigger back to the main logical result
# TODO What about expanding a logical relationship that maps to 2 different
#      refs, each containing somewhat different sets of properties
# TODO Work on unified data field + logical field + data type + logical type
#      implementation
# TODO What about scenario where a dataField is used in 2 different data
#      schemas (e.g., re-using a data schema as an embedded doc and a ref)
# TODO Message passing along a directed graph sounds like OOP
# TODO find vs findOne when using onUpstream and emitDownstream

# Root queries can be their own nodes

LogicalQuery::fire = (logicalQueryCallback) ->
  logicalResult = new FindOneResult
    onFound: (result) -> logicalQueryCallback null, result
    onNotFound: -> logicalQueryCallback null

  queryGraph = depGraphForQuery @, LogicalSchema.depGraph
  currDataFields = queryGraph.rootNodes()

  # APPROACH 1 - Fire the root queries in parallel
  queryPartitions = partitionFieldsByDbQuery currDataFields
  for dFields in queryPartitions
    query = buildDbQuery @, dFields, logicalResult, queryGraph
    query.fire()
  return logicalResult

  # APPROACH 2 - Construct all queries before-hand
  queryPartitionsByPhase = []
  while currDataFields
    # Partitions currDataFields [a, b, c, d, e]
    # to queryPartitions, e.g., [[a, b], [c], [d, e]]
    queryPartitions = partitionFieldsByDbQuery currDataFields
    queryPartitionsByPhase.push queryPartitions
    deps = collectDependents currDataFields, queryGraph, @
    currDepEdges = if deps?.length then deps else null
    currDataFields = (dataField for [dataField] in currDepEdges)
  queriesByPhase = []
  for queryPartitions in queryPartitionsByPhase
    queriesByPhase.push(parallelQueries = [])
    for dFields in queryPartitions
      [dField_0, edgeVal] = dFields[0]
      continue if edgeVal.query # If a virtual parent node assigned us a query
      if dFields.length == 1 && dField_0.queryForChildren
        dField_0 = dFields[0]
        if buildDbQueryForChildren = dField_0.queryForChildren
          childEdges = queryGraph.edgesFrom dField_0
          childrenFields = (childDataField for [childDataField] in childEdges)
          query = buildDbQueryForChildren childrenFields, logicalResult, logicalQueryCallback
          dFields = childrenFields
      else
        query = buildDbQuery @, dFields, logicalResult
      addQueryToGraph query, queryGraph, dFields
      return parallelQueries.push query
  q.fire() for q in queriesByPhase[0]
  return logicalResult

# Takes the dependency graph of data fields defined by your Logical and Data
# Schema setups, and returns a new graph filtered down to the relevant data
# fields and dependencies dictated by the incoming query.
depGraphForQuery = (query, depGraph) ->
  {depGraph} = LogicalSchema
  queryGraph = new Graph
  logicalPathsToQuery = @_selects
  for path in logicalPathsToQuery
    parts = path.split '.'
    lastPropIndex = parts.length-1
    CurrSkema = @schema
    for prop, i in parts
      # TODO What about logical fields defined in an embedded, nested way?
      {dataFields} = logicalField = CurrSkema.fields[prop]
      CurrSkema = type if (type = logicalField.type) instanceof LogicalSchema
      doesTriggerAssignment = (i == lastPropIndex)
      for dField in dataFields
        continue unless dField.shouldBeInQuery @
        queryGraph.addNode dField
        dependencies = depGraph.edgesTo   dField
        dependents   = depGraph.edgesFrom dField
        for [depField, config] in dependencies
          continue unless queryGraph.containsNode depField
          continue if config.shouldBeRoot? dField, @
          queryGraph.addEdge depField, dField, Object.create config
        for [depField, config] in dependents
          continue unless queryGraph.containsNode depField
          continue if config.shouldBeRoot? depField, @
          queryGraph.addEdge dField, depField, Object.create config
        if doesTriggerAssignment
          logicalResult.fullPath dField, path
          logicalResult.incrNumPotentialFetches path
  return queryGraph

# Partitions fields by whether they belong in the same query or not
partitionFieldsByDbQuery = (dFields) ->
  partitions = []
  for dField in dFields
    assignedToPartition = false
    for partition in partitions
      dFieldX = partition[0]
      if shouldBePartOfSameDbCommand dField, dFieldX
        partition.push [dField]
        assignedToPartition = true
        break
    partitions.push [[dField]] unless assignedToPartition
  return partitions

shouldBePartOfSameDbCommand fieldA, fieldB
  return false if fieldA.initQuery || fieldB.initQuery
  return fieldA.schema.containsField fieldB

collectDependents = (dFields, depGraph, query) ->
  deps = []
  for dField in dFields
    dependencies = depGraph.edgesFrom dField
    for depField in dependencies
      deps.push depField if -1 == deps.indexOf depField
  return deps


# TODO ? More generically wrap a real data query with db query. Generalize
#      DbQuery so it works across more databases with less impedance of assumptions.
buildDbQuery = (logicalQuery, dataFields, logicalResult, callback) ->
  if dataFields[0].initQuery logicalQuery
    # TODO Create a query based on the nested data fields which are now
    #      immediate. Are data fields associated with a ref result necessarily
    #      children of the Ref data field? Or does a Ref data field implicitly
    #      connect what would have been its parent to its children and keep
    #      itself out of the dependency graph?
    #      Another scenario is if the ref returns an object with another ref
    #      that needs de-referencing
    #      Scenario - select 'author'
    #      Scenario - select 'author.name', 'author.birthday'
    #      What happens if `onUpstream` is meant only for one parent, but
    #      the data field actually has multiple parents - e.g., it's a fallback
    #      but also potentially the first choice in a ref query
    #
    #      Fields get assigned to the schema structure. Its other role is to
    #      manipulate the dependency graph, which is used for query building
    #      and triggering.
    #  Options:
    #  - Do not include the Ref (virtual) data field in the dep graph
    query = dataFields[0].initQuery()
    query._logicalResult = logicalResult
    query._dataFields = dataFields
  else
    {source, ns, queryMethod, conditions} = logicalQuery
    return new DbQuery source, queryMethod, conditions, dataFields, logicalResult, callback

addQueryToGraph = (query, graph, dataFields) ->
  for field in dataFields
    edges = graph.edgesTo field
    conf.query = query for [_, conf] in edges
  return

LogicalSchema.fallbackRead fieldA, fieldB
# This would setup a dependency edge between the 2 fields, adding an onUpstream fn for the edge
LogicalSchema.fallbackRead = (fieldA, fieldB) ->
  {depGraph} = LogicalSchema
  depGraph.addEdge fieldA, fieldB,
    onUpstream: (value, queryWithMe) ->
      return queryWithMe.shouldntFetch fieldB.path if value
      return queryWithMe.shouldFetch fieldA.path

dataField.onFound = (value, logicalResult) ->
  dependents = queryGraph.edgesFrom @
  for [dField, {onUpstream}] in dependents
    onUpstream value

  NORM_PATH = ??
  return logicalResult.assign NORM_PATH, @uncast value if value isnt undefined
  return logicalResult.notFound NORM_PATH



# Do we need to handle fields that represent documents or arrays of documents
# differently?
#
# - If the data field is an embedded doc:
#   - It will be assigned to the logicalResult properly
#
# - If the data field is a Ref
#   - It generates its own db query. This is used to create its own query
#     after the field is placed into its own partition.
#   - onUpstream takes the ObjectId and uses it to modify the query it
#     needs to make
#
 virtuals =
   blog:
     logicalFieldPath: 'blog'
     query: dependsOn 'blogId',
       replaceNodeWith: (logicalQuery) ->
       initQuery: (logicalQuery) ->
         # Could be useful for one-off redis queries
         # TODO Build a new DbQuery based on dependencies
       onUpstream:
         dependency: (blogId, queryForMe) ->
           queryForMe.conditions['blogId'] = blogId
           queryForMe.fire()
       childFields:
         onUpstream:

virtuals =
  blog:
    logicalFieldPath: 'blog'
    decorateReadDepGraph: (readDepGraph) ->
      # Any way to do this in a more intuitively readable way using query
      # syntax?
      blogFields = BlogDataSchema.fields
      readDepGraph.addEdge userBlogIdDataField, virtualField,
        ignore: true # ignore me as a data field for the query
        queryForChildren: (childrenDataFields, logicalResult, onError) ->
          query = buildDbQuery {queryMethod: 'findOne'}, blogFields, logicalResult, onError
        onUpstream: (blogId, queryForMe) ->
          queryForMe.conditions['blogId'] = blogId
          queryForMe.fire()
      for dataField in blogFields
        do (dataField) ->
          readDepGraph.addEdge userBlogIdDataField, dataField,
            onUpstream: (blogId, queryForMe) ->
              queryForMe.conditions['blogId'] = blogId
              queryForMe.shouldFetch dataField.path
            queryToClone: query
    descorateSchema: (schema, path) ->
      schema.virtuals[path] =


# WHAT IF INSTEAD!!!!
virtuals =
  blog:
    onUpstream: ({blogId, queryForMe, queryGraph}, myNode, upstreamNode, edgeVal) ->
      # Pair this with sendDownstream {blogId, query}
      query = buildDbQuery {queryMethod: 'findOne'}, blogFields, logicalResult, onError
      queryGraph.sendDownstream myNode, query
      @sendDownstream

# Perhaps all data field configs strictly just:
# - Decorate the dpendency graphs
# - Decorate the data schema

# What if the data field is the field of an embedded doc -- e.g., 'pet.name' ?
# Or what if this can be stored in multiple data fields?
# - e.g., in an embedded doc and in a ref

# IDEAS:
# - DataField should contain method `shouldBeInQuery(logicalQuery)`
# - Edges may contain fn `shouldBeRoot` which forces the field to be included
#   in phase 0
# - `onUpstream` of an edge can implicitly define the chain. You can think
#   of them as links in the chain that determine what to do next

# PAGINATION SCENARIO HANDLING (inside LogicalQuery::fire)
# `shouldBeInQuery` takes care of some pagination scenarios
# such as if dFieldA contains 0, 10 and dFieldB contains 0...infinity but
# we only want to find 0..5. Then dFieldB is irrelevant.
# BUT what about pagination scenario of find 0, 10. dFieldA may contain
# 0,10 but if it fetches [], then we want to fetch dFieldB which definitely
# contains 0...infinity. In this case, we would specify that dFieldB is a
# fallback to dFieldA.
# In the case where we are fetching 5 to 15, then `shouldBeRoot` would move
# dFieldB to a root node, so it's in the same phase as dFieldA


# Query with fields that can have one of 3 states:
# 0. Unassigned (default)
# 1. shouldFetch
# 2. shouldntFetch
#
# When all fields have been assigned shouldFetch or shouldntFetch, then compile
# and run the query.
# Or, we can just fire the query via query.fire()
DbQuery = (@source, @queryMethod, @conditions, @_dataFields, @_logicalResult, onError) ->
  @_fieldReadyPromises = {}
  for {path} in @_dataFields
    @_fieldReadyPromises[path] = new Promise
  Promise.parallel @_fieldReadyPromises, (err) =>
    return onError err if err
    @fire()
  @_resultPromise = new Promise errback: onError
  return

DbQuery::=
  shouldFetch: (path) ->
    @_fieldReadyPromises[path].fulfill true
  shouldntFetch: (path) ->
    for dField in @_dataFields
      dFieldToRemove = dField if dField.path == path
      break
    throw new Error "Trying to mark a field #{path} not present" unless dFieldToRemove
    @_dataFields.splice @_dataFields.indexOf(dFieldToRemove), 1
    @_fieldReadyPromises[path].fulfill false
  fire: (callback) ->
    @_resultPromise.bothback callback if callback
    @source.adapter[@queryMethod] @ns, @conditions, selects: @_selects, (err, json) =>
      @_resultPromise.fulfill err, json, @_dataFields, @_logicalResult
    return @_resultPromise

  callback: (callback) -> @_resultPromise.bothback callback

FindOneQuery = (@source, @conditions, @_dataFields, @_logicalResult, errback) ->
  DbQuery.call @, @source, 'findOne', @conditions, @_dataFields, @_logicalResult, errback
  @callback @_onFindOne

FindOneQuery:: = merge DbQuery::,
  # When a query is done:
  # - If the dataField was found:
  #   - Uncast and assign the value to the logicalField
  #   - Potentially write fn(foundValue) to the db or another db
  # - For any dataField dependents, use a predicate to see if the dependent
  #   requires fetching (conditional reads) - e.g., predicate could be
  #   named `onUpstream`
  #   - If it does, then tell the dependent's associated query that it
  #     should read it
  #   - If not, tell the dependent's associated query that it shouldn't read it
  #   - When all of an awaiting query's paths have been told whether they should or
  #     should not be fetched, then fire off the query.
  #
  # Something like this needs to be invoked for every upstream and downstream
  # query
  _onFindOne: (err, found, dataFields, logicalResult) ->
    return if err

    if found isnt undefined then for path, val of found
      # ? Maybe move this into an `onFoundValue` per dField config
      field = dataFields[path]
      fullPath = logicalResult.fullPath field
      logicalResult.assign fullPath, field.uncast val
    else for field in dataFields
      fullPath = logicalResult.fullPath field
      logicalResult.notFound fullPath

    # Notify dependents of my value
    for dataField in dataFields
      value = found?[dataField.path]
      dependents = queryGraph.edgesFrom dataField
      for [dField, {onUpstream, queryWithMe}] in dependents
        sendDownstream {value, queryWithMe}
        onUpstream value, queryWithMe

FindQuery = (@source, @conditions, @_dataFields, @_logicalResult, errback) ->
  DbQuery.call @, @source, 'find', @conditions, @_dataFields, @_logicalResult, errback
  @callback @_onFind

FindQuery::= merge DbQuery::,
  _onFind: (err, found, dataFields, logicalResult) ->
    return if err

    if found.length then for doc in found

# Setting a result to null
# - If all paths expected by a result register via
#   `logicalResult.notFound(path)`, then the result should emit null or
#   undefined

# What about detecting when a logical field's data fields have all been
# attempted for fetching?
# - We could associate the number of data fields with the path in logicalResult. When
#   `logicalResult.notFound(path)` is invoked those N times, then the
#   logicalResult registers that path as not being found anywhere.
LogicalResult = ({@onFound, @onNotFound}) ->
  @expectPaths = {}
  @result = {}
  @_fullPaths = {} # Maps data field hashes to the full path relative to the query
  return

LogicalResult:: =
  onFound: (result) -> throw new Error 'Define onFound'
  onNotFound: -> throw new Error 'Define onNotFound'

  # fullPath is relative to the query. A data field by itself only knows its
  # path relative to its immediate parent data schema. This enables temporary
  # assignment and lookup of fullPath to a data field
  fullPath: (dataField, fullPath) ->
    if arguments.length == 2
      return @_fullPaths[dataField.hash()] = fullPath
    return @_fullPaths[dataField.hash()]

  incrNumPotentialFetches: (logicalPath) ->
    (@expectPaths[logicalPath] ||= 0)++

  assign: (path, val) ->
    delete @expectPaths[path]
    nestedAssign path, val, @result
    @emitResult() if @isResultReady()

  didNotFind: (path) ->
    delete @expectPaths[path] if --@expectPaths[path] == 0
    @emitResult() if @isResultReady()

  isResultReady: ->
    for k of @expectPaths
      return false
    return true

  emitResult: ->
    return @onNotFound() unless Object.keys(@result).length
    @onFound @result

nestedAssign = (path, val, target) ->
  parts = path.split '.'
  lastPartIndex = parts.length-1
  for prop, i in parts
    if i == lastPartIndex
      target[prop] = val
    else
      target = target[prop] ||= {}
