{expect} = require './util'
Graph = require '../lib/Schema/Graph'

describe 'Graph', ->
  Node = (@id) -> return
  Node::hash = -> @id
  describe '#containsNode', ->
    it 'should return true if the node was added', ->
      g = new Graph
      node = g.addNode new Node 1
      expect(g.containsNode(node)).to.be(true)

    it 'should return false if the node was never added', ->
      g = new Graph
      node = new Node 1
      expect(g.containsNode(node)).to.be(false)

  describe '#nodesPointedToBy', ->
    it 'should return all nodes pointed to by the given node', ->
      g = new Graph
      from = new Node 'from'
      toA = new Node 'toA'
      toB = new Node 'toB'
      g.addNode from
      g.addNode toA
      g.addNode toB
      g.addEdge from, toA
      expect(g.nodesPointedToBy(from)).to.eql [toA]
      g.addEdge from, toB
      expect(g.nodesPointedToBy(from)).to.eql [toA, toB]

    it 'should not include irrelevant edges', ->
      g = new Graph
      from = new Node 'from'
      to = new Node 'toA'
      g.addNode from
      g.addNode to
      g.addEdge from, to
      expect(g.nodesPointedToBy(to)).to.be.empty

  describe '#nodesPointingTo', ->
    it 'should return all nodes pointing to the given node', ->
      g = new Graph
      to = new Node 'to'
      fromA = new Node 'fromA'
      fromB = new Node 'fromB'
      g.addNode to
      g.addNode fromA
      g.addNode fromB
      g.addEdge fromA, to
      expect(g.nodesPointingTo(to)).to.eql [fromA]
      g.addEdge fromB, to
      expect(g.nodesPointingTo(to)).to.eql [fromA, fromB]

  describe '#filter', ->
    it 'should return a new graph containing only filtered nodes', ->
      g = new Graph
      toA = new Node 'keepMe1'
      toB = new Node 'dontKeepMe1'
      fromA = new Node 'keepMe2'
      fromB = new Node 'dontKeepMe2'
      g.addNode toA
      g.addNode toB
      g.addNode fromA
      g.addNode fromB

      fg = g.filter (node) ->
        return /^keepMe/.test node.hash()

      expect(fg.containsNode(toA)).to.be(true)
      expect(fg.containsNode(toB)).to.be(false)
      expect(fg.containsNode(fromA)).to.be(true)
      expect(fg.containsNode(fromB)).to.be(false)

    it 'should return a new graph containing only edges containing the filtered nodes', ->
      g = new Graph
      toA = new Node 'keepMe1'
      toB = new Node 'dontKeepMe1'
      fromA = new Node 'keepMe2'
      fromB = new Node 'dontKeepMe2'
      g.addNode toA
      g.addNode toB
      g.addNode fromA
      g.addNode fromB

      g.addEdge fromA, toA
      g.addEdge fromA, toB
      g.addEdge fromB, toA
      g.addEdge fromB, toB

      fg = g.filter (node) ->
        return /^keepMe/.test node.hash()

      expect(fg.nodesPointedToBy(fromA)).to.eql [toA]
      expect(fg.nodesPointingTo(toA)).to.eql [fromA]

  describe '#rootNodes', ->
    it 'should return all nodes that are not pointed to by another node', ->
      g = new Graph
      rootA = new Node 'rootA'
      rootB = new Node 'rootB'
      leafA = new Node 'leafA'
      leafB = new Node 'leafB'

      for node in [rootA, rootB, leafA, leafB]
        g.addNode node

      g.addEdge rootA, leafA
      g.addEdge rootA, leafB
      g.addEdge rootB, leafA
      g.addEdge rootB, leafB

      rootNodes = g.rootNodes()
      expect(rootNodes).to.contain rootA
      expect(rootNodes).to.contain rootB
      expect(rootNodes).to.not.contain leafA
      expect(rootNodes).to.not.contain leafB
