/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.lite.support.security;

import com.couchbase.lite.support.security.SymmetricKeyException;
import com.couchbase.lite.util.ArrayUtils;
import com.couchbase.lite.util.Log;
import com.couchbase.lite.util.Utils;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.AlgorithmParameters;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

public class SymmetricKey {
    public static final int KEY_SIZE = 32;
    private static final int BLOCK_SIZE = 16;
    private static final int IV_SIZE = 16;
    private byte[] keyData = null;
    private boolean useBCProvider = false;

    public SymmetricKey() throws SymmetricKeyException {
        this(SymmetricKey.generateKey(32));
    }

    public SymmetricKey(byte[] key) throws SymmetricKeyException {
        this.initWithKey(key);
    }

    private void initWithKey(byte[] key) throws SymmetricKeyException {
        if (key == null) {
            throw new SymmetricKeyException("Key cannot be null");
        }
        if (key.length != 32) {
            throw new SymmetricKeyException("Key size is not 32bytes");
        }
        this.keyData = key;
    }

    public byte[] getKey() {
        return this.keyData;
    }

    public String getHexData() {
        return Utils.bytesToHex(this.keyData);
    }

    public String toString() {
        return this.getHexData() + " (" + Arrays.toString(this.keyData) + ')';
    }

    public byte[] encryptData(byte[] data) throws SymmetricKeyException {
        Encryptor encryptor = this.createEncryptor();
        byte[] encrypted = encryptor.encrypt(data);
        byte[] trailer = encryptor.encrypt(null);
        if (encrypted == null || trailer == null) {
            throw new SymmetricKeyException("Cannot encrypt data");
        }
        byte[] result = ArrayUtils.concat(encrypted, trailer);
        return result;
    }

    public byte[] decryptData(byte[] data) throws SymmetricKeyException {
        if (data.length < 16) {
            throw new SymmetricKeyException("Invalid encrypted data, no IV prepended");
        }
        byte[] iv = ArrayUtils.subarray(data, 0, 16);
        Cipher cipher = this.getCipher(2, iv);
        try {
            return cipher.doFinal(data, iv.length, data.length - iv.length);
        }
        catch (Exception e) {
            throw new SymmetricKeyException(e);
        }
    }

    public InputStream decryptStream(InputStream input) throws SymmetricKeyException {
        try {
            EncryptedInputStream encryptedInputStream = new EncryptedInputStream(input);
            byte[] iv = encryptedInputStream.getIv();
            Cipher cipher = this.getCipher(2, iv);
            return new CipherInputStream(encryptedInputStream, cipher);
        }
        catch (IOException e) {
            throw new SymmetricKeyException(e);
        }
    }

    private static byte[] generateKey(int size) throws SymmetricKeyException {
        if (size <= 0) {
            throw new IllegalArgumentException("Size cannot be zero or less than zero.");
        }
        try {
            SecureRandom secureRandom = new SecureRandom();
            KeyGenerator keyGenerator = KeyGenerator.getInstance("AES");
            keyGenerator.init(size * 8, secureRandom);
            return keyGenerator.generateKey().getEncoded();
        }
        catch (NoSuchAlgorithmException e) {
            throw new SymmetricKeyException(e);
        }
    }

    private static byte[] secureRandom(int size) {
        if (size <= 0) {
            throw new IllegalArgumentException("Size cannot be zero or less than zero.");
        }
        SecureRandom secureRandom = new SecureRandom();
        byte[] bytes = new byte[size];
        secureRandom.nextBytes(bytes);
        return bytes;
    }

    private Cipher getCipher(int mode, byte[] iv) throws SymmetricKeyException {
        Cipher cipher = null;
        try {
            cipher = this.getCipherInstance("AES/CBC/PKCS7Padding");
            if (cipher == null) {
                throw new SymmetricKeyException("Cannot get a cipher instance for AES/CBC/PKCS7Padding algorithm");
            }
            SecretKeySpec secret = new SecretKeySpec(this.getKey(), "AES");
            cipher.init(mode, (Key)secret, new IvParameterSpec(iv));
        }
        catch (InvalidKeyException e) {
            throw new SymmetricKeyException("Couchbase Lite uses the AES 256-bit key to provide data encryption. Please make sure you have installed 'Java Cryptography Extension (JCE) Unlimited Strength Jurisdiction' Policy provided by Oracle.", e);
        }
        catch (SymmetricKeyException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SymmetricKeyException(e);
        }
        return cipher;
    }

    private Cipher getCipherInstance(String algorithm) {
        Cipher cipher = null;
        if (!this.useBCProvider) {
            try {
                cipher = Cipher.getInstance(algorithm);
            }
            catch (NoSuchAlgorithmException e) {
                Log.v("SymmetricKey", "Cannot find a cipher (no algorithm); will try with Bouncy Castle provider.");
            }
            catch (NoSuchPaddingException e) {
                Log.v("SymmetricKey", "Cannot find a cipher (no padding); will try with Bouncy Castle provider.");
            }
        }
        if (cipher == null) {
            try {
                if (Security.getProvider("BC") == null) {
                    try {
                        Class<?> bc = Class.forName("org.bouncycastle.jce.provider.BouncyCastleProvider");
                        Security.addProvider((Provider)bc.newInstance());
                    }
                    catch (Exception e) {
                        Log.e("SymmetricKey", "Cannot instantiate Bouncy Castle provider", e);
                        return null;
                    }
                }
                cipher = Cipher.getInstance(algorithm, "BC");
                this.useBCProvider = true;
            }
            catch (Exception e) {
                Log.e("SymmetricKey", "Cannot find a cipher with Bouncy Castle provider", e);
            }
        }
        return cipher;
    }

    public Encryptor createEncryptor() throws SymmetricKeyException {
        return new Encryptor();
    }

    private class EncryptedInputStream
    extends FilterInputStream {
        private byte[] iv;

        protected EncryptedInputStream(InputStream in) throws IOException, SymmetricKeyException {
            super(in);
            this.iv = null;
            if (in != null) {
                byte[] ivBuffer = new byte[16];
                if (in.read(ivBuffer, 0, 16) == 16) {
                    this.iv = ivBuffer;
                } else {
                    throw new SymmetricKeyException("Invalid encrypted data, no IV prepended");
                }
            }
        }

        public byte[] getIv() {
            return this.iv;
        }
    }

    public class Encryptor {
        private Cipher cipher;
        private boolean wroteIV;

        public Encryptor() throws SymmetricKeyException {
            byte[] iv = SymmetricKey.secureRandom(16);
            this.cipher = SymmetricKey.this.getCipher(1, iv);
            this.wroteIV = false;
        }

        public byte[] encrypt(byte[] data) throws SymmetricKeyException {
            return this.encrypt(data, 0, data != null ? data.length : 0);
        }

        public byte[] encrypt(byte[] data, int offset, int len) throws SymmetricKeyException {
            byte[] dataOut;
            try {
                dataOut = data != null ? this.cipher.update(data, offset, len) : this.cipher.doFinal();
                if (!this.wroteIV) {
                    AlgorithmParameters params = this.cipher.getParameters();
                    byte[] iv = params.getParameterSpec(IvParameterSpec.class).getIV();
                    dataOut = ArrayUtils.concat(iv, dataOut);
                    this.wroteIV = true;
                }
            }
            catch (Exception e) {
                throw new SymmetricKeyException(e);
            }
            return dataOut;
        }
    }
}

