import { AnalyticsBridge } from "./NativeBridges";
/**
 * Custom dimensions accept only strings and numbers.
 * @param customDimensionVal
 * @returns {boolean}
 */
function isValidCustomDimension(customDimensionVal) {
    const customDimensionValType = typeof customDimensionVal;
    return (customDimensionValType === "string" || customDimensionValType === "number");
}
/**
 * Used to bridge tracker data to native Google analytics.
 * Saves necessary tracker (specific) data to format data as native part of Google analytics expect.
 */
export default class GoogleAnalyticsTracker {
    /**
     * Save all tracker related data that is needed to call native methods with proper data.
     * @param {String} trackerId
     * @param {{fieldName: fieldIndex}} customDimensionsFieldsIndexMap Custom dimensions field/index pairs
     */
    constructor(trackerId, customDimensionsFieldsIndexMap) {
        this.id = trackerId;
        this.customDimensionsFieldsIndexMap = customDimensionsFieldsIndexMap;
    }
    /**
     * If Tracker has customDimensionsFieldsIndexMap, it will transform
     * customDimensions map pairs {field: value} to {fieldIndex: value}.
     * Otherwise customDimensions are passed trough untouched.
     * Underlay native methods will transform provided customDimensions map to expected format.
     * Google analytics expect dimensions to be tracker with 'dimension{index}' keys,
     * not dimension field names.
     * @param {CustomDimensionsByIndex} customDimensions
     * @returns {CustomDimensionsByField}
     */
    transformCustomDimensionsFieldsToIndexes(customDimensions) {
        if (this.customDimensionsFieldsIndexMap) {
            return Object.keys(this.customDimensionsFieldsIndexMap)
                .filter(key => isValidCustomDimension(customDimensions[key]))
                .reduce((mappedCustomDimensions, key) => {
                const dimensionIndex = this.customDimensionsFieldsIndexMap[key];
                mappedCustomDimensions[dimensionIndex] = customDimensions[key];
                return mappedCustomDimensions;
            }, {});
        }
        return customDimensions;
    }
    /**
     * Track the current screen/view
     * @param  {String} screenName The name of the current screen
     */
    trackScreenView(screenName, payload = null) {
        AnalyticsBridge.trackScreenView(this.id, screenName, payload);
    }
    /**
     * Track an event that has occured
     * @param  {String} category       The event category
     * @param  {String} action         The event action
     * @param  {OptionalValue} optionalValues An object containing optional label and value
     */
    trackEvent(category, action, payload = null, label = null, value = null) {
        AnalyticsBridge.trackEvent(this.id, category, action, label, value, payload);
    }
    /**
     * Track an event that has occured
     * @param  {String} category       The event category
     * @param  {Number} value         	The timing measurement in milliseconds
     * @param  {OptionalTimingValue} optionalValues An object containing optional name and label
     */
    trackTiming(category, value, payload = null, name = null, label = null) {
        AnalyticsBridge.trackTiming(this.id, category, value, name, label, payload);
    }
    /**
     * Track an exception
     * @param  {String} error The description of the error
     * @param  {Boolean} fatal A value indiciating if the error was fatal, defaults to false
     */
    trackException(error, fatal = false, payload = null) {
        AnalyticsBridge.trackException(this.id, error, fatal, payload);
    }
    /**
     * Track a social interaction, Facebook, Twitter, etc.
     * @param  {String} network
     * @param  {String} action
     * @param  {String} targetUrl
     */
    trackSocialInteraction(network, action, targetUrl, payload) {
        AnalyticsBridge.trackSocialInteraction(this.id, network, action, targetUrl, payload);
    }
    /**
     * Sets the current userId for tracking.
     * @param {String} userId The current userId
     */
    setUser(userId) {
        AnalyticsBridge.setUser(this.id, userId);
    }
    /**
     * Sets the current clientId for tracking.
     * @param {String} clientId The current userId
     */
    setClient(clientId) {
        AnalyticsBridge.setClient(this.id, clientId);
    }
    /**
     * Sets if IDFA (identifier for advertisers) collection should be enabled
     * @param  {Boolean} enabled Defaults to true
     */
    allowIDFA(enabled) {
        AnalyticsBridge.allowIDFA(this.id, enabled);
    }
    /**
     * Sets the trackers appName
     * The Bundle name is used by default
     * @param {String} appName
     */
    setAppName(appName) {
        AnalyticsBridge.setAppName(this.id, appName);
    }
    /**
     * Sets the trackers appVersion
     * @param {String} appVersion
     */
    setAppVersion(appVersion) {
        AnalyticsBridge.setAppVersion(this.id, appVersion);
    }
    /**
     * Sets if AnonymizeIp is enabled
     * If enabled the last octet of the IP address will be removed
     * @param {Boolean} enabled
     */
    setAnonymizeIp(enabled) {
        AnalyticsBridge.setAnonymizeIp(this.id, enabled);
    }
    /**
     * Sets tracker sampling rate.
     * @param {Float} sampleRatio Percentage 0 - 100
     */
    setSamplingRate(sampleRatio) {
        AnalyticsBridge.setSamplingRate(this.id, sampleRatio);
    }
    /**
     * Sets the currency for tracking.
     * @param {String} currencyCode The currency ISO 4217 code
     */
    setCurrency(currencyCode) {
        AnalyticsBridge.setCurrency(this.id, currencyCode);
    }
    /**
     * Sets if uncaught exceptions should be tracked
     * Important to note: On iOS this option is set on all trackers. On Android it is set per tracker.
     * If you are using multiple trackers on iOS, this will enable & disable on all trackers.
     * @param {Boolean} enabled
     */
    setTrackUncaughtExceptions(enabled) {
        AnalyticsBridge.setTrackUncaughtExceptions(this.id, enabled);
    }
}
