import { HitPayload } from "./HitPayload";
import { CustomDimensionsByField, CustomDimensionsByIndex, CustomDimensionsFieldIndexMap } from "./Custom";
/**
 * Used to bridge tracker data to native Google analytics.
 * Saves necessary tracker (specific) data to format data as native part of Google analytics expect.
 */
export default class GoogleAnalyticsTracker {
    id: string;
    customDimensionsFieldsIndexMap: CustomDimensionsFieldIndexMap;
    /**
     * Save all tracker related data that is needed to call native methods with proper data.
     * @param {String} trackerId
     * @param {{fieldName: fieldIndex}} customDimensionsFieldsIndexMap Custom dimensions field/index pairs
     */
    constructor(trackerId: string, customDimensionsFieldsIndexMap?: CustomDimensionsFieldIndexMap);
    /**
     * If Tracker has customDimensionsFieldsIndexMap, it will transform
     * customDimensions map pairs {field: value} to {fieldIndex: value}.
     * Otherwise customDimensions are passed trough untouched.
     * Underlay native methods will transform provided customDimensions map to expected format.
     * Google analytics expect dimensions to be tracker with 'dimension{index}' keys,
     * not dimension field names.
     * @param {CustomDimensionsByIndex} customDimensions
     * @returns {CustomDimensionsByField}
     */
    transformCustomDimensionsFieldsToIndexes(customDimensions: CustomDimensionsByIndex): CustomDimensionsByField;
    /**
     * Track the current screen/view
     * @param  {String} screenName The name of the current screen
     */
    trackScreenView(screenName: string, payload?: HitPayload): void;
    /**
     * Track an event that has occured
     * @param  {String} category       The event category
     * @param  {String} action         The event action
     * @param  {OptionalValue} optionalValues An object containing optional label and value
     */
    trackEvent(category: string, action: string, payload?: HitPayload, label?: string, value?: number): void;
    /**
     * Track an event that has occured
     * @param  {String} category       The event category
     * @param  {Number} value         	The timing measurement in milliseconds
     * @param  {OptionalTimingValue} optionalValues An object containing optional name and label
     */
    trackTiming(category: string, value: number, payload?: HitPayload, name?: string, label?: string): void;
    /**
     * Track an exception
     * @param  {String} error The description of the error
     * @param  {Boolean} fatal A value indiciating if the error was fatal, defaults to false
     */
    trackException(error: string, fatal?: boolean, payload?: HitPayload): void;
    /**
     * Track a social interaction, Facebook, Twitter, etc.
     * @param  {String} network
     * @param  {String} action
     * @param  {String} targetUrl
     */
    trackSocialInteraction(network: string, action: string, targetUrl: string, payload: HitPayload): void;
    /**
     * Sets the current userId for tracking.
     * @param {String} userId The current userId
     */
    setUser(userId: string): void;
    /**
     * Sets the current clientId for tracking.
     * @param {String} clientId The current userId
     */
    setClient(clientId: string): void;
    /**
     * Sets if IDFA (identifier for advertisers) collection should be enabled
     * @param  {Boolean} enabled Defaults to true
     */
    allowIDFA(enabled: boolean): void;
    /**
     * Sets the trackers appName
     * The Bundle name is used by default
     * @param {String} appName
     */
    setAppName(appName: string): void;
    /**
     * Sets the trackers appVersion
     * @param {String} appVersion
     */
    setAppVersion(appVersion: string): void;
    /**
     * Sets if AnonymizeIp is enabled
     * If enabled the last octet of the IP address will be removed
     * @param {Boolean} enabled
     */
    setAnonymizeIp(enabled: boolean): void;
    /**
     * Sets tracker sampling rate.
     * @param {Float} sampleRatio Percentage 0 - 100
     */
    setSamplingRate(sampleRatio: number): void;
    /**
     * Sets the currency for tracking.
     * @param {String} currencyCode The currency ISO 4217 code
     */
    setCurrency(currencyCode: string): void;
    /**
     * Sets if uncaught exceptions should be tracked
     * Important to note: On iOS this option is set on all trackers. On Android it is set per tracker.
     * If you are using multiple trackers on iOS, this will enable & disable on all trackers.
     * @param {Boolean} enabled
     */
    setTrackUncaughtExceptions(enabled: boolean): void;
}
