/**
 *
 * All properties on the payload is optional. Preferably leave them empty unless you actually need to send something extra.
 * @class HitPayload
 * @typicalname payload
 * @prop {Product[]} [products] (Optional) Products which an action has occured on.
 * @prop {ProductAction} [productAction] (Optional) The action which occured on the products. Required if `products` is not empty.
 * @prop {Product[]} [impressionProducts] (Optional) Products which has had an impression.
 * @prop {string} [impressionList] (Optional) The list which product impressions occured on. Required if `impressionProducts` is not empty.
 * @prop {string} [impressionSource] (Optional) The source which impression occured from (Android only).
 * @prop {CustomDimensionsByIndex | CustomDimensionsByField} [customDimensions] (Optional) Custom dimensions.
 * @prop {CustomMetrics} [customMetrics] (Optional) Custom metrics.
 * @prop {string} [utmCampaignUrl] (Optional) UTM campaign URL.
 * @prop {number} [startSession] (Optional) Start new session.
 */
export class HitPayload {
    /**
     * Adds a product in the payload.
     * `productAction` must also be set on the payload if you add products.
     * @param {Product} product
     * @returns {HitPayload}
     * @memberof HitPayload
     */
    addProduct(product) {
        this.products.push(product);
        return this;
    }
    /**
     * Adds a product impression.
     * `impressionList` must also be set on the payload if you add impression products.
     * @param {Product} product
     * @returns {HitPayload}
     * @memberof HitPayload
     */
    addProductImpression(product) {
        this.impressionProducts.push(product);
        return this;
    }
}
/**
 * Base class for all Hits.
 * @class Hit
 * @prop {HitPayload} [payload=null] (Optional) Payload to send with the hit.
 */
export class Hit {
    constructor() {
        this.payload = null;
    }
}
/**
 * @export
 * @class ScreenHit
 * @extends {Hit}
 * @prop {string} screenName The current screen name.
 */
export class ScreenHit extends Hit {
    /**
     * Creates an instance of ScreenHit.
     * @param {string} screenName
     * @memberof ScreenHit
     */
    constructor(screenName) {
        super();
        this.screenName = screenName;
    }
}
/**
 * @export
 * @class EventHit
 * @extends {Hit}
 * @prop {string} category The event category
 * @prop {string} action The event action
 * @prop {string} [label=null] The event label
 * @prop {number} [value=null] The event value
 */
export class EventHit extends Hit {
    /**
     * Creates an instance of EventHit.
     * @param {string} category
     * @param {string} action
     * @param {string} [label=null]
     * @param {number} [value=null]
     * @memberof EventHit
     */
    constructor(category, action, label = null, value = null) {
        super();
        this.category = category;
        this.action = action;
        this.label = label;
        this.value = value;
    }
}
/**
 * @export
 * @class TimingHit
 * @extends {Hit}
 * @prop {string} category The timing category
 * @prop {number} interval The timing interval
 * @prop {string} name The timing name
 * @prop {string} [label=null] The timing label
 */
export class TimingHit extends Hit {
    /**
     * Creates an instance of TimingHit.
     * @param {string} category
     * @param {number} interval
     * @param {string} name
     * @param {string} [label=null]
     * @memberof TimingHit
     */
    constructor(category, interval, name, label = null) {
        super();
        this.category = category;
        this.interval = interval;
        this.name = name;
        this.label = label;
    }
}
/**
 * @export
 * @class ExceptionHit
 * @extends {Hit}
 * @prop {string} error The error
 * @prop {boolean} [fatal=false] If the error was fatal. Defaults to false.
 */
export class ExceptionHit extends Hit {
    /**
     * Creates an instance of ExceptionHit.
     * @param {string} error
     * @param {boolean} [fatal=false]
     * @memberof ExceptionHit
     */
    constructor(error, fatal = false) {
        super();
        this.error = error;
        this.fatal = fatal;
    }
}
/**
 *
 *
 * @export
 * @class SocialHit
 * @extends {Hit}
 * @prop {string} network The social network
 * @prop {string} action The social action. +1, like, etc
 * @prop {string} targetUrl The target url.
 */
export class SocialHit extends Hit {
    /**
     * Creates an instance of SocialHit.
     * @param {string} network
     * @param {string} action
     * @param {string} targetUrl
     * @memberof SocialHit
     */
    constructor(network, action, targetUrl) {
        super();
        this.network = network;
        this.action = action;
        this.targetUrl = targetUrl;
    }
}
