import { CustomDimensionsByIndex, CustomMetrics, CustomDimensionsByField } from "./CustomValues";
import { Product, ProductAction } from "./Product";
export interface IHitPayload {
    products: Product[];
    impressionProducts: Product[];
    impressionList?: string;
    impressionSource?: string;
    productAction?: ProductAction;
    customDimensions?: CustomDimensionsByIndex | CustomDimensionsByField;
    customMetrics?: CustomMetrics;
    utmCampaignUrl?: string;
    startSession?: number;
}
/**
 *
 * All properties on the payload is optional. Preferably leave them empty unless you actually need to send something extra.
 * @class HitPayload
 * @typicalname payload
 * @prop {Product[]} [products] (Optional) Products which an action has occured on.
 * @prop {ProductAction} [productAction] (Optional) The action which occured on the products. Required if `products` is not empty.
 * @prop {Product[]} [impressionProducts] (Optional) Products which has had an impression.
 * @prop {string} [impressionList] (Optional) The list which product impressions occured on. Required if `impressionProducts` is not empty.
 * @prop {string} [impressionSource] (Optional) The source which impression occured from (Android only).
 * @prop {CustomDimensionsByIndex | CustomDimensionsByField} [customDimensions] (Optional) Custom dimensions.
 * @prop {CustomMetrics} [customMetrics] (Optional) Custom metrics.
 * @prop {string} [utmCampaignUrl] (Optional) UTM campaign URL.
 * @prop {number} [startSession] (Optional) Start new session.
 */
export declare class HitPayload implements IHitPayload {
    products: Product[];
    productAction?: ProductAction;
    impressionProducts: Product[];
    impressionList?: string;
    impressionSource?: string;
    customDimensions?: CustomDimensionsByIndex | CustomDimensionsByField;
    customMetrics?: CustomMetrics;
    utmCampaignUrl?: string;
    startSession?: number;
    /**
     * Adds a product in the payload.
     * `productAction` must also be set on the payload if you add products.
     * @param {Product} product
     * @returns {HitPayload}
     * @memberof HitPayload
     */
    addProduct(product: Product): HitPayload;
    /**
     * Adds a product impression.
     * `impressionList` must also be set on the payload if you add impression products.
     * @param {Product} product
     * @returns {HitPayload}
     * @memberof HitPayload
     */
    addProductImpression(product: Product): HitPayload;
}
/**
 * Base class for all Hits.
 * @class Hit
 * @prop {HitPayload} [payload=null] (Optional) Payload to send with the hit.
 */
export declare class Hit {
    payload: HitPayload;
}
/**
 * @export
 * @class ScreenHit
 * @extends {Hit}
 * @prop {string} screenName The current screen name.
 */
export declare class ScreenHit extends Hit {
    screenName: string;
    /**
     * Creates an instance of ScreenHit.
     * @param {string} screenName
     * @memberof ScreenHit
     */
    constructor(screenName: string);
}
/**
 * @export
 * @class EventHit
 * @extends {Hit}
 * @prop {string} category The event category
 * @prop {string} action The event action
 * @prop {string} [label=null] The event label
 * @prop {number} [value=null] The event value
 */
export declare class EventHit extends Hit {
    category: string;
    action: string;
    label?: string;
    value?: number;
    /**
     * Creates an instance of EventHit.
     * @param {string} category
     * @param {string} action
     * @param {string} [label=null]
     * @param {number} [value=null]
     * @memberof EventHit
     */
    constructor(category: string, action: string, label?: string, value?: number);
}
/**
 * @export
 * @class TimingHit
 * @extends {Hit}
 * @prop {string} category The timing category
 * @prop {number} interval The timing interval
 * @prop {string} name The timing name
 * @prop {string} [label=null] The timing label
 */
export declare class TimingHit extends Hit {
    category: string;
    interval: number;
    name: string;
    label?: string;
    /**
     * Creates an instance of TimingHit.
     * @param {string} category
     * @param {number} interval
     * @param {string} name
     * @param {string} [label=null]
     * @memberof TimingHit
     */
    constructor(category: string, interval: number, name: string, label?: string);
}
/**
 * @export
 * @class ExceptionHit
 * @extends {Hit}
 * @prop {string} error The error
 * @prop {boolean} [fatal=false] If the error was fatal. Defaults to false.
 */
export declare class ExceptionHit extends Hit {
    error: string;
    fatal: boolean;
    /**
     * Creates an instance of ExceptionHit.
     * @param {string} error
     * @param {boolean} [fatal=false]
     * @memberof ExceptionHit
     */
    constructor(error: string, fatal?: boolean);
}
/**
 *
 *
 * @export
 * @class SocialHit
 * @extends {Hit}
 * @prop {string} network The social network
 * @prop {string} action The social action. +1, like, etc
 * @prop {string} targetUrl The target url.
 */
export declare class SocialHit extends Hit {
    network: string;
    action: string;
    targetUrl: string;
    /**
     * Creates an instance of SocialHit.
     * @param {string} network
     * @param {string} action
     * @param {string} targetUrl
     * @memberof SocialHit
     */
    constructor(network: string, action: string, targetUrl: string);
}
