import { AnalyticsBridge } from "./NativeBridges";
/**
 * Custom dimensions accept only strings and numbers.
 * @ignore
 * @param customDimensionVal
 * @returns {boolean}
 */
function isValidCustomDimension(customDimensionVal) {
    const customDimensionValType = typeof customDimensionVal;
    return (customDimensionValType === "string" || customDimensionValType === "number");
}
/**
 * @typicalname tracker
 */
class NewGoogleAnalyticsTracker {
    /**
     * Save all tracker related data that is needed to call native methods with proper data.
     * @param {string} trackerId Your tracker id, something like: UA-12345-1
     * @param {{fieldName: fieldIndex}} customDimensionsFieldsIndexMap Custom dimensions field/index pairs
     * @example
     * ```js
     * import { GoogleAnalyticsTracker } from 'react-native-google-analytics-bridge';
     * let tracker = new GoogleAnalyticsTracker('UA-12345-1');
     * ```
     */
    constructor(trackerId, customDimensionsFieldsIndexMap) {
        this.id = trackerId;
        this.customDimensionsFieldsIndexMap = customDimensionsFieldsIndexMap;
    }
    /**
     *
     *
     * @param {string} screenName
     * @memberof NewGoogleAnalyticsTracker
     */
    trackScreen(screenName) {
        AnalyticsBridge.trackScreenView(this.id, screenName, null);
    }
    /**
     *
     *
     * @param {ScreenHit} hit
     * @memberof NewGoogleAnalyticsTracker
     */
    trackScreenHit(hit) {
        AnalyticsBridge.trackScreenView(this.id, hit.screenName, this.sanitizePayload(hit.payload));
    }
    /**
     *
     *
     * @param {string} category
     * @param {string} action
     * @param {string} [label=null]
     * @param {number} [value=null]
     * @memberof NewGoogleAnalyticsTracker
     */
    trackEvent(category, action, label = null, value = null) {
        AnalyticsBridge.trackEvent(this.id, category, action, label, value, null);
    }
    /**
     *
     *
     * @param {EventHit} hit
     * @memberof NewGoogleAnalyticsTracker
     */
    trackEvenHit(hit) {
        AnalyticsBridge.trackEvent(this.id, hit.category, hit.action, hit.label, hit.value, this.sanitizePayload(hit.payload));
    }
    /**
     *
     *
     * @param {string} category
     * @param {number} interval
     * @param {string} name
     * @param {string} [label=null]
     * @memberof NewGoogleAnalyticsTracker
     */
    trackTiming(category, interval, name, label = null) {
        AnalyticsBridge.trackTiming(this.id, category, interval, name, label, null);
    }
    /**
     *
     *
     * @param {TimingHit} hit
     * @memberof NewGoogleAnalyticsTracker
     */
    trackTimingHit(hit) {
        AnalyticsBridge.trackTiming(this.id, hit.category, hit.interval, hit.name, hit.label, this.sanitizePayload(hit.payload));
    }
    /**
     *
     *
     * @param {string} error
     * @param {boolean} [fatal=false]
     * @memberof NewGoogleAnalyticsTracker
     */
    trackException(error, fatal = false) {
        AnalyticsBridge.trackException(this.id, error, fatal, null);
    }
    /**
     *
     *
     * @param {ExceptionHit} hit
     * @memberof NewGoogleAnalyticsTracker
     */
    trackExceptionHit(hit) {
        AnalyticsBridge.trackException(this.id, hit.error, hit.fatal, this.sanitizePayload(hit.payload));
    }
    /**
     *
     *
     * @param {string} network
     * @param {string} action
     * @param {string} targetUrl
     * @memberof NewGoogleAnalyticsTracker
     */
    trackSocial(network, action, targetUrl) {
        AnalyticsBridge.trackSocialInteraction(this.id, network, action, targetUrl, null);
    }
    /**
     *
     *
     * @param {SocialHit} hit
     * @memberof NewGoogleAnalyticsTracker
     */
    trackSocialHit(hit) {
        AnalyticsBridge.trackSocialInteraction(this.id, hit.network, hit.action, hit.targetUrl, this.sanitizePayload(hit.payload));
    }
    /**
     * Sets the current userId for tracking.
     * @param {string} userId The current userId
     */
    setUser(userId) {
        AnalyticsBridge.setUser(this.id, userId);
    }
    /**
     * Sets the current clientId for tracking.
     * @param {string} clientId The current userId
     */
    setClient(clientId) {
        AnalyticsBridge.setClient(this.id, clientId);
    }
    /**
     * Sets if IDFA (identifier for advertisers) collection should be enabled
     * @param  {boolean} enabled (Optional) Defaults to true
     */
    allowIDFA(enabled = true) {
        AnalyticsBridge.allowIDFA(this.id, enabled);
    }
    /**
     * Sets the trackers appName
     * The Bundle name is used by default
     * @param {string} appName (Required) The app name
     */
    setAppName(appName) {
        AnalyticsBridge.setAppName(this.id, appName);
    }
    /**
     * Sets the trackers appVersion
     * @param {string} appVersion (Required) The app version
     */
    setAppVersion(appVersion) {
        AnalyticsBridge.setAppVersion(this.id, appVersion);
    }
    /**
     * Sets if AnonymizeIp is enabled
     * If enabled the last octet of the IP address will be removed
     * @param {boolean} enabled (Required) Enable anonymize IP
     */
    setAnonymizeIp(enabled) {
        AnalyticsBridge.setAnonymizeIp(this.id, enabled);
    }
    /**
     * Sets tracker sampling rate.
     * @param {number} sampleRatio (Required) Percentage 0 - 100
     */
    setSamplingRate(sampleRatio) {
        AnalyticsBridge.setSamplingRate(this.id, sampleRatio);
    }
    /**
     * Sets the currency for tracking.
     * @param {string} currencyCode (Required) The currency ISO 4217 code
     */
    setCurrency(currencyCode) {
        AnalyticsBridge.setCurrency(this.id, currencyCode);
    }
    /**
     * Sets if uncaught exceptions should be tracked
     * Important to note: On iOS this option is set on all trackers. On Android it is set per tracker.
     * If you are using multiple trackers on iOS, this will enable & disable on all trackers.
     * @param {boolean} enabled Should enable uncaught exception tracking
     */
    setTrackUncaughtExceptions(enabled) {
        AnalyticsBridge.setTrackUncaughtExceptions(this.id, enabled);
    }
    /**
     * If Tracker has customDimensionsFieldsIndexMap, it will transform
     * customDimensions map pairs {field: value} to {fieldIndex: value}.
     * Otherwise customDimensions are passed trough untouched.
     * Underlying native methods will transform provided customDimensions map to expected format.
     * Google analytics expect dimensions to be tracker with 'dimension{index}' keys,
     * not dimension field names.
     * @ignore
     * @param {{fieldName: value}} customDimensions
     * @returns {{fieldIndex: value}}
     */
    transformCustomDimensionsFieldsToIndexes(customDimensions) {
        if (this.customDimensionsFieldsIndexMap) {
            return Object.keys(this.customDimensionsFieldsIndexMap)
                .filter(key => isValidCustomDimension(customDimensions[key]))
                .reduce((mappedCustomDimensions, key) => {
                const dimensionIndex = this.customDimensionsFieldsIndexMap[key];
                mappedCustomDimensions[dimensionIndex] = customDimensions[key];
                return mappedCustomDimensions;
            }, {});
        }
        return customDimensions;
    }
    /**
     * Transforms payload so it is valid for native side
     *
     * @ignore
     */
    sanitizePayload(payload) {
        if (payload && payload.customDimensions) {
            payload.customDimensions = this.transformCustomDimensionsFieldsToIndexes(payload.customDimensions);
        }
        return payload;
    }
}
export default NewGoogleAnalyticsTracker;
