/*
 *   OneModule.java
 *   Thunderhead
 *
 *   Copyright © 2017 Thunderhead. All rights reserved.
 */

package com.thunderhead.one.react;

import com.facebook.react.bridge.Promise;
import com.facebook.react.bridge.ReactApplicationContext;
import com.facebook.react.bridge.ReactContextBaseJavaModule;
import com.facebook.react.bridge.ReactMethod;
import com.facebook.react.bridge.ReadableArray;
import com.facebook.react.bridge.ReadableMap;
import com.facebook.react.bridge.ReadableMapKeySetIterator;
import com.facebook.react.bridge.ReadableNativeMap;
import com.thunderhead.One;
import com.thunderhead.OneLogLevel;
import com.thunderhead.OneModes;
import com.thunderhead.connectivity.entitys.responses.BaseResponse;
import com.thunderhead.interfaces.GetCallback;
import com.thunderhead.utils.ThunderheadException;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

public class OneModule extends ReactContextBaseJavaModule {

    final One one;

    public OneModule(ReactApplicationContext reactContext) {
        super(reactContext);
        one = One.getInstance(getReactApplicationContext());
    }

    @Override
    public String getName() {
        return "One";
    }

    @ReactMethod
    public void init(String siteKey, String touchpointUri, String apiKey,
                     String sharedSecret, String userId, boolean isAdminMode,
                     String hostName) {
        if (isAdminMode) {
            one.init(siteKey,touchpointUri,apiKey,sharedSecret, userId, OneModes.ADMIN_MODE, hostName);
        } else {
            one.init(siteKey,touchpointUri,apiKey,sharedSecret, userId, OneModes.USER_MODE, hostName);
        }
        one.disableAutomaticInteractionDetection(true);
    }

    @ReactMethod
    public void sendInteraction(String interaction, ReadableMap propertiesMap, final Promise promise) {
        HashMap<String, String> properties = getPropertiesFromReadableMap(propertiesMap);
        if (properties != null && !properties.isEmpty()) {
           if (promise == null) {
               one.sendInteraction(interaction, properties);
           } else  {
               one.sendInteraction(interaction, properties, new GetCallback<BaseResponse>() {
                   @Override
                   public void done(BaseResponse baseResponse, ThunderheadException e) {
                       if (e == null) {
                           notifyResult(promise, baseResponse.toString());
                       } else {
                           notifyProblem(promise, "" + e.getErrorCode(), e.getErrorMessage());
                       }
                   }
               });
           }
        } else  {
            if (promise == null) {
                one.sendInteraction(interaction);
            } else  {
                one.sendInteraction(interaction, new GetCallback<BaseResponse>() {
                    @Override
                    public void done(BaseResponse baseResponse, ThunderheadException e) {
                        if (e == null) {
                            notifyResult(promise, baseResponse.toString());
                        } else {
                            notifyProblem(promise, "" + e.getErrorCode(), e.getErrorMessage());
                        }
                    }
                });
            }
        }
    }

    @ReactMethod
    public void sendProperties(String interaction, ReadableMap propertiesMap) {
        HashMap<String, String> properties = getPropertiesFromReadableMap(propertiesMap);
        one.sendProperties(interaction, properties);
    }

    @ReactMethod
    public void sendBaseTouchpointProperties(ReadableMap propertiesMap) {
        HashMap<String, String> properties = getPropertiesFromReadableMap(propertiesMap);
        one.sendBaseTouchpointProperties(properties);
    }

    @ReactMethod
    public void sendResponseCode(String responseCode, String interaction) {
        one.sendResponseCode(responseCode,interaction);
    }

    @ReactMethod
    public void sendInteractionForOutboundLink(String stringUrl, Promise promise) {
        URL url = null;
        try {
            url = new URL(stringUrl);
            one.sendInteractionForOutboundLink(url);
        } catch (MalformedURLException e) {
            e.printStackTrace();
            if (promise != null) {
                notifyProblem(promise, e);
            }
        }
    }

    @ReactMethod
    public void getURLWithOneTid(String urlString, Promise promise) {
        if (promise == null) {
            return;
        }
        try {
            URL url = new URL(urlString);
            URL urlWithOneTid = one.getURLWithOneTid(url);
            notifyResult(promise, urlWithOneTid.toString());
        } catch (Exception e) {
            e.printStackTrace();
            notifyProblem(promise, e);
        }
    }

    @ReactMethod
    public void blacklistIdentityTransferLinks(ReadableArray links, Promise promise) {
        ArrayList<String> linksToBlacklist = getArrayListFromReadableArray(links);
        if (null == linksToBlacklist || linksToBlacklist.isEmpty()) {
            notifyProblem(promise, null, "An array of links to be blacklisted is empty!");
            return;
        }
        one.blacklistIdentityTransferLinks(linksToBlacklist);
        notifyResult(promise, null);
    }

    @ReactMethod
    public void whitelistIdentityTransferLinks(ReadableArray links, Promise promise) {
        ArrayList<String> linksToWhitelist = getArrayListFromReadableArray(links);
        if (null == linksToWhitelist || linksToWhitelist.isEmpty()) {
            notifyProblem(promise, null, "An array of links to be whitelisted is empty!");
            return;
        }
        one.whitelistIdentityTransferLinks(linksToWhitelist);
        notifyResult(promise, null);
    }    

    @ReactMethod
    public void getTid(Promise promise) {
        notifyResult(promise, one.getTid());
    }

    @ReactMethod
    public void clearUserProfile() {
        one.clearUserProfile();
    }

    @ReactMethod
    public void setLogLevel(Integer logLevel) {
        one.setLogLevel(logLevelFromInteger(logLevel));
    }

    @ReactMethod
    public void disableIdentityTransfer(Boolean disable) {
        one.disableIdentityTransfer(disable);
    }

    @ReactMethod
    public void disableAutomaticOutboundLinkTracking(Boolean disable) {
        one.disableAutomaticOutboundLinkTracking(disable);
    }

    @ReactMethod
    public void enablePushNotifications(Boolean enable, String senderId) {
        one.enablePushNotifications(enable, senderId);
    }

    @ReactMethod
    public void getPushToken(Promise promise) {
        if (promise == null) {
            return;
        }
        notifyResult(promise, one.getPushToken());
    }

    @ReactMethod
    public void sendPushToken(String pushToken) {
        one.sendPushToken(pushToken);
    }

    @ReactMethod
    public void identitySync() {
        this.identitySyncWithURL(null);
    }
    
    @ReactMethod
    public void identitySyncWithURL(String urlString) {
        URL url = null;
        if (urlString != null) {
            try {
                url = new URL(urlString);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        one.identitySync(url);
    }

    @Override
    public Map<String, Object> getConstants() {
        /*
        * Export the SDK's log levels to js. They will be accessible 
        * like this: One.LogLevelNone, One.LogLevelWebService,
        * One.LogLevelFramework and One.LogLevelAll 
        */
        final Map<String, Object> constants = new HashMap<>();
        constants.put("LogLevelNone", 0);
        constants.put("LogLevelWebService", 1);
        constants.put("LogLevelFramework", 2);
        constants.put("LogLevelAll", 3);
        return constants;
    }

    private OneLogLevel logLevelFromInteger(Integer logLevel) {
        OneLogLevel level;
        switch (logLevel) {
            case 0:
                level = OneLogLevel.NONE;
                break;
            case 1:
                level = OneLogLevel.WEB_SERVICE;
                break;
            case 2:
                level = OneLogLevel.FRAMEWORK;
                break;
            case 3:
                level = OneLogLevel.ALL;
                break;
            default:
                level = OneLogLevel.NONE;
        }
        return level;
    }

    private HashMap<String, String> getPropertiesFromReadableMap(ReadableMap readableMap) {
        if (readableMap != null && readableMap instanceof ReadableNativeMap) {
            ReadableMapKeySetIterator iterator = ((ReadableNativeMap) readableMap).keySetIterator();
            HashMap<String, String> hashMap = new HashMap<>();
            while (iterator.hasNextKey()) {
                String key = iterator.nextKey();
                switch (readableMap.getType(key)) {
                    case String:
                        hashMap.put(key, readableMap.getString(key));
                        break;
                }
            }
            return hashMap;
        }
        return null;
    }

    private ArrayList<String> getArrayListFromReadableArray(ReadableArray readableArray) {
        ArrayList<String> result = new ArrayList<>(readableArray.size());
        for (int index = 0; index < readableArray.size(); index++) {
            switch (readableArray.getType(index)) {
                case String: {
                    String entry = (String)readableArray.getString(index);
                    if (!entry.isEmpty()) {
                        result.add(entry);
                    }
                    break;
                }
            }
        }
        return result;
    }

    private void notifyProblem(Promise promise, Throwable throwable) {
        /*
        * We have to catch the rejection to prevent the following exception:
        *
        * java.lang.RuntimeException: Illegal callback invocation from native module. 
        * This callback type only permits a single invocation from native code.
        *
        * It's not clear at the moment why React Native throws the exception relating
        * to this module. There is an issue raised about it: 
        * https://github.com/facebook/react-native/issues/13595
        */
        try {
            promise.reject(throwable);
        } catch (RuntimeException e) {
            e.printStackTrace();
        }
    }

    private void notifyProblem(Promise promise, String code, String message) {
        try {
            promise.reject("" + code, message, null);
        } catch (RuntimeException e) {
            e.printStackTrace();
        }
    }

    private void notifyResult(Promise promise, String result) {
        try {
            promise.resolve(result);
        } catch (RuntimeException e) {
            e.printStackTrace();
        }
    }

}