//
//  RCTOne.m
//  RCTOne
//
//  Copyright © 2017 Thunderhead. All rights reserved.
//

#import "RCTOne.h"
#import "RCTConvert+OneLogLevel.h"
#import <UIKit/UIKit.h>
#import <Thunderhead/One.h>

@implementation RCTOne

RCT_EXPORT_MODULE(One)

RCT_EXPORT_METHOD(init:(NSString *)siteKey uri:(NSString *)uri apiKey:(NSString *)apiKey sharedSecret:(NSString *)sharedSecret userId:(NSString *)userId adminMode:(BOOL)adminMode hostName:(NSString *)hostName)
{
    [One startSessionWithSK:siteKey
                        uri:uri
                     apiKey:apiKey
               sharedSecret:sharedSecret
                     userId:userId
                  adminMode:adminMode
                   hostName:hostName];
    [One disableAutomaticInteractionDetection:YES];
}

RCT_EXPORT_METHOD(setLogLevel:(OneLogLevel)level)
{
    [One setLogLevel:level];
}

RCT_EXPORT_METHOD(sendInteraction:(NSString *)interaction
                  withProperties:(NSDictionary *)properties
                  resolver:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    [One sendInteraction:interaction withProperties:properties andBlock:^(NSDictionary *response, NSError *error) {
        if (error && reject) {
            reject([NSString stringWithFormat:@"%@",@(error.code)],error.localizedDescription, error);
        } else if (resolve) {
            resolve(response);
        }
    }];
}

// The parameters of this method have been swapped in this method
// declaration in order to reconcile with the ONE SDK for Android
// and Javascript.
RCT_EXPORT_METHOD(sendProperties:(NSString *)interaction 
                  forInteraction:(NSDictionary *)properties)
{
    [One sendProperties:properties forInteractionPath:interaction];
}

RCT_EXPORT_METHOD(sendBaseTouchpointProperties:(NSDictionary *)properties)
{
    [One sendBaseTouchpointProperties:properties];
}

RCT_EXPORT_METHOD(sendResponseCode:(NSString *)responseCode forInteraction:(NSString *)interaction)
{
    [One sendResponseCode:responseCode forInteractionPath:interaction];
}

RCT_EXPORT_METHOD(sendInteractionForOutboundLink:(NSString *)urlString
                  resolver:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    NSURL *url = [NSURL URLWithString:urlString];
    if (!url) {
        NSError *error = [NSError errorWithDomain:NSURLErrorDomain
                                             code:NSURLErrorBadURL
                                         userInfo:@{NSLocalizedDescriptionKey: @"Invalid URL"}];
        reject(nil, nil, error);
        return;
    }
    [One sendInteractionForOutboundLink:url];
}

RCT_EXPORT_METHOD(getURLWithOneTid:(NSString *)urlString
                  resolver:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    NSURL *url = [NSURL URLWithString:urlString];
    NSURL *urlWithOneTid = [One getURLWithOneTid:url];
    
    if (!urlWithOneTid.absoluteString && reject) {
        NSError *error = [NSError errorWithDomain:NSURLErrorDomain
                                             code:NSURLErrorBadURL
                                         userInfo:@{NSLocalizedDescriptionKey: @"Invalid URL"}];
        reject(nil, nil, error);
        return;
    }
    
    if (resolve) {
        resolve(urlWithOneTid.absoluteString);
    }
}

RCT_EXPORT_METHOD(identitySync)
{
    [self identitySyncWithURL:nil];
}

RCT_EXPORT_METHOD(identitySyncWithURL:(NSString *)urlString)
{
    NSURL *url = nil;
    if ([urlString isKindOfClass:[NSString class]]) {
        url = [NSURL URLWithString:urlString];
    }
    [One identitySyncWithURL:url];
}

RCT_EXPORT_METHOD(blacklistIdentityTransferLinks:(NSArray *)links
                  resolver:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    if (!links.count) {
        reject(nil, @"An array of links to be blacklisted is empty!", nil);
        return;
    }
    
    [One blacklistIdentityTransferLinks:links];
    
    if (resolve) {
        resolve(nil);
    }
}

RCT_EXPORT_METHOD(whitelistIdentityTransferLinks:(NSArray *)links
                  resolver:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    if (!links.count) {
        reject(nil, @"An array of links to be whitelisted is empty!", nil);
        return;
    }
    
    [One whitelistIdentityTransferLinks:links];
    
    if (resolve) {
        resolve(nil);
    }
}

RCT_EXPORT_METHOD(getTid:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    if (!resolve) {
        return;
    }
    
    resolve([One getTid]);
}

RCT_EXPORT_METHOD(clearUserProfile)
{
    [One clearUserProfile];
}

RCT_EXPORT_METHOD(disableIdentityTransfer:(BOOL)disable)
{
    [One disableIdentityTransfer:disable];
}

RCT_EXPORT_METHOD(disableAutomaticOutboundLinkTracking:(BOOL)disable)
{
    [One disableAutomaticOutboundLinkTracking:disable];
}

RCT_EXPORT_METHOD(enablePushNotifications:(BOOL)enable)
{
    [One enablePushNotifications:enable];
}

RCT_EXPORT_METHOD(getPushToken:(RCTPromiseResolveBlock)resolve
                  rejecter:(RCTPromiseRejectBlock)reject)
{
    if (!resolve) {
        return;
    }
    
    resolve([One getPushToken]);
}

RCT_EXPORT_METHOD(sendPushToken:(NSString *)pushToken)
{
    NSData *data = [self dataFromPushTokenString:pushToken];
    [One sendPushToken:data];
}

- (NSData *)dataFromPushTokenString:(NSString *)pushToken
{
    const char *ptr = [pushToken cStringUsingEncoding:NSASCIIStringEncoding];
    NSUInteger len = [pushToken length]/2;
    NSMutableData *dataPushToken = [NSMutableData dataWithCapacity:len];
    while(len--) {
        char num[5] = (char[]){ '0', 'x', 0, 0, 0 };
        num[2] = *ptr++;
        num[3] = *ptr++;
        uint8_t n = (uint8_t)strtol(num, NULL, 0);
        [dataPushToken appendBytes:&n length:1];
    }
    
    return [dataPushToken copy];
}

- (NSDictionary *)constantsToExport
{
    return @{ @"LogLevelNone" : @(kOneLogLevelNone),
              @"LogLevelAll" : @(kOneLogLevelAll),
              @"LogLevelWebService" : @(kOneLogLevelWebService),
              @"LogLevelFramework" : @(kOneLogLevelFramework)
              };
}

- (dispatch_queue_t)methodQueue
{
    return dispatch_get_main_queue();
}

@end
