"use strict";

exports.__esModule = true;
exports.default = convertFunctionRest;

var _core = require("@babel/core");

var buildRest = (0, _core.template)("\n  for (var LEN = ARGUMENTS.length,\n           ARRAY = new Array(ARRAY_LEN),\n           KEY = START;\n       KEY < LEN;\n       KEY++) {\n    ARRAY[ARRAY_KEY] = ARGUMENTS[KEY];\n  }\n");
var restIndex = (0, _core.template)("\n  (INDEX < OFFSET || ARGUMENTS.length <= INDEX) ? undefined : ARGUMENTS[INDEX]\n");
var restIndexImpure = (0, _core.template)("\n  REF = INDEX, (REF < OFFSET || ARGUMENTS.length <= REF) ? undefined : ARGUMENTS[REF]\n");
var restLength = (0, _core.template)("\n  ARGUMENTS.length <= OFFSET ? 0 : ARGUMENTS.length - OFFSET\n");

function referencesRest(path, state) {
  if (path.node.name === state.name) {
    return path.scope.bindingIdentifierEquals(state.name, state.outerBinding);
  }

  return false;
}

var memberExpressionOptimisationVisitor = {
  Scope: function Scope(path, state) {
    if (!path.scope.bindingIdentifierEquals(state.name, state.outerBinding)) {
      path.skip();
    }
  },
  Flow: function Flow(path) {
    if (path.isTypeCastExpression()) return;
    path.skip();
  },
  "Function|ClassProperty": function FunctionClassProperty(path, state) {
    var oldNoOptimise = state.noOptimise;
    state.noOptimise = true;
    path.traverse(memberExpressionOptimisationVisitor, state);
    state.noOptimise = oldNoOptimise;
    path.skip();
  },
  ReferencedIdentifier: function ReferencedIdentifier(path, state) {
    var node = path.node;

    if (node.name === "arguments") {
      state.deopted = true;
    }

    if (!referencesRest(path, state)) return;

    if (state.noOptimise) {
      state.deopted = true;
    } else {
      var parentPath = path.parentPath;

      if (parentPath.listKey === "params" && parentPath.key < state.offset) {
        return;
      }

      if (parentPath.isMemberExpression({
        object: node
      })) {
        var grandparentPath = parentPath.parentPath;
        var argsOptEligible = !state.deopted && !(grandparentPath.isAssignmentExpression() && parentPath.node === grandparentPath.node.left || grandparentPath.isLVal() || grandparentPath.isForXStatement() || grandparentPath.isUpdateExpression() || grandparentPath.isUnaryExpression({
          operator: "delete"
        }) || (grandparentPath.isCallExpression() || grandparentPath.isNewExpression()) && parentPath.node === grandparentPath.node.callee);

        if (argsOptEligible) {
          if (parentPath.node.computed) {
            if (parentPath.get("property").isBaseType("number")) {
              state.candidates.push({
                cause: "indexGetter",
                path: path
              });
              return;
            }
          } else if (parentPath.node.property.name === "length") {
            state.candidates.push({
              cause: "lengthGetter",
              path: path
            });
            return;
          }
        }
      }

      if (state.offset === 0 && parentPath.isSpreadElement()) {
        var call = parentPath.parentPath;

        if (call.isCallExpression() && call.node.arguments.length === 1) {
          state.candidates.push({
            cause: "argSpread",
            path: path
          });
          return;
        }
      }

      state.references.push(path);
    }
  },
  BindingIdentifier: function BindingIdentifier(path, state) {
    if (referencesRest(path, state)) {
      state.deopted = true;
    }
  }
};

function hasRest(node) {
  var length = node.params.length;
  return length > 0 && _core.types.isRestElement(node.params[length - 1]);
}

function optimiseIndexGetter(path, argsId, offset) {
  var offsetLiteral = _core.types.numericLiteral(offset);

  var index;

  if (_core.types.isNumericLiteral(path.parent.property)) {
    index = _core.types.numericLiteral(path.parent.property.value + offset);
  } else if (offset === 0) {
    index = path.parent.property;
  } else {
    index = _core.types.binaryExpression("+", path.parent.property, _core.types.cloneNode(offsetLiteral));
  }

  var scope = path.scope;

  if (!scope.isPure(index)) {
    var temp = scope.generateUidIdentifierBasedOnNode(index);
    scope.push({
      id: temp,
      kind: "var"
    });
    path.parentPath.replaceWith(restIndexImpure({
      ARGUMENTS: argsId,
      OFFSET: offsetLiteral,
      INDEX: index,
      REF: _core.types.cloneNode(temp)
    }));
  } else {
    var parentPath = path.parentPath;
    parentPath.replaceWith(restIndex({
      ARGUMENTS: argsId,
      OFFSET: offsetLiteral,
      INDEX: index
    }));
    var offsetTestPath = parentPath.get("test").get("left");
    var valRes = offsetTestPath.evaluate();

    if (valRes.confident) {
      if (valRes.value === true) {
        parentPath.replaceWith(parentPath.scope.buildUndefinedNode());
      } else {
        parentPath.get("test").replaceWith(parentPath.get("test").get("right"));
      }
    }
  }
}

function optimiseLengthGetter(path, argsId, offset) {
  if (offset) {
    path.parentPath.replaceWith(restLength({
      ARGUMENTS: argsId,
      OFFSET: _core.types.numericLiteral(offset)
    }));
  } else {
    path.replaceWith(argsId);
  }
}

function convertFunctionRest(path) {
  var node = path.node,
      scope = path.scope;
  if (!hasRest(node)) return false;
  var rest = node.params.pop().argument;

  var argsId = _core.types.identifier("arguments");

  var state = {
    references: [],
    offset: node.params.length,
    argumentsNode: argsId,
    outerBinding: scope.getBindingIdentifier(rest.name),
    candidates: [],
    name: rest.name,
    deopted: false
  };
  path.traverse(memberExpressionOptimisationVisitor, state);

  if (!state.deopted && !state.references.length) {
    var _arr = state.candidates;

    for (var _i = 0; _i < _arr.length; _i++) {
      var _ref = _arr[_i];
      var _path = _ref.path,
          cause = _ref.cause;

      var clonedArgsId = _core.types.cloneNode(argsId);

      switch (cause) {
        case "indexGetter":
          optimiseIndexGetter(_path, clonedArgsId, state.offset);
          break;

        case "lengthGetter":
          optimiseLengthGetter(_path, clonedArgsId, state.offset);
          break;

        default:
          _path.replaceWith(clonedArgsId);

      }
    }

    return true;
  }

  state.references = state.references.concat(state.candidates.map(function (_ref2) {
    var path = _ref2.path;
    return path;
  }));

  var start = _core.types.numericLiteral(node.params.length);

  var key = scope.generateUidIdentifier("key");
  var len = scope.generateUidIdentifier("len");
  var arrKey, arrLen;

  if (node.params.length) {
    arrKey = _core.types.binaryExpression("-", _core.types.cloneNode(key), _core.types.cloneNode(start));
    arrLen = _core.types.conditionalExpression(_core.types.binaryExpression(">", _core.types.cloneNode(len), _core.types.cloneNode(start)), _core.types.binaryExpression("-", _core.types.cloneNode(len), _core.types.cloneNode(start)), _core.types.numericLiteral(0));
  } else {
    arrKey = _core.types.identifier(key.name);
    arrLen = _core.types.identifier(len.name);
  }

  var loop = buildRest({
    ARGUMENTS: argsId,
    ARRAY_KEY: arrKey,
    ARRAY_LEN: arrLen,
    START: start,
    ARRAY: rest,
    KEY: key,
    LEN: len
  });

  if (state.deopted) {
    node.body.body.unshift(loop);
  } else {
    var target = path.getEarliestCommonAncestorFrom(state.references).getStatementParent();
    target.findParent(function (path) {
      if (path.isLoop()) {
        target = path;
      } else {
        return path.isFunction();
      }
    });
    target.insertBefore(loop);
  }

  return true;
}