'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = buildWall;

var _three = require('three');

var Three = _interopRequireWildcard(_three);

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) newObj[key] = obj[key]; } } newObj.default = obj; return newObj; } }

function buildWall(element, layer, scene, textures) {

  // Get the two vertices of the wall
  var vertex0 = layer.vertices.get(element.vertices.get(0));
  var vertex1 = layer.vertices.get(element.vertices.get(1));
  var inverted = false;

  // The first vertex is the smaller one
  if (vertex0.x > vertex1.x) {
    var app = vertex0;
    vertex0 = vertex1;
    vertex1 = app;
    inverted = true;
  }

  // Get height and thickness of the wall converting them into the current scene units
  var height = element.properties.get('height').get('length');
  var thickness = element.properties.get('thickness').get('length');

  var bevelRadius = thickness; // This is useful for linking two walls together

  // Get holes data for this wall
  var holes = [];
  element.holes.forEach(function (holeID) {
    var hole = layer.holes.get(holeID);
    holes.push(hole);
  });

  /*
   * First of all I need to build the wall shape. We can build it drawing a rectangle
   * with the left bottom vertex in the origin and where the length is given by the distance between the
   * two vertices and the height is the height of the wall
   */

  // Compute the distance between the two vertices
  var distance = Math.sqrt(Math.pow(vertex0.x - vertex1.x, 2) + Math.pow(vertex0.y - vertex1.y, 2));

  // I use this workaround to link two adjacent walls together
  distance += bevelRadius; //TODO: REMOVE WORKAROUND BEVELING

  // Now I can build the coordinates of the wall shape:

  /**
   *
   *  (bevelRadius/2,height)     (distance,height)
   *      x------------------x
   *      |                  |            ^ y
   *      |                  |            |
   *      |                  |            |
   *      |                  |          --|-----> x
   *      |                  |
   *      |                  |
   *      x------------------x
   *    (bevelRadius/2,0)            (distance,0)
   *
   */

  var wallCoords = [[0, 0], [distance, 0], [distance, height], [0, height]];

  // Now I compute the rotation angle of the wall (see the bevel radius trick)
  var alpha = Math.asin((vertex1.y - vertex0.y) / (distance - bevelRadius)); //TODO: REMOVE WORKAROUND BEVELING

  // We will rotate on the bottom-left vertex, so we need a pivot in the origin
  var pivot = new Three.Object3D();

  // Create a Three.Shape from the coordinates
  var rectShape = createShape(wallCoords);

  // Now we have to create the holes for the wall
  holes.forEach(function (holeData) {

    // Get the sizes of the holes converting them to the scene units
    var holeWidth = holeData.properties.get('width').get('length');
    var holeHeight = holeData.properties.get('height').get('length');
    var holeAltitude = holeData.properties.get('altitude').get('length');
    var offset = inverted ? 1 - holeData.offset : holeData.offset;

    // Now I can build the coordinates of the hole shape:

    // The starting point is at (distance - bevelRadius) * offset + bevelRadius / 2 - width / 2 and is called startAt

    /**
     *
     *  (startAt,holeAltitude + holeHeight)     (holeWidth + startAt,holeAltitude + holeHeight)
     *                x-------------------------------------------x
     *                |                                           |
     *                |                                           |
     *                |                                           |
     *                |                                           |
     *                |                                           |
     *                x-------------------------------------------x
     *      (startAt,holeAltitude)                (holeWidth + startAt,holeAltitude)
     *
     */

    var startAt = (distance - bevelRadius) * offset - holeWidth / 2 + bevelRadius / 2;

    // I add 0.00001 to the holeAltitude to avoid a warning with the Three triangulation algorithm
    var holeCoords = [[startAt, holeAltitude + 0.00001], [holeWidth + startAt, holeAltitude + 0.00001], [holeWidth + startAt, holeHeight + holeAltitude], [startAt, holeHeight + holeAltitude]];

    // Now I can create the Three.shape of the hole and, push it into the wall shape holes
    var holeShape = createShape(holeCoords);
    rectShape.holes.push(holeShape);

    // At this point I can create a set of rectangles which surrounds the hole

    var holeClosures = buildShapeClosures({ x: 0, y: 0 }, { x: holeWidth, y: 0 }, holeHeight, thickness);

    var topHoleClosureCoords = holeClosures.topShape;
    var topHoleShape = createShape(topHoleClosureCoords);

    var leftHoleClosureCoords = holeClosures.leftShape;
    var leftHoleShape = createShape(leftHoleClosureCoords);

    var topHoleClosureGeometry = new Three.ShapeGeometry(topHoleShape);
    var leftHoleClosureGeometry = new Three.ShapeGeometry(leftHoleShape);

    var topHoleClosure = new Three.Mesh(topHoleClosureGeometry, new Three.MeshLambertMaterial({
      side: Three.DoubleSide
    }));

    var leftHoleClosure = new Three.Mesh(leftHoleClosureGeometry, new Three.MeshLambertMaterial({
      side: Three.DoubleSide
    }));

    var rightHoleClosure = new Three.Mesh(leftHoleClosureGeometry, new Three.MeshLambertMaterial({
      side: Three.DoubleSide
    }));

    topHoleClosure.rotation.x += Math.PI / 2;
    topHoleClosure.position.z -= thickness / 2;
    topHoleClosure.position.y += holeHeight + holeAltitude;
    topHoleClosure.position.x = startAt - bevelRadius / 2;

    leftHoleClosure.rotation.y -= Math.PI / 2;
    leftHoleClosure.position.z -= thickness / 2;
    leftHoleClosure.position.y += holeAltitude;
    leftHoleClosure.position.x = startAt - bevelRadius / 2;

    rightHoleClosure.rotation.y -= Math.PI / 2;
    rightHoleClosure.position.z -= thickness / 2;
    rightHoleClosure.position.y += holeAltitude;
    rightHoleClosure.position.x = startAt + holeWidth - bevelRadius / 2;

    pivot.add(topHoleClosure);
    pivot.add(leftHoleClosure);
    pivot.add(rightHoleClosure);

    if (holeAltitude !== 0) {

      var bottomHoleClosure = new Three.Mesh(topHoleClosureGeometry, new Three.MeshLambertMaterial({
        side: Three.DoubleSide
      }));

      bottomHoleClosure.rotation.x += Math.PI / 2;
      bottomHoleClosure.position.z -= thickness / 2;
      bottomHoleClosure.position.y += holeAltitude;
      bottomHoleClosure.position.x = startAt - bevelRadius / 2;

      pivot.add(bottomHoleClosure);
    }
  });

  // Now I can create the geometry of a single face of the wall
  var wallGeometry = new Three.ShapeGeometry(rectShape);
  wallGeometry.computeVertexNormals();

  // I define two materials (one for every face of the wall)
  var wallMaterial1 = new Three.MeshPhongMaterial({
    side: Three.BackSide
  });

  var wallMaterial2 = new Three.MeshPhongMaterial({
    side: Three.FrontSide
  });

  // I can choose the correct texture observing the angle of the wall
  if (alpha < 0) {
    applyTexture(wallMaterial1, element.properties.get('textureB'), distance, height, textures);
    applyTexture(wallMaterial2, element.properties.get('textureA'), distance, height, textures);
  } else {
    applyTexture(wallMaterial1, element.properties.get('textureA'), distance, height, textures);
    applyTexture(wallMaterial2, element.properties.get('textureB'), distance, height, textures);
  }

  // Assign the correct UV coordinates
  assignUVs(wallGeometry);

  var wall1 = new Three.Mesh(wallGeometry, wallMaterial1);
  var wall2 = new Three.Mesh(wallGeometry, wallMaterial2);

  // Expand the wall at the center
  wall1.position.z -= thickness / 2;
  wall2.position.z += thickness / 2;

  // Change walls x position to link two adjacent walls
  wall1.position.x -= bevelRadius / 2; //TODO: REMOVE WORKAROUND BEVELING
  wall2.position.x -= bevelRadius / 2; //TODO: REMOVE WORKAROUND BEVELING

  // Rotate the wall around the bottom-left vertex
  pivot.rotation.y = alpha;

  // Add the two wall faces to the pivot
  pivot.add(wall1);
  pivot.add(wall2);

  // Build closures for wall

  var closures = buildShapeClosures({ x: 0, y: 0 }, { x: distance, y: 0 }, height, thickness);

  var topClosureCoords = closures.topShape;
  var topShape = createShape(topClosureCoords);

  var leftClosureCoords = closures.leftShape;
  var leftShape = createShape(leftClosureCoords);

  var topClosureGeometry = new Three.ShapeGeometry(topShape);
  var leftClosureGeometry = new Three.ShapeGeometry(leftShape);

  var topClosure = new Three.Mesh(topClosureGeometry, new Three.MeshLambertMaterial({
    side: Three.BackSide
  }));

  var leftClosure = new Three.Mesh(leftClosureGeometry, new Three.MeshLambertMaterial({
    side: Three.FrontSide
  }));

  var rightClosure = new Three.Mesh(leftClosureGeometry, new Three.MeshLambertMaterial({
    side: Three.BackSide
  }));

  // Move the wall closures
  topClosure.rotation.x += Math.PI / 2;
  topClosure.position.z -= thickness / 2;
  topClosure.position.y += height;

  topClosure.position.x -= bevelRadius / 2; //TODO: REMOVE WORKAROUND BEVELING

  leftClosure.rotation.y -= Math.PI / 2;
  leftClosure.position.z -= bevelRadius / 2;

  leftClosure.position.x -= bevelRadius / 2 - 1; //TODO: REMOVE WORKAROUND BEVELING

  rightClosure.rotation.y -= Math.PI / 2;
  rightClosure.position.z -= thickness / 2;
  rightClosure.position.x += distance - 1;

  rightClosure.position.x -= bevelRadius / 2; //TODO: REMOVE WORKAROUND BEVELING

  pivot.add(topClosure);
  pivot.add(leftClosure);
  pivot.add(rightClosure);

  // If the wall is selected show a bounding box around it
  if (element.selected) {
    var box1 = new Three.BoxHelper(wall1, 0x99c3fb);
    box1.material.depthTest = false;
    box1.material.linewidth = 2;
    box1.material.vertexColor = Three.VertexColors;
    box1.renderOrder = 1000;
    pivot.add(box1);

    var box2 = new Three.BoxHelper(wall2, 0x99c3fb);
    box2.material.depthTest = false;
    box2.material.linewidth = 2;
    box2.renderOrder = 1000;
    pivot.add(box2);

    var box3 = new Three.BoxHelper(topClosure, 0x99c3fb);
    box3.material.depthTest = false;
    box3.material.linewidth = 2;
    box3.renderOrder = 1000;
    pivot.add(box3);

    var box4 = new Three.BoxHelper(leftClosure, 0x99c3fb);
    box4.material.depthTest = false;
    box4.material.linewidth = 2;
    box4.renderOrder = 1000;
    pivot.add(box4);

    var box5 = new Three.BoxHelper(rightClosure, 0x99c3fb);
    box5.material.depthTest = false;
    box5.material.linewidth = 2;
    box5.renderOrder = 1000;
    pivot.add(box5);
  }

  return Promise.resolve(pivot);
}

/**
 * This function build the closures around the holes and the walls
 * @param vertex0: Start vertex
 * @param vertex1: End vertex
 * @param height: Height of the shape
 * @param thickness: Thickness of the closure
 * @returns {{topShape: *[], leftShape: *[]}}: The left and top shape (the others can be computed from these two)
 */
function buildShapeClosures(vertex0, vertex1, height, thickness) {

  var distance = Math.sqrt(Math.pow(vertex0.x - vertex1.x, 2) + Math.pow(vertex0.y - vertex1.y, 2));

  var topShape = [[vertex0.x, vertex0.y], [vertex0.x + distance, vertex0.y], [vertex0.x + distance, vertex0.y + thickness], [vertex0.x, vertex0.y + thickness]];

  var leftShape = [[vertex0.x, vertex0.y], [vertex0.x + thickness, vertex0.y], [vertex0.x + thickness, vertex0.y + height], [vertex0.x, vertex0.y + height]];

  return { topShape: topShape, leftShape: leftShape };
}

/**
 * This function will create a shape given a list of coordinates
 * @param shapeCoords
 * @returns {THREE.Shape}
 */
function createShape(shapeCoords) {
  var shape = new Three.Shape();
  shape.moveTo(shapeCoords[0][0], shapeCoords[0][1]);
  for (var i = 1; i < shapeCoords.length; i++) {
    shape.lineTo(shapeCoords[i][0], shapeCoords[i][1]);
  }
  return shape;
}

/**
 * Apply a texture to a wall face
 * @param material: The material of the face
 * @param textureName: The name of the texture to load
 * @param length: The lenght of the face
 * @param height: The height of the face
 * @param textures: The list of textures available for this wall
 */
function applyTexture(material, textureName, length, height, textures) {

  var loader = new Three.TextureLoader();

  var textureParams = textures[textureName];

  if (textureParams) {
    material.map = loader.load(textureParams.uri);
    material.needsUpdate = true;
    material.map.wrapS = Three.RepeatWrapping;
    material.map.wrapT = Three.RepeatWrapping;
    material.map.repeat.set(length * textureParams.lengthRepeatScale, height * textureParams.heightRepeatScale);

    if (textureParams.normal) {
      material.normalMap = loader.load(textureParams.normal.uri);
      material.normalScale = new Three.Vector2(textureParams.normal.normalScaleX, textureParams.normal.normalScaleY);
      material.normalMap.wrapS = Three.RepeatWrapping;
      material.normalMap.wrapT = Three.RepeatWrapping;
      material.normalMap.repeat.set(length * textureParams.normal.lengthRepeatScale, height * textureParams.normal.heightRepeatScale);
    }
  }
}

/**
 * Function that assign UV coordinates to a geometry
 * @param geometry
 */
function assignUVs(geometry) {
  geometry.computeBoundingBox();

  var max = geometry.boundingBox.max;
  var min = geometry.boundingBox.min;

  var offset = new Three.Vector2(0 - min.x, 0 - min.y);
  var range = new Three.Vector2(max.x - min.x, max.y - min.y);

  geometry.faceVertexUvs[0] = [];
  var faces = geometry.faces;

  for (var i = 0; i < geometry.faces.length; i++) {

    var v1 = geometry.vertices[faces[i].a];
    var v2 = geometry.vertices[faces[i].b];
    var v3 = geometry.vertices[faces[i].c];

    geometry.faceVertexUvs[0].push([new Three.Vector2((v1.x + offset.x) / range.x, (v1.y + offset.y) / range.y), new Three.Vector2((v2.x + offset.x) / range.x, (v2.y + offset.y) / range.y), new Three.Vector2((v3.x + offset.x) / range.x, (v3.y + offset.y) / range.y)]);
  }
  geometry.uvsNeedUpdate = true;
}