/**
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * 
 */

'use strict';









const HasteMap = require('jest-haste-map');
const Resolver = require('jest-resolve');

const fs = require('graceful-fs');
const path = require('path');
const shouldInstrument = require('./shouldInstrument');
const transform = require('./transform');
const utils = require('jest-util');





















const NODE_MODULES = path.sep + 'node_modules' + path.sep;
const SNAPSHOT_EXTENSION = 'snap';

const getModuleNameMapper = config => {
  if (config.moduleNameMapper.length) {
    const moduleNameMapper = Object.create(null);
    config.moduleNameMapper.forEach(
    map => moduleNameMapper[map[1]] = new RegExp(map[0]));

    return moduleNameMapper;
  }
  return null;
};

const mockParentModule = {
  exports: {},
  filename: 'mock.js',
  id: 'mockParent' };


const normalizedIDCache = Object.create(null);
const unmockRegExpCache = new WeakMap();



class Runtime {





















  constructor(
  config,
  environment,
  resolver)
  {
    this._moduleRegistry = Object.create(null);
    this._internalModuleRegistry = Object.create(null);
    this._mockRegistry = Object.create(null);
    this._config = config;
    this._environment = environment;
    this._resolver = resolver;
    this._moduleMocker = this._environment.moduleMocker;

    this._currentlyExecutingModulePath = '';
    this._explicitShouldMock = Object.create(null);
    this._isCurrentlyExecutingManualMock = null;
    this._mockFactories = Object.create(null);
    this._mocksPattern =
    config.mocksPattern ? new RegExp(config.mocksPattern) : null;
    this._shouldAutoMock = config.automock;
    this._testRegex = new RegExp(config.testRegex.replace(/\//g, path.sep));
    this._virtualMocks = Object.create(null);

    this._mockMetaDataCache = Object.create(null);
    this._shouldMockModuleCache = Object.create(null);
    this._shouldUnmockTransitiveDependenciesCache = Object.create(null);
    this._transitiveShouldMock = Object.create(null);

    this._unmockList = unmockRegExpCache.get(config);
    if (
    !this._unmockList &&
    config.automock &&
    config.unmockedModulePathPatterns)
    {
      this._unmockList =
      new RegExp(config.unmockedModulePathPatterns.join('|'));
      unmockRegExpCache.set(config, this._unmockList);
    }

    if (config.automock) {
      config.setupFiles.forEach(filePath => {
        if (filePath && filePath.includes(NODE_MODULES)) {
          const moduleID = this._normalizeID(filePath);
          this._transitiveShouldMock[moduleID] = false;
        }
      });
    }

    this.resetModules();

    if (config.setupFiles.length) {
      for (let i = 0; i < config.setupFiles.length; i++) {
        this.requireModule(config.setupFiles[i]);
      }
    }
  }

  static shouldInstrument(filename, config) {
    return shouldInstrument(filename, config);
  }

  static transformSource(
  filename,
  config,
  content,
  instrument)
  {
    return transform.transformSource(filename, config, content, instrument);
  }

  static createHasteContext(
  config,
  options)



  {
    utils.createDirectory(config.cacheDirectory);
    const instance = Runtime.createHasteMap(config, {
      console: options.console,
      maxWorkers: options.maxWorkers,
      resetCache: !config.cache });

    return instance.build().then(
    hasteMap => ({
      hasteFS: hasteMap.hasteFS,
      resolver: Runtime.createResolver(config, hasteMap.moduleMap) }),

    error => {
      throw error;
    });

  }

  static createHasteMap(
  config,
  options)
  {
    const ignorePattern = new RegExp(
    [config.cacheDirectory].concat(config.modulePathIgnorePatterns).join('|'));


    return new HasteMap({
      cacheDirectory: config.cacheDirectory,
      console: options && options.console,
      extensions: [SNAPSHOT_EXTENSION].concat(config.moduleFileExtensions),
      ignorePattern,
      maxWorkers: options && options.maxWorkers || 1,
      mocksPattern: config.mocksPattern,
      name: config.name,
      platforms: config.haste.platforms || ['ios', 'android'],
      providesModuleNodeModules: config.haste.providesModuleNodeModules,
      resetCache: options && options.resetCache,
      retainAllFiles: false,
      roots: config.testPathDirs,
      useWatchman: config.watchman });

  }

  static createResolver(
  config,
  moduleMap)
  {
    return new Resolver(moduleMap, {
      browser: config.browser,
      defaultPlatform: config.haste.defaultPlatform,
      extensions: config.moduleFileExtensions.map(extension => '.' + extension),
      hasCoreModules: true,
      moduleDirectories: config.moduleDirectories,
      moduleNameMapper: getModuleNameMapper(config),
      modulePaths: config.modulePaths,
      platforms: config.haste.platforms });

  }

  static runCLI(args, info) {
    return require('./cli').run(args, info);
  }

  static getCLIOptions() {
    return require('./cli/args').options;
  }

  requireModule(
  from,
  moduleName,
  options)
  {
    const moduleID = this._normalizeID(from, moduleName);
    let modulePath;

    const moduleRegistry = !options || !options.isInternalModule ?
    this._moduleRegistry :
    this._internalModuleRegistry;

    // Some old tests rely on this mocking behavior. Ideally we'll change this
    // to be more explicit.
    const moduleResource = moduleName && this._resolver.getModule(moduleName);
    const manualMock =
    moduleName && this._resolver.getMockModule(from, moduleName);
    if (
    (!options || !options.isInternalModule) &&
    !moduleResource &&
    manualMock &&
    manualMock !== this._isCurrentlyExecutingManualMock &&
    this._explicitShouldMock[moduleID] !== false)
    {
      modulePath = manualMock;
    }

    if (moduleName && this._resolver.isCoreModule(moduleName)) {
      // $FlowFixMe
      return require(moduleName);
    }

    if (!modulePath) {
      modulePath = this._resolveModule(from, moduleName);
    }

    if (!moduleRegistry[modulePath]) {
      // We must register the pre-allocated module object first so that any
      // circular dependencies that may arise while evaluating the module can
      // be satisfied.
      const localModule = {
        exports: {},
        filename: modulePath,
        id: modulePath };

      moduleRegistry[modulePath] = localModule;
      if (path.extname(modulePath) === '.json') {
        localModule.exports = this._environment.global.JSON.parse(
        fs.readFileSync(modulePath, 'utf8'));

      } else if (path.extname(modulePath) === '.node') {
        // $FlowFixMe
        localModule.exports = require(modulePath);
      } else {
        this._execModule(localModule, options);
      }
    }
    return moduleRegistry[modulePath].exports;
  }

  requireInternalModule(from, to) {
    return this.requireModule(from, to, { isInternalModule: true });
  }

  requireMock(from, moduleName) {
    const moduleID = this._normalizeID(from, moduleName);

    if (this._mockRegistry[moduleID]) {
      return this._mockRegistry[moduleID];
    }

    if (moduleID in this._mockFactories) {
      return this._mockRegistry[moduleID] = this._mockFactories[moduleID]();
    }

    let manualMock = this._resolver.getMockModule(from, moduleName);
    let modulePath;
    if (manualMock) {
      modulePath = this._resolveModule(from, manualMock);
    } else {
      modulePath = this._resolveModule(from, moduleName);

      // If the actual module file has a __mocks__ dir sitting immediately next
      // to it, look to see if there is a manual mock for this file.
      //
      // subDir1/MyModule.js
      // subDir1/__mocks__/MyModule.js
      // subDir2/MyModule.js
      // subDir2/__mocks__/MyModule.js
      //
      // Where some other module does a relative require into each of the
      // respective subDir{1,2} directories and expects a manual mock
      // corresponding to that particular MyModule.js file.
      const moduleDir = path.dirname(modulePath);
      const moduleFileName = path.basename(modulePath);
      const potentialManualMock =
      path.join(moduleDir, '__mocks__', moduleFileName);
      if (fs.existsSync(potentialManualMock)) {
        manualMock = true;
        modulePath = potentialManualMock;
      }
    }

    if (manualMock) {
      const localModule = {
        exports: {},
        filename: modulePath,
        id: modulePath };

      this._execModule(localModule);
      this._mockRegistry[moduleID] = localModule.exports;
    } else {
      // Look for a real module to generate an automock from
      this._mockRegistry[moduleID] = this._generateMock(from, moduleName);
    }

    return this._mockRegistry[moduleID];
  }

  requireModuleOrMock(from, moduleName) {
    if (this._shouldMock(from, moduleName)) {
      return this.requireMock(from, moduleName);
    } else {
      return this.requireModule(from, moduleName);
    }
  }

  resetModules() {
    this._mockRegistry = Object.create(null);
    this._moduleRegistry = Object.create(null);

    if (this._environment && this._environment.global) {
      const envGlobal = this._environment.global;
      Object.keys(envGlobal).forEach(key => {
        const globalMock = envGlobal[key];
        if (
        typeof globalMock === 'object' && globalMock !== null ||
        typeof globalMock === 'function')
        {
          globalMock._isMockFunction && globalMock.mockClear();
        }
      });

      if (envGlobal.mockClearTimers) {
        envGlobal.mockClearTimers();
      }
    }
  }

  getAllCoverageInfo() {
    return this._environment.global.__coverage__;
  }

  setMock(
  from,
  moduleName,
  mockFactory,
  options)
  {
    if (options && options.virtual) {
      const mockPath = this._getVirtualMockPath(from, moduleName);
      this._virtualMocks[mockPath] = true;
    }
    const moduleID = this._normalizeID(from, moduleName);
    this._explicitShouldMock[moduleID] = true;
    this._mockFactories[moduleID] = mockFactory;
  }

  clearAllMocks() {
    this._moduleMocker.clearAllMocks();
  }

  _resolveModule(from, to) {
    return to ? this._resolver.resolveModule(from, to) : from;
  }

  _execModule(localModule, options) {
    // If the environment was disposed, prevent this module from being executed.
    if (!this._environment.global) {
      return;
    }

    const isInternalModule = !!(options && options.isInternalModule);
    const filename = localModule.filename;
    const lastExecutingModulePath = this._currentlyExecutingModulePath;
    this._currentlyExecutingModulePath = filename;
    const origCurrExecutingManualMock = this._isCurrentlyExecutingManualMock;
    this._isCurrentlyExecutingManualMock = filename;

    const dirname = path.dirname(filename);
    localModule.children = [];
    localModule.parent = mockParentModule;
    localModule.paths = this._resolver.getModulePaths(dirname);
    localModule.require = this._createRequireImplementation(filename, options);

    const script = transform(filename, this._config, { isInternalModule });

    const wrapper = this._runScript(script, filename);
    wrapper.call(
    localModule.exports, // module context
    localModule, // module object
    localModule.exports, // module exports
    localModule.require, // require implementation
    dirname, // __dirname
    filename, // __filename
    this._environment.global, // global object
    this._createRuntimeFor(filename));


    this._isCurrentlyExecutingManualMock = origCurrExecutingManualMock;
    this._currentlyExecutingModulePath = lastExecutingModulePath;
  }

  _runScript(script, filename) {
    try {
      return this._environment.runScript(script)[
      transform.EVAL_RESULT_VARIABLE];

    } catch (e) {
      const config = this._config;
      const relative = filePath => path.relative(config.rootDir, filePath);
      if (e.constructor.name === 'SyntaxError') {
        const hasPreprocessor = config.scriptPreprocessor;
        const preprocessorInfo = hasPreprocessor ?
        relative(config.scriptPreprocessor) :
        `No preprocessor specified, consider installing 'babel-jest'`;
        const babelInfo = config.usesBabelJest ?
        `Make sure your '.babelrc' is set up correctly, ` +
        `for example it should include the 'es2015' preset.\n` :
        '';
        /* eslint-disable max-len */
        throw new SyntaxError(
        `${ e.message } in file '${ relative(filename) }'.\n\n` +
        `Make sure your preprocessor is set up correctly and ensure ` +
        `your 'preprocessorIgnorePatterns' configuration is correct: http://facebook.github.io/jest/docs/configuration.html#preprocessorignorepatterns-array-string\n` +
        'If you are currently setting up Jest or modifying your preprocessor, try `jest --no-cache`.\n' +
        `Preprocessor: ${ preprocessorInfo }.\n${ babelInfo }`);

        /* eslint-enable max-len */
      }
      throw e;
    }
  }

  _generateMock(from, moduleName) {
    const modulePath = this._resolveModule(from, moduleName);

    if (!(modulePath in this._mockMetaDataCache)) {
      // This allows us to handle circular dependencies while generating an
      // automock
      this._mockMetaDataCache[modulePath] =
      this._moduleMocker.getMetadata({});

      // In order to avoid it being possible for automocking to potentially
      // cause side-effects within the module environment, we need to execute
      // the module in isolation. This could cause issues if the module being
      // mocked has calls into side-effectful APIs on another module.
      const origMockRegistry = this._mockRegistry;
      const origModuleRegistry = this._moduleRegistry;
      this._mockRegistry = Object.create(null);
      this._moduleRegistry = Object.create(null);

      const moduleExports = this.requireModule(from, moduleName);

      // Restore the "real" module/mock registries
      this._mockRegistry = origMockRegistry;
      this._moduleRegistry = origModuleRegistry;

      const mockMetadata = this._moduleMocker.getMetadata(moduleExports);
      if (mockMetadata == null) {
        throw new Error(
        `Failed to get mock metadata: ${ modulePath }\n\n` +
        `See: http://facebook.github.io/jest/docs/manual-mocks.html#content`);

      }
      this._mockMetaDataCache[modulePath] = mockMetadata;
    }
    return this._moduleMocker.generateFromMetadata(
    this._mockMetaDataCache[modulePath]);

  }

  _normalizeID(from, moduleName) {
    if (!moduleName) {
      moduleName = '';
    }

    const key = from + path.delimiter + moduleName;
    if (normalizedIDCache[key]) {
      return normalizedIDCache[key];
    }

    let moduleType;
    let mockPath = null;
    let absolutePath = null;

    if (this._resolver.isCoreModule(moduleName)) {
      moduleType = 'node';
      absolutePath = moduleName;
    } else {
      moduleType = 'user';
      if (
      !this._resolver.getModule(moduleName) &&
      !this._resolver.getMockModule(from, moduleName))
      {
        if (moduleName) {
          const virtualMockPath = this._getVirtualMockPath(from, moduleName);
          if (virtualMockPath in this._virtualMocks) {
            absolutePath = virtualMockPath;
          }
        }

        if (absolutePath === null) {
          absolutePath = this._resolveModule(from, moduleName);
        }
      }

      if (absolutePath === null) {
        const moduleResource = this._resolver.getModule(moduleName);
        if (moduleResource) {
          absolutePath = moduleResource;
        }
      }

      if (mockPath === null) {
        const mockResource = this._resolver.getMockModule(from, moduleName);
        if (mockResource) {
          mockPath = mockResource;
        }
      }
    }

    const sep = path.delimiter;
    const id = moduleType + sep + (absolutePath || '') + sep + (mockPath || '');
    return normalizedIDCache[key] = id;
  }

  _getVirtualMockPath(from, moduleName) {
    if (moduleName[0] !== '.' && moduleName[0] !== '/') {
      return moduleName;
    }
    return path.normalize(path.dirname(from) + '/' + moduleName);
  }

  _shouldMock(from, moduleName) {
    const mockPath = this._getVirtualMockPath(from, moduleName);
    if (mockPath in this._virtualMocks) {
      return true;
    }

    const explicitShouldMock = this._explicitShouldMock;
    const moduleID = this._normalizeID(from, moduleName);
    const key = from + path.delimiter + moduleID;

    if (moduleID in explicitShouldMock) {
      return explicitShouldMock[moduleID];
    }

    if (
    !this._shouldAutoMock ||
    this._resolver.isCoreModule(moduleName) ||
    this._shouldUnmockTransitiveDependenciesCache[key])
    {
      return false;
    }

    if (moduleID in this._shouldMockModuleCache) {
      return this._shouldMockModuleCache[moduleID];
    }

    const manualMock = this._resolver.getMockModule(from, moduleName);
    let modulePath;
    try {
      modulePath = this._resolveModule(from, moduleName);
    } catch (e) {
      if (manualMock) {
        this._shouldMockModuleCache[moduleID] = true;
        return true;
      }
      throw e;
    }

    if (this._unmockList && this._unmockList.test(modulePath)) {
      this._shouldMockModuleCache[moduleID] = false;
      return false;
    }

    // transitive unmocking for package managers that store flat packages (npm3)
    const currentModuleID = this._normalizeID(from);
    if (
    this._transitiveShouldMock[currentModuleID] === false ||
    from.includes(NODE_MODULES) &&
    modulePath.includes(NODE_MODULES) && (

    this._unmockList && this._unmockList.test(from) ||
    explicitShouldMock[currentModuleID] === false))


    {
      this._transitiveShouldMock[moduleID] = false;
      this._shouldUnmockTransitiveDependenciesCache[key] = true;
      return false;
    }

    return this._shouldMockModuleCache[moduleID] = true;
  }

  _createRequireImplementation(
  from,
  options)
  {
    const moduleRequire = options && options.isInternalModule ?
    moduleName => this.requireInternalModule(from, moduleName) :
    this.requireModuleOrMock.bind(this, from);
    moduleRequire.cache = Object.create(null);
    moduleRequire.extensions = Object.create(null);
    moduleRequire.requireActual = this.requireModule.bind(this, from);
    moduleRequire.requireMock = this.requireMock.bind(this, from);
    moduleRequire.resolve = moduleName => this._resolveModule(from, moduleName);
    return moduleRequire;
  }

  _createRuntimeFor(from) {
    const disableAutomock = () => {
      this._shouldAutoMock = false;
      return runtime;
    };
    const enableAutomock = () => {
      this._shouldAutoMock = true;
      return runtime;
    };
    const unmock = moduleName => {
      const moduleID = this._normalizeID(from, moduleName);
      if (
      !this._shouldAutoMock &&
      this._config.automock === false &&
      this._explicitShouldMock[moduleID] !== true)
      {
        this._environment.global.console.warn(
        `jest.unmock('${ moduleName }') was called but automocking is ` +
        `disabled. Remove the unnecessary call to \`jest.unmock\` or ` +
        `enable automocking for this test via \`jest.enableAutomock();\`. ` +
        `This warning is likely a result of a default configuration change ` +
        `in Jest 15.\n\n` +
        `Release Blog Post: https://facebook.github.io/jest/blog/2016/09/01/jest-15.html`);

      }
      this._explicitShouldMock[moduleID] = false;
      return runtime;
    };
    const deepUnmock = moduleName => {
      const moduleID = this._normalizeID(from, moduleName);
      this._explicitShouldMock[moduleID] = false;
      this._transitiveShouldMock[moduleID] = false;
      return runtime;
    };
    const mock = (
    moduleName,
    mockFactory,
    options) =>
    {
      if (mockFactory !== undefined) {
        return setMockFactory(moduleName, mockFactory, options);
      }

      const moduleID = this._normalizeID(from, moduleName);
      this._explicitShouldMock[moduleID] = true;
      return runtime;
    };
    const setMockFactory = (moduleName, mockFactory, options) => {
      this.setMock(from, moduleName, mockFactory, options);
      return runtime;
    };
    const clearAllMocks = () => {
      this.clearAllMocks();
      return runtime;
    };
    const useFakeTimers = () => {
      this._environment.fakeTimers.useFakeTimers();
      return runtime;
    };
    const useRealTimers = () => {
      this._environment.fakeTimers.useRealTimers();
      return runtime;
    };
    const resetModules = () => {
      this.resetModules();
      return runtime;
    };

    const runtime = {
      addMatchers: matchers => {
        const jasmine = this._environment.global.jasmine;
        const addMatchers =
        jasmine.addMatchers || jasmine.getEnv().currentSpec.addMatchers;
        addMatchers(matchers);
      },

      autoMockOff: disableAutomock,
      disableAutomock,

      autoMockOn: enableAutomock,
      enableAutomock,

      clearAllTimers: () => this._environment.fakeTimers.clearAllTimers(),

      dontMock: unmock,
      unmock,

      genMockFromModule: moduleName => {
        return this._generateMock(from, moduleName);
      },
      genMockFunction:
      this._moduleMocker.getMockFunction.bind(this._moduleMocker),
      genMockFn: this._moduleMocker.getMockFunction.bind(this._moduleMocker),
      fn: impl => {
        const fn = this._moduleMocker.getMockFunction();
        if (impl) {
          return fn.mockImplementation(impl);
        }
        return fn;
      },
      isMockFunction: this._moduleMocker.isMockFunction,

      doMock: mock,
      mock,

      clearAllMocks,

      resetModules,
      resetModuleRegistry: resetModules,

      runAllTicks: () => this._environment.fakeTimers.runAllTicks(),
      runAllImmediates: () => this._environment.fakeTimers.runAllImmediates(),
      runAllTimers: () => this._environment.fakeTimers.runAllTimers(),
      runTimersToTime: msToRun =>
      this._environment.fakeTimers.runTimersToTime(msToRun),
      runOnlyPendingTimers: () =>
      this._environment.fakeTimers.runOnlyPendingTimers(),

      setMock: (moduleName, mock) =>
      setMockFactory(moduleName, () => mock),

      deepUnmock,

      useFakeTimers,
      useRealTimers };

    return runtime;
  }}


module.exports = Runtime;