'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _crypto = require('crypto');

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _mkdirp = require('mkdirp');

var _mkdirp2 = _interopRequireDefault(_mkdirp);

var _chalk = require('chalk');

var _chalk2 = _interopRequireDefault(_chalk);

var _moment = require('moment');

var _moment2 = _interopRequireDefault(_moment);

var _filesize = require('filesize');

var _filesize2 = _interopRequireDefault(_filesize);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * When 'webpack' program is used, constructor name is equal to 'NodeOutputFileSystem'.
 * When 'webpack-dev-server' program is used, constructor name is equal to 'MemoryFileSystem'.
 */
const isMemoryFileSystem = outputFileSystem => {
  return outputFileSystem.constructor.name === 'MemoryFileSystem';
};

/**
 * @property test A regular expression used to test if file should be written. When not present, all bundle will be written.
 * @property useHashIndex Use hash index to write only files that have changed since the last iteration (default: true).
 * @property log Logs names of the files that are being written (or skipped because they have not changed) (default: true).
 * @property exitOnErors Stop writing files on webpack errors (default: true).
 */

exports.default = function () {
  let userOptions = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

  const options = _lodash2.default.assign({}, {
    exitOnErrors: true,
    force: false,
    log: true,
    test: null,
    useHashIndex: true
  }, userOptions);

  if (!_lodash2.default.isNull(options.test) && !_lodash2.default.isRegExp(options.test)) {
    throw new Error('options.test value must be an instance of RegExp.');
  }

  if (!_lodash2.default.isBoolean(options.useHashIndex)) {
    throw new Error('options.useHashIndex value must be of boolean type.');
  }

  if (!_lodash2.default.isBoolean(options.log)) {
    throw new Error('options.log value must be of boolean type.');
  }

  if (!_lodash2.default.isBoolean(options.exitOnErrors)) {
    throw new Error('options.exitOnErrors value must be of boolean type.');
  }

  if (!_lodash2.default.isBoolean(options.force)) {
    throw new Error('options.force value must be of boolean type.');
  }

  const log = function () {
    var _console;

    for (var _len = arguments.length, append = Array(_len), _key = 0; _key < _len; _key++) {
      append[_key] = arguments[_key];
    }

    if (!options.log) {
      return;
    }

    // eslint-disable-next-line no-console
    (_console = console).log.apply(_console, [_chalk2.default.dim('[' + (0, _moment2.default)().format('HH:mm:ss') + '] [write-file-webpack-plugin]')].concat(append));
  };

  const assetSourceHashIndex = {};

  log('options', options);

  const apply = compiler => {
    let outputPath, setupDone, setupStatus;

    const setup = () => {
      if (setupDone) {
        return setupStatus;
      }

      setupDone = true;

      log('compiler.outputFileSystem is "' + _chalk2.default.cyan(compiler.outputFileSystem.constructor.name) + '".');

      if (!isMemoryFileSystem(compiler.outputFileSystem) && !options.force) {
        return false;
      }

      // https://github.com/gajus/write-file-webpack-plugin/issues/1
      // `compiler.options.output.path` will be hardcoded to '/' in
      // webpack-dev-server's command line wrapper. So it should be
      // ignored here.
      if (_lodash2.default.has(compiler, 'options.output.path') && compiler.options.output.path !== '/') {
        outputPath = compiler.options.output.path;
      }

      if (!outputPath) {
        if (!_lodash2.default.has(compiler, 'options.devServer.outputPath')) {
          throw new Error('output.path is not accessible and devServer.outputPath is not defined. Define devServer.outputPath.');
        }

        outputPath = compiler.options.devServer.outputPath;
      }

      log('compiler.options.devServer.outputPath is "' + _chalk2.default.cyan(outputPath) + '".');

      setupStatus = true;

      return setupStatus;
    };

    compiler.plugin('done', stats => {
      if (!setup()) {
        return;
      }

      if (options.exitOnErrors && stats.compilation.errors.length) {
        return;
      }

      log('stats.compilation.errors.length is "' + _chalk2.default.cyan(stats.compilation.errors.length) + '".');

      _lodash2.default.forEach(stats.compilation.assets, (asset, assetPath) => {
        const outputFilePath = _path2.default.join(outputPath, assetPath);
        const relativeOutputPath = _path2.default.relative(process.cwd(), outputFilePath);
        const targetDefinition = 'asset: ' + _chalk2.default.cyan('./' + assetPath) + '; destination: ' + _chalk2.default.cyan('./' + relativeOutputPath);

        if (options.test && !options.test.test(assetPath)) {
          log(targetDefinition, _chalk2.default.yellow('[skipped; does not match test]'));

          return;
        }

        const assetSize = asset.size();
        const assetSource = asset.source();

        if (options.useHashIndex) {
          const assetSourceHash = (0, _crypto.createHash)('sha256').update(assetSource).digest('hex');

          if (assetSourceHashIndex[assetPath] && assetSourceHashIndex[assetPath] === assetSourceHash) {
            log(targetDefinition, _chalk2.default.yellow('[skipped; matched hash index]'));

            return;
          }

          assetSourceHashIndex[assetPath] = assetSourceHash;
        }

        log(targetDefinition, _chalk2.default.green('[written]'), _chalk2.default.magenta('(' + (0, _filesize2.default)(assetSize) + ')'));

        _mkdirp2.default.sync(_path2.default.dirname(relativeOutputPath));

        _fs2.default.writeFileSync(relativeOutputPath.split('?')[0], assetSource);
      });
    });
  };

  return {
    apply
  };
};

module.exports = exports['default'];
//# sourceMappingURL=index.js.map