# 68 "src/reason-parser/reason_lexer.mll"
 
open Lexing
open Reason_parser
open Lexer_warning

type error =
  | Illegal_character of char
  | Illegal_escape of string
  | Unterminated_comment of Location.t
  | Unterminated_string
  | Unterminated_string_in_comment of Location.t * Location.t
  | Keyword_as_label of string
  | Literal_overflow of string
  | Invalid_literal of string
;;

exception Error of error * Location.t;;

(* The table of keywords *)

let keyword_table =
  let create_hashtable n l =
    let t = Hashtbl.create n in
    List.iter (fun (k, v) -> Hashtbl.add t k v) l;
    t
  in
  create_hashtable 149 [
    "and", AND;
    "as", AS;
    "assert", ASSERT;
    "begin", BEGIN;
    "class", CLASS;
    "constraint", CONSTRAINT;
    "do", DO;
    "done", DONE;
    "downto", DOWNTO;
    "else", ELSE;
    "end", END;
    "exception", EXCEPTION;
    "external", EXTERNAL;
    "false", FALSE;
    "for", FOR;
    "fun", FUN;
    "esfun", ES6_FUN;
    "function", FUNCTION;
    "functor", FUNCTOR;
    "if", IF;
    "in", IN;
    "include", INCLUDE;
    "inherit", INHERIT;
    "initializer", INITIALIZER;
    "lazy", LAZY;
    "let", LET;
    "switch", SWITCH;
    "module", MODULE;
    "pub", PUB;
    "mutable", MUTABLE;
    "new", NEW;
    "nonrec", NONREC;
    "object", OBJECT;
    "of", OF;
    "open", OPEN;
    "or", OR;
(*  "parser", PARSER; *)
    "pri", PRI;
    "rec", REC;
    "sig", SIG;
    "struct", STRUCT;
    "then", THEN;
    "to", TO;
    "true", TRUE;
    "try", TRY;
    "type", TYPE;
    "val", VAL;
    "virtual", VIRTUAL;
    "when", WHEN;
    "while", WHILE;
    "with", WITH;

    "mod", INFIXOP3("mod");
    "land", INFIXOP3("land");
    "lor", INFIXOP3("lor");
    "lxor", INFIXOP3("lxor");
    "lsl", INFIXOP4("lsl");
    "lsr", INFIXOP4("lsr");
    "asr", INFIXOP4("asr")
]

(* To buffer string literals *)

let string_buffer = Buffer.create 256

let reset_string_buffer () =
  Buffer.reset string_buffer

let store_string_char c =
  Buffer.add_char string_buffer c

let store_string s =
  Buffer.add_string string_buffer s

let store_lexeme lexbuf =
  store_string (Lexing.lexeme lexbuf)

let get_stored_string () =
  let s = Buffer.contents string_buffer in
  Buffer.reset string_buffer;
  s

(* To store the position of the beginning of a string and comment *)
let string_start_loc = ref Location.none;;
let comment_start_loc = ref [];;
let in_comment () = !comment_start_loc <> [];;
let is_in_string = ref false
let in_string () = !is_in_string
let print_warnings = ref true

(* To "unlex" a few characters *)
let set_lexeme_length buf n = (
  let open Lexing in
  if n < 0 then
    invalid_arg "set_lexeme_length: offset should be positive";
  if n > buf.lex_curr_pos - buf.lex_start_pos then
    invalid_arg "set_lexeme_length: offset larger than lexeme";
  buf.lex_curr_pos <- buf.lex_start_pos + n;
  buf.lex_curr_p <- {buf.lex_start_p
                     with pos_cnum = buf.lex_abs_pos + buf.lex_curr_pos};
)

(* This cut comment characters of the current buffer.
 * Operators (including "/*" and "*/") are lexed with the same rule, and this
 * function cuts the lexeme at the beginning of an operator. *)
let lexeme_without_comment buf = (
  let lexeme = Lexing.lexeme buf in
  let i = ref 0 and len = String.length lexeme - 1 in
  let found = ref (-1) in
  while !i < len && !found = -1 do
    begin match lexeme.[!i], lexeme.[!i+1] with
      | ('/', '*') | ('*', '/') ->
        found := !i;
      | _ -> ()
    end;
    incr i
  done;
  match !found with
  | -1 -> lexeme
  | n ->
      set_lexeme_length buf n;
      String.sub lexeme 0 n
)

(* Operators that could conflict with comments (those containing /*, */ and //)
 * are escaped in the source. The lexer removes the escapes so that the
 * identifier looks like OCaml ones.
 * An escape in first position is kept to distinguish "verbatim" operators
 * (\=== for instance). *)
let unescape_operator str =
  if (str <> "" && String.contains_from str 1 '\\') then (
    let b = Buffer.create (String.length str) in
    Buffer.add_char b str.[0];
    for i = 1 to String.length str - 1 do
      let c = str.[i] in
      if c <> '\\' then Buffer.add_char b c
    done;
    Buffer.contents b
  ) else str

let lexeme_operator lexbuf =
  unescape_operator (lexeme_without_comment lexbuf)

(* To translate escape sequences *)

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let char_for_decimal_code lexbuf i =
  let c = 100 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           10 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
                (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  if (c < 0 || c > 255) then
    if in_comment ()
    then 'x'
    else raise (Error(Illegal_escape (Lexing.lexeme lexbuf),
                      Location.curr lexbuf))
  else Char.chr c

let char_for_hexadecimal_code lexbuf i =
  let d1 = Char.code (Lexing.lexeme_char lexbuf i) in
  let val1 = if d1 >= 97 then d1 - 87
             else if d1 >= 65 then d1 - 55
             else d1 - 48
  in
  let d2 = Char.code (Lexing.lexeme_char lexbuf (i+1)) in
  let val2 = if d2 >= 97 then d2 - 87
             else if d2 >= 65 then d2 - 55
             else d2 - 48
  in
  Char.chr (val1 * 16 + val2)

(* To convert integer literals, allowing max_int + 1 (PR#4210) *)

let cvt_int_literal s =
  - int_of_string ("-" ^ s)
let cvt_int32_literal s =
  Int32.neg (Int32.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_int64_literal s =
  Int64.neg (Int64.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_nativeint_literal s =
  Nativeint.neg (Nativeint.of_string ("-" ^ String.sub s 0
                                                       (String.length s - 1)))

(* Remove underscores from float literals *)

let remove_underscores s =
  let l = String.length s in
  let b = Bytes.create l in
  let rec remove src dst =
    if src >= l then
      if dst >= l then s else Bytes.sub_string b 0 dst
    else
      match s.[src] with
        '_' -> remove (src + 1) dst
      |  c  -> Bytes.set b dst c; remove (src + 1) (dst + 1)
  in remove 0 0

(* Update the current location with file name and line number. *)

let update_loc lexbuf file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = match file with
                 | None -> pos.pos_fname
                 | Some s -> s
  in
  lexbuf.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  }
;;

let preprocessor = ref None

(* Error report *)

open Format

let report_error ppf = function
  | Illegal_character c ->
      fprintf ppf "Illegal character (%s)" (Char.escaped c)
  | Illegal_escape s ->
      fprintf ppf "Illegal backslash escape in string or character (%s)" s
  | Unterminated_comment _ ->
      fprintf ppf "Comment not terminated"
  | Unterminated_string ->
      fprintf ppf "String literal not terminated"
  | Unterminated_string_in_comment (_, loc) ->
      fprintf ppf "This comment contains an unterminated string literal@.\
                   %aString literal begins here"
              Location.print_error loc
  | Keyword_as_label kwd ->
      fprintf ppf "`%s' is a keyword, it cannot be used as label name" kwd
  | Literal_overflow ty ->
      fprintf ppf "Integer literal exceeds the range of representable \
                   integers of type %s" ty
  | Invalid_literal s ->
      fprintf ppf "Invalid literal %s" s

let () =
  Location.register_error_of_exn
    (function
      | Error (err, loc) ->
          Some (Location.error_of_printer loc report_error err)
      | _ ->
          None
    )

(* To "unlex" a few characters *)
let set_lexeme_length buf n = (
  let open Lexing in
  if n < 0 then
    invalid_arg "set_lexeme_length: offset should be positive";
  if n > buf.lex_curr_pos - buf.lex_start_pos then
    invalid_arg "set_lexeme_length: offset larger than lexeme";
  buf.lex_curr_pos <- buf.lex_start_pos + n;
  buf.lex_curr_p <- {buf.lex_start_p
                     with pos_cnum = buf.lex_abs_pos + buf.lex_curr_pos};
)


# 296 "src/reason-parser/reason_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\156\255\157\255\224\000\003\001\038\001\073\001\108\001\
    \143\001\178\001\191\255\213\001\248\001\198\255\091\000\059\002\
    \068\000\071\000\084\000\151\002\220\255\186\002\222\255\223\255\
    \225\255\221\002\060\003\012\001\090\003\238\255\174\003\002\004\
    \086\004\038\005\246\005\198\006\165\007\200\007\235\007\063\008\
    \122\000\254\255\001\000\053\009\088\009\123\009\158\009\193\009\
    \228\009\255\255\005\000\007\010\093\000\042\010\077\010\094\000\
    \112\010\095\000\096\000\097\000\006\000\174\010\246\255\103\000\
    \008\011\100\011\135\011\144\004\165\011\249\011\077\012\161\012\
    \245\012\073\013\157\013\241\013\069\014\153\014\237\014\108\000\
    \065\015\149\015\233\015\061\016\145\016\118\000\096\005\237\255\
    \058\003\229\016\064\017\107\000\108\000\007\000\236\255\235\255\
    \231\255\010\002\128\000\109\000\234\255\151\002\110\000\233\255\
    \238\002\113\000\232\255\104\006\114\006\089\017\144\000\130\017\
    \165\017\228\255\011\000\012\000\008\001\024\001\016\000\228\255\
    \017\000\200\017\235\017\014\018\141\000\214\255\210\255\211\255\
    \212\255\208\255\049\018\084\018\151\018\235\018\069\019\102\019\
    \199\255\192\019\227\019\151\000\184\255\202\255\203\255\129\000\
    \187\255\182\255\193\255\006\020\073\020\190\255\008\007\165\020\
    \121\008\001\021\036\021\188\255\073\021\108\021\143\021\178\021\
    \213\021\216\000\233\000\248\021\040\001\005\001\027\022\012\001\
    \070\001\250\255\143\000\205\000\030\001\252\255\046\001\253\255\
    \222\000\251\255\124\018\243\255\244\255\018\000\245\255\253\001\
    \025\022\253\255\218\000\231\000\255\255\254\255\252\255\057\022\
    \148\022\237\000\240\000\111\001\251\255\250\255\249\255\024\020\
    \077\003\244\000\248\255\087\003\252\000\247\255\182\022\004\001\
    \246\255\112\001\217\002\247\255\248\255\249\255\113\001\253\022\
    \255\255\250\255\031\023\219\004\253\255\046\001\129\001\139\001\
    \171\005\252\255\070\023\251\255\146\001\220\001\252\255\078\023\
    \254\255\255\255\147\001\148\001\253\255\107\023\009\001\020\001\
    \047\001\069\001\035\001\093\001\034\001\013\000\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\095\000\090\000\089\000\088\000\079\000\
    \074\000\096\000\255\255\063\000\060\000\255\255\050\000\048\000\
    \046\000\042\000\039\000\082\000\255\255\034\000\255\255\255\255\
    \255\255\028\000\038\000\031\000\058\000\255\255\012\000\012\000\
    \011\000\010\000\008\000\007\000\049\000\005\000\004\000\003\000\
    \002\000\255\255\099\000\099\000\096\000\096\000\092\000\255\255\
    \255\255\255\255\255\255\087\000\255\255\077\000\078\000\255\255\
    \094\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \088\000\037\000\006\000\066\000\016\000\016\000\014\000\013\000\
    \016\000\013\000\013\000\012\000\014\000\013\000\014\000\255\255\
    \015\000\015\000\012\000\012\000\014\000\255\255\059\000\255\255\
    \255\255\055\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\024\000\024\000\024\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\026\000\255\255\026\000\
    \025\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\029\000\083\000\036\000\040\000\255\255\255\255\255\255\
    \255\255\255\255\088\000\084\000\088\000\054\000\085\000\255\255\
    \255\255\088\000\086\000\072\000\255\255\255\255\255\255\051\000\
    \255\255\255\255\255\255\061\000\088\000\255\255\070\000\088\000\
    \069\000\093\000\096\000\255\255\075\000\076\000\091\000\081\000\
    \080\000\255\255\255\255\090\000\255\255\255\255\097\000\255\255\
    \255\255\255\255\005\000\005\000\000\000\255\255\001\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\012\000\255\255\012\000\
    \012\000\255\255\012\000\012\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\008\000\008\000\
    \255\255\255\255\005\000\005\000\255\255\001\000\005\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\003\000\
    \255\255\255\255\003\000\255\255\255\255\255\255\002\000\255\255\
    \255\255\001\000\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\000\000\
    \000\000\255\255\255\255\091\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\096\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \255\255\255\255\000\000\115\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\115\000\116\000\115\000\118\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \169\000\000\000\255\255\255\255\255\255\000\000\255\255\000\000\
    \255\255\000\000\179\000\000\000\000\000\255\255\000\000\193\000\
    \255\255\000\000\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\211\000\000\000\000\000\000\000\255\255\217\000\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\000\000\255\255\230\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \240\000\243\000\255\255\243\000\255\255\255\255\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\040\000\041\000\041\000\040\000\042\000\060\000\049\000\
    \041\000\092\000\050\000\060\000\093\000\113\000\113\000\246\000\
    \114\000\114\000\119\000\119\000\180\000\120\000\120\000\209\000\
    \040\000\008\000\029\000\026\000\006\000\003\000\025\000\027\000\
    \023\000\022\000\021\000\007\000\020\000\019\000\018\000\009\000\
    \031\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\017\000\016\000\015\000\036\000\011\000\037\000\
    \005\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\014\000\043\000\013\000\004\000\039\000\
    \024\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\028\000\012\000\010\000\038\000\129\000\
    \139\000\128\000\124\000\040\000\127\000\126\000\040\000\051\000\
    \046\000\056\000\056\000\044\000\051\000\046\000\044\000\056\000\
    \044\000\006\000\095\000\094\000\100\000\103\000\006\000\142\000\
    \106\000\141\000\040\000\140\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \101\000\101\000\109\000\125\000\145\000\144\000\177\000\109\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\143\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\172\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \002\000\166\000\158\000\166\000\166\000\166\000\166\000\158\000\
    \176\000\189\000\166\000\166\000\255\255\166\000\166\000\166\000\
    \255\255\188\000\113\000\005\000\198\000\114\000\092\000\197\000\
    \005\000\093\000\166\000\202\000\166\000\166\000\166\000\166\000\
    \166\000\117\000\113\000\205\000\163\000\114\000\163\000\163\000\
    \163\000\163\000\118\000\208\000\239\000\163\000\163\000\163\000\
    \163\000\163\000\163\000\255\255\163\000\240\000\166\000\221\000\
    \117\000\241\000\116\000\166\000\167\000\163\000\166\000\163\000\
    \163\000\163\000\163\000\163\000\244\000\245\000\000\000\005\000\
    \174\000\005\000\005\000\005\000\005\000\173\000\221\000\242\000\
    \005\000\005\000\163\000\005\000\005\000\005\000\163\000\163\000\
    \176\000\000\000\000\000\000\000\166\000\175\000\166\000\164\000\
    \005\000\163\000\005\000\005\000\005\000\005\000\005\000\242\000\
    \090\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \170\000\000\000\000\000\006\000\006\000\171\000\006\000\006\000\
    \006\000\194\000\180\000\213\000\195\000\209\000\228\000\163\000\
    \000\000\163\000\162\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\221\000\000\000\046\000\223\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\221\000\046\000\158\000\
    \223\000\046\000\160\000\046\000\213\000\233\000\233\000\228\000\
    \235\000\235\000\005\000\000\000\005\000\063\000\046\000\006\000\
    \046\000\159\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\051\000\051\000\000\000\051\000\051\000\051\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \055\000\051\000\046\000\051\000\156\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\044\000\000\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\044\000\044\000\000\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\000\000\233\000\000\000\
    \046\000\234\000\046\000\052\000\044\000\051\000\044\000\044\000\
    \153\000\044\000\044\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\194\000\
    \255\255\000\000\195\000\051\000\255\255\051\000\059\000\006\000\
    \044\000\148\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \255\255\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\196\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\044\000\255\255\
    \044\000\063\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\000\000\255\255\255\255\000\000\
    \000\000\000\000\000\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\006\000\149\000\006\000\063\000\146\000\006\000\000\000\
    \231\000\192\000\000\000\000\000\006\000\255\255\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\006\000\000\000\
    \006\000\130\000\132\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\000\000\000\000\000\000\147\000\006\000\006\000\006\000\
    \006\000\131\000\006\000\006\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\063\000\
    \000\000\006\000\133\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\006\000\
    \046\000\006\000\046\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\046\000\046\000\000\000\046\000\122\000\046\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\046\000\000\000\046\000\046\000\123\000\046\000\046\000\
    \000\000\000\000\000\000\044\000\232\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\213\000\056\000\044\000\214\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\055\000\044\000\046\000\044\000\044\000\
    \044\000\044\000\044\000\216\000\000\000\255\255\006\000\000\000\
    \006\000\006\000\006\000\121\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\046\000\000\000\046\000\057\000\006\000\
    \044\000\006\000\006\000\006\000\006\000\006\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\215\000\044\000\000\000\
    \044\000\063\000\000\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\108\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\108\000\109\000\000\000\109\000\
    \109\000\109\000\109\000\000\000\000\000\000\000\109\000\109\000\
    \000\000\109\000\109\000\109\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\109\000\000\000\
    \109\000\111\000\109\000\109\000\109\000\203\000\203\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\204\000\
    \204\000\204\000\204\000\204\000\204\000\204\000\204\000\204\000\
    \204\000\000\000\000\000\000\000\000\000\000\000\086\000\000\000\
    \110\000\088\000\109\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\000\000\087\000\000\000\
    \109\000\088\000\109\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\068\000\087\000\000\000\
    \000\000\212\000\000\000\000\000\070\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \068\000\068\000\068\000\069\000\068\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\000\000\000\000\000\000\000\000\030\000\000\000\068\000\
    \068\000\068\000\068\000\069\000\068\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\068\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \070\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\068\000\072\000\068\000\068\000\069\000\
    \068\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\073\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\074\000\071\000\071\000\000\000\000\000\000\000\
    \000\000\030\000\000\000\068\000\072\000\068\000\068\000\069\000\
    \068\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\073\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\074\000\071\000\071\000\032\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\000\000\000\000\000\000\000\000\032\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\000\000\000\000\000\000\067\000\
    \000\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\033\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\000\000\000\000\000\000\033\000\000\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\000\000\000\000\000\000\000\000\089\000\
    \000\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\034\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\000\000\000\000\000\000\000\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\117\000\113\000\000\000\000\000\114\000\000\000\000\000\
    \000\000\000\000\000\000\108\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \117\000\000\000\116\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\108\000\000\000\000\000\000\000\000\000\000\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\035\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\000\000\035\000\000\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\000\000\000\000\000\000\000\000\150\000\
    \255\255\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\000\000\000\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\006\000\
    \000\000\064\000\006\000\065\000\066\000\006\000\000\000\000\000\
    \000\000\051\000\000\000\051\000\051\000\051\000\051\000\000\000\
    \000\000\000\000\051\000\051\000\000\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\051\000\006\000\051\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\051\000\000\000\051\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\051\000\051\000\000\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\052\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\051\000\000\000\051\000\052\000\
    \000\000\051\000\061\000\000\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\035\000\051\000\
    \000\000\051\000\000\000\000\000\000\000\000\000\000\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\000\000\000\000\000\000\000\000\
    \152\000\000\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\049\000\
    \000\000\000\000\050\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\048\000\000\000\
    \000\000\006\000\000\000\006\000\000\000\000\000\000\000\045\000\
    \046\000\000\000\046\000\000\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\006\000\006\000\047\000\005\000\000\000\000\000\
    \000\000\044\000\000\000\044\000\044\000\044\000\044\000\000\000\
    \000\000\000\000\044\000\044\000\000\000\044\000\044\000\044\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\044\000\004\000\044\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\044\000\000\000\044\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\056\000\044\000\000\000\
    \044\000\044\000\044\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\047\000\059\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \046\000\046\000\000\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\044\000\000\000\044\000\057\000\
    \046\000\044\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \000\000\000\000\051\000\000\000\051\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\051\000\051\000\000\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\000\000\000\000\000\000\044\000\
    \000\000\044\000\055\000\051\000\046\000\051\000\051\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\051\000\000\000\051\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\051\000\051\000\
    \000\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\046\000\000\000\046\000\052\000\051\000\051\000\
    \051\000\053\000\051\000\051\000\051\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\051\000\051\000\000\000\051\000\051\000\051\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\051\000\000\000\051\000\
    \052\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\051\000\000\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\051\000\051\000\000\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\052\000\051\000\051\000\051\000\054\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\051\000\
    \051\000\000\000\051\000\051\000\051\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\051\000\000\000\051\000\052\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\056\000\000\000\056\000\056\000\056\000\056\000\000\000\
    \000\000\000\000\056\000\056\000\000\000\056\000\056\000\056\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\052\000\056\000\051\000\056\000\056\000\056\000\056\000\
    \056\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\051\000\000\000\051\000\058\000\000\000\056\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \000\000\000\000\000\000\062\000\056\000\000\000\056\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\000\000\000\000\000\000\000\000\061\000\000\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \006\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\063\000\000\000\006\000\067\000\
    \000\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \063\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\000\000\
    \006\000\000\000\006\000\063\000\000\000\006\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\006\000\068\000\006\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\085\000\000\000\085\000\000\000\
    \000\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\069\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \000\000\000\000\000\000\000\000\070\000\000\000\068\000\068\000\
    \068\000\068\000\069\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\083\000\083\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\076\000\
    \000\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\077\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\000\000\000\000\000\000\000\000\
    \075\000\000\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\077\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\081\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \000\000\000\000\000\000\000\000\076\000\000\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\081\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \068\000\000\000\000\000\000\000\079\000\000\000\079\000\000\000\
    \000\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \000\000\000\000\000\000\000\000\078\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \079\000\000\000\079\000\000\000\000\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\000\000\000\000\000\000\000\000\
    \082\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\083\000\083\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \000\000\000\000\000\000\000\000\083\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\000\000\000\000\000\000\000\000\
    \084\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\089\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\089\000\000\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \000\000\000\000\000\000\000\000\089\000\000\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \099\000\000\000\099\000\000\000\000\000\000\000\000\000\099\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\098\000\109\000\000\000\109\000\109\000\109\000\109\000\
    \000\000\000\000\000\000\109\000\109\000\000\000\109\000\109\000\
    \109\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\109\000\000\000\109\000\109\000\109\000\
    \109\000\109\000\000\000\000\000\099\000\000\000\000\000\000\000\
    \000\000\000\000\099\000\109\000\000\000\109\000\109\000\109\000\
    \109\000\000\000\000\000\000\000\109\000\109\000\099\000\109\000\
    \109\000\109\000\099\000\000\000\099\000\110\000\000\000\109\000\
    \097\000\000\000\000\000\000\000\109\000\000\000\112\000\109\000\
    \109\000\109\000\109\000\000\000\000\000\000\000\109\000\000\000\
    \109\000\109\000\109\000\109\000\000\000\000\000\000\000\109\000\
    \109\000\000\000\109\000\109\000\109\000\109\000\000\000\109\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\110\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\109\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\109\000\000\000\
    \109\000\110\000\006\000\109\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\046\000\000\000\046\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\109\000\000\000\109\000\063\000\046\000\006\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \046\000\046\000\000\000\046\000\046\000\046\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\006\000\000\000\006\000\055\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \000\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\006\000\000\000\006\000\137\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\055\000\006\000\046\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\180\000\000\000\
    \000\000\181\000\046\000\000\000\046\000\063\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\185\000\000\000\
    \000\000\000\000\000\000\183\000\000\000\000\000\186\000\000\000\
    \000\000\000\000\000\000\187\000\000\000\006\000\000\000\006\000\
    \063\000\000\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\134\000\006\000\006\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\000\000\063\000\000\000\006\000\135\000\184\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\133\000\006\000\000\000\006\000\000\000\000\000\
    \000\000\133\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\000\000\
    \000\000\000\000\133\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\182\000\000\000\000\000\006\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\135\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\135\000\000\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \000\000\063\000\000\000\006\000\136\000\000\000\000\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\006\000\000\000\006\000\000\000\135\000\000\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\006\000\000\000\006\000\006\000\138\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\000\000\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\006\000\000\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\063\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \000\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \006\000\006\000\000\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\006\000\000\000\006\000\063\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\006\000\000\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\206\000\206\000\206\000\206\000\206\000\206\000\006\000\
    \000\000\006\000\063\000\000\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\006\000\000\000\006\000\006\000\
    \151\000\206\000\206\000\206\000\206\000\206\000\206\000\000\000\
    \000\000\000\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\000\000\063\000\000\000\006\000\
    \150\000\000\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\000\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\006\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \000\000\063\000\000\000\006\000\152\000\000\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \000\000\006\000\044\000\006\000\044\000\044\000\044\000\044\000\
    \000\000\000\000\000\000\044\000\044\000\000\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\044\000\000\000\044\000\044\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\044\000\000\000\044\000\
    \044\000\044\000\044\000\000\000\000\000\000\000\044\000\044\000\
    \000\000\044\000\044\000\044\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\059\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\051\000\000\000\051\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\051\000\051\000\000\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\000\000\154\000\000\000\044\000\
    \059\000\155\000\044\000\051\000\000\000\051\000\157\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\051\000\000\000\051\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\051\000\051\000\
    \000\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \044\000\000\000\044\000\000\000\000\000\052\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\000\000\000\000\000\000\
    \158\000\000\000\158\000\158\000\158\000\158\000\000\000\000\000\
    \000\000\158\000\158\000\000\000\158\000\158\000\158\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\051\000\000\000\051\000\
    \052\000\158\000\051\000\158\000\158\000\158\000\158\000\158\000\
    \000\000\000\000\000\000\046\000\000\000\046\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\046\000\046\000\000\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\161\000\046\000\158\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\046\000\
    \046\000\000\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\158\000\000\000\158\000\055\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\163\000\000\000\163\000\163\000\163\000\163\000\000\000\
    \000\000\000\000\163\000\163\000\000\000\163\000\163\000\163\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\046\000\000\000\
    \046\000\055\000\163\000\046\000\163\000\163\000\163\000\163\000\
    \163\000\000\000\000\000\000\000\166\000\000\000\166\000\166\000\
    \166\000\166\000\000\000\000\000\000\000\166\000\166\000\000\000\
    \166\000\166\000\166\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\000\000\046\000\165\000\166\000\163\000\166\000\
    \166\000\166\000\166\000\166\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\163\000\000\000\163\000\167\000\
    \191\000\166\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\000\000\190\000\000\000\166\000\
    \191\000\166\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\201\000\190\000\201\000\000\000\
    \000\000\000\000\000\000\201\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\200\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \201\000\000\000\000\000\000\000\000\000\000\000\201\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\000\000\000\000\000\000\
    \000\000\000\000\201\000\000\000\000\000\000\000\201\000\221\000\
    \201\000\000\000\222\000\000\000\199\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\220\000\000\000\220\000\
    \000\000\000\000\000\000\000\000\220\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\220\000\000\000\000\000\000\000\000\000\000\000\220\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\000\000\000\000\
    \000\000\000\000\000\000\220\000\000\000\000\000\000\000\220\000\
    \000\000\220\000\000\000\000\000\000\000\218\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\000\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\237\000\000\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\000\000\237\000\236\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\000\000\000\000\
    \236\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\042\000\000\000\000\000\042\000\050\000\
    \060\000\093\000\050\000\060\000\093\000\114\000\115\000\245\000\
    \114\000\115\000\118\000\120\000\181\000\118\000\120\000\181\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\016\000\
    \014\000\017\000\018\000\040\000\017\000\017\000\040\000\052\000\
    \055\000\057\000\058\000\059\000\052\000\055\000\057\000\058\000\
    \059\000\063\000\091\000\092\000\099\000\102\000\063\000\014\000\
    \105\000\014\000\040\000\014\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\098\000\110\000\124\000\139\000\143\000\170\000\110\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\014\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\171\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\161\000\003\000\003\000\003\000\003\000\161\000\
    \176\000\186\000\003\000\003\000\115\000\003\000\003\000\003\000\
    \118\000\187\000\116\000\162\000\193\000\116\000\027\000\194\000\
    \162\000\027\000\003\000\201\000\003\000\003\000\003\000\003\000\
    \003\000\117\000\117\000\204\000\004\000\117\000\004\000\004\000\
    \004\000\004\000\116\000\207\000\238\000\004\000\004\000\165\000\
    \004\000\004\000\004\000\027\000\165\000\239\000\167\000\221\000\
    \117\000\240\000\117\000\167\000\003\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\242\000\244\000\255\255\005\000\
    \172\000\005\000\005\000\005\000\005\000\172\000\221\000\241\000\
    \005\000\005\000\164\000\005\000\005\000\005\000\164\000\164\000\
    \174\000\255\255\255\255\255\255\003\000\174\000\003\000\004\000\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\243\000\
    \027\000\255\255\006\000\255\255\006\000\006\000\006\000\006\000\
    \168\000\255\255\255\255\006\000\006\000\168\000\006\000\006\000\
    \006\000\195\000\209\000\214\000\195\000\209\000\214\000\004\000\
    \255\255\004\000\005\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\222\000\255\255\007\000\222\000\007\000\
    \007\000\007\000\007\000\255\255\255\255\223\000\007\000\007\000\
    \223\000\007\000\007\000\007\000\228\000\234\000\235\000\228\000\
    \234\000\235\000\005\000\255\255\005\000\006\000\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \008\000\255\255\008\000\008\000\008\000\008\000\255\255\255\255\
    \255\255\008\000\008\000\255\255\008\000\008\000\008\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\006\000\
    \007\000\008\000\007\000\008\000\008\000\008\000\008\000\008\000\
    \255\255\255\255\255\255\009\000\255\255\009\000\009\000\009\000\
    \009\000\255\255\255\255\255\255\009\000\009\000\255\255\009\000\
    \009\000\009\000\255\255\255\255\255\255\255\255\229\000\255\255\
    \007\000\229\000\007\000\008\000\009\000\008\000\009\000\009\000\
    \009\000\009\000\009\000\255\255\255\255\255\255\011\000\255\255\
    \011\000\011\000\011\000\011\000\255\255\255\255\255\255\011\000\
    \011\000\255\255\011\000\011\000\011\000\255\255\255\255\183\000\
    \116\000\255\255\183\000\008\000\027\000\008\000\009\000\011\000\
    \009\000\011\000\011\000\011\000\011\000\011\000\255\255\255\255\
    \117\000\012\000\255\255\012\000\012\000\012\000\012\000\255\255\
    \255\255\255\255\012\000\012\000\183\000\012\000\012\000\012\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\009\000\240\000\
    \009\000\011\000\012\000\011\000\012\000\012\000\012\000\012\000\
    \012\000\255\255\097\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\255\255\241\000\168\000\255\255\
    \255\255\255\255\255\255\097\000\097\000\097\000\097\000\097\000\
    \097\000\011\000\011\000\011\000\012\000\012\000\012\000\255\255\
    \229\000\183\000\255\255\255\255\015\000\243\000\015\000\015\000\
    \015\000\015\000\255\255\255\255\255\255\015\000\015\000\255\255\
    \015\000\015\000\015\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\255\255\255\255\255\255\012\000\015\000\012\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\255\255\015\000\
    \255\255\015\000\015\000\255\255\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\255\255\015\000\
    \019\000\015\000\019\000\019\000\019\000\019\000\255\255\255\255\
    \255\255\019\000\019\000\255\255\019\000\019\000\019\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \101\000\019\000\255\255\019\000\019\000\019\000\019\000\019\000\
    \255\255\255\255\255\255\021\000\229\000\021\000\021\000\021\000\
    \021\000\255\255\255\255\210\000\021\000\021\000\210\000\021\000\
    \021\000\021\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\019\000\021\000\019\000\021\000\021\000\
    \021\000\021\000\021\000\210\000\255\255\183\000\025\000\255\255\
    \025\000\025\000\025\000\025\000\255\255\255\255\255\255\025\000\
    \025\000\255\255\025\000\025\000\025\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\019\000\255\255\019\000\021\000\025\000\
    \021\000\025\000\025\000\025\000\025\000\025\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\104\000\
    \104\000\104\000\104\000\104\000\104\000\210\000\021\000\255\255\
    \021\000\025\000\255\255\025\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\026\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\104\000\
    \104\000\104\000\104\000\104\000\104\000\255\255\255\255\255\255\
    \255\255\025\000\255\255\025\000\026\000\026\000\255\255\026\000\
    \026\000\026\000\026\000\255\255\255\255\255\255\026\000\026\000\
    \255\255\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \026\000\026\000\026\000\026\000\026\000\200\000\200\000\200\000\
    \200\000\200\000\200\000\200\000\200\000\200\000\200\000\203\000\
    \203\000\203\000\203\000\203\000\203\000\203\000\203\000\203\000\
    \203\000\255\255\255\255\255\255\255\255\255\255\028\000\255\255\
    \026\000\088\000\026\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\255\255\088\000\255\255\
    \026\000\028\000\026\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\030\000\028\000\255\255\
    \255\255\210\000\255\255\255\255\030\000\255\255\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\255\255\255\255\255\255\255\255\030\000\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\031\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\255\255\255\255\
    \255\255\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\032\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\255\255\255\255\255\255\032\000\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\255\255\255\255\255\255\255\255\067\000\
    \255\255\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\033\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\255\255\255\255\255\255\255\255\033\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\255\255\255\255\255\255\255\255\086\000\
    \255\255\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\034\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\255\255\255\255\255\255\255\255\034\000\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\107\000\107\000\255\255\255\255\107\000\255\255\255\255\
    \255\255\255\255\255\255\108\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \107\000\255\255\107\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\108\000\255\255\255\255\255\255\255\255\255\255\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\035\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\255\255\255\255\255\255\255\255\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\255\255\255\255\255\255\255\255\150\000\
    \107\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\255\255\255\255\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\036\000\255\255\
    \036\000\036\000\036\000\036\000\255\255\255\255\255\255\036\000\
    \036\000\255\255\036\000\036\000\036\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\036\000\
    \255\255\036\000\036\000\036\000\036\000\036\000\255\255\255\255\
    \255\255\037\000\255\255\037\000\037\000\037\000\037\000\255\255\
    \255\255\255\255\037\000\037\000\255\255\037\000\037\000\037\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\036\000\037\000\036\000\037\000\037\000\037\000\037\000\
    \037\000\255\255\255\255\255\255\038\000\255\255\038\000\038\000\
    \038\000\038\000\255\255\255\255\255\255\038\000\038\000\255\255\
    \038\000\038\000\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\036\000\255\255\036\000\037\000\038\000\037\000\038\000\
    \038\000\038\000\038\000\038\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\255\255\037\000\038\000\
    \255\255\038\000\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\039\000\038\000\
    \255\255\038\000\255\255\255\255\255\255\255\255\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\255\255\255\255\255\255\255\255\039\000\255\255\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\255\255\255\255\255\255\255\255\
    \152\000\255\255\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\043\000\
    \255\255\255\255\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\043\000\255\255\
    \255\255\043\000\255\255\043\000\255\255\255\255\255\255\043\000\
    \043\000\255\255\043\000\255\255\043\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\043\000\043\000\043\000\043\000\255\255\255\255\
    \255\255\044\000\255\255\044\000\044\000\044\000\044\000\255\255\
    \255\255\255\255\044\000\044\000\255\255\044\000\044\000\044\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\043\000\044\000\044\000\044\000\044\000\
    \044\000\255\255\255\255\255\255\045\000\255\255\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\045\000\045\000\255\255\
    \045\000\045\000\045\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\255\255\043\000\044\000\045\000\044\000\045\000\
    \045\000\045\000\045\000\045\000\255\255\255\255\255\255\046\000\
    \255\255\046\000\046\000\046\000\046\000\255\255\255\255\255\255\
    \046\000\046\000\255\255\046\000\046\000\046\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\044\000\045\000\
    \046\000\045\000\046\000\046\000\046\000\046\000\046\000\255\255\
    \255\255\255\255\047\000\255\255\047\000\047\000\047\000\047\000\
    \255\255\255\255\255\255\047\000\047\000\255\255\047\000\047\000\
    \047\000\255\255\255\255\255\255\255\255\255\255\255\255\045\000\
    \255\255\045\000\046\000\047\000\046\000\047\000\047\000\047\000\
    \047\000\047\000\255\255\255\255\255\255\048\000\255\255\048\000\
    \048\000\048\000\048\000\255\255\255\255\255\255\048\000\048\000\
    \255\255\048\000\048\000\048\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\046\000\255\255\046\000\047\000\048\000\047\000\
    \048\000\048\000\048\000\048\000\048\000\255\255\255\255\255\255\
    \051\000\255\255\051\000\051\000\051\000\051\000\255\255\255\255\
    \255\255\051\000\051\000\255\255\051\000\051\000\051\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\047\000\255\255\047\000\
    \048\000\051\000\048\000\051\000\051\000\051\000\051\000\051\000\
    \255\255\255\255\255\255\053\000\255\255\053\000\053\000\053\000\
    \053\000\255\255\255\255\255\255\053\000\053\000\255\255\053\000\
    \053\000\053\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \048\000\255\255\048\000\051\000\053\000\051\000\053\000\053\000\
    \053\000\053\000\053\000\255\255\255\255\255\255\054\000\255\255\
    \054\000\054\000\054\000\054\000\255\255\255\255\255\255\054\000\
    \054\000\255\255\054\000\054\000\054\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\051\000\255\255\051\000\053\000\054\000\
    \053\000\054\000\054\000\054\000\054\000\054\000\255\255\255\255\
    \255\255\056\000\255\255\056\000\056\000\056\000\056\000\255\255\
    \255\255\255\255\056\000\056\000\255\255\056\000\056\000\056\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\053\000\255\255\
    \053\000\054\000\056\000\054\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\054\000\255\255\054\000\056\000\255\255\056\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\061\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \255\255\255\255\255\255\061\000\056\000\255\255\056\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\255\255\255\255\255\255\255\255\061\000\255\255\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\064\000\255\255\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\064\000\064\000\255\255\064\000\064\000\064\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\064\000\255\255\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\255\255\064\000\255\255\064\000\064\000\
    \255\255\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\255\255\064\000\065\000\064\000\065\000\
    \065\000\065\000\065\000\255\255\255\255\255\255\065\000\065\000\
    \255\255\065\000\065\000\065\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\065\000\255\255\
    \065\000\065\000\065\000\065\000\065\000\255\255\255\255\255\255\
    \066\000\255\255\066\000\066\000\066\000\066\000\255\255\255\255\
    \255\255\066\000\066\000\255\255\066\000\066\000\066\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \065\000\066\000\065\000\066\000\066\000\066\000\066\000\066\000\
    \255\255\255\255\255\255\255\255\068\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\255\255\
    \065\000\255\255\065\000\066\000\255\255\066\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \255\255\255\255\255\255\066\000\068\000\066\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \069\000\255\255\255\255\255\255\069\000\255\255\069\000\255\255\
    \255\255\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\255\255\255\255\255\255\255\255\
    \069\000\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\070\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \255\255\255\255\255\255\255\255\070\000\255\255\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \071\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\255\255\255\255\255\255\255\255\
    \071\000\255\255\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\072\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\255\255\072\000\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \073\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\255\255\255\255\255\255\255\255\
    \073\000\255\255\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\074\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \255\255\255\255\255\255\255\255\074\000\255\255\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \075\000\255\255\255\255\255\255\255\255\255\255\255\255\075\000\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\255\255\255\255\255\255\255\255\
    \075\000\255\255\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\076\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \255\255\255\255\255\255\255\255\076\000\255\255\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \077\000\255\255\255\255\255\255\077\000\255\255\077\000\255\255\
    \255\255\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\255\255\255\255\255\255\255\255\
    \077\000\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\078\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \255\255\255\255\255\255\255\255\078\000\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \080\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\255\255\255\255\255\255\255\255\
    \080\000\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\081\000\255\255\255\255\255\255\
    \081\000\255\255\081\000\255\255\255\255\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \255\255\255\255\255\255\255\255\081\000\255\255\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \082\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\255\255\255\255\255\255\255\255\
    \082\000\255\255\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\083\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \255\255\255\255\255\255\255\255\083\000\255\255\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \084\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\255\255\255\255\255\255\255\255\
    \084\000\255\255\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\089\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\089\000\255\255\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \255\255\255\255\255\255\255\255\089\000\255\255\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \090\000\255\255\090\000\255\255\255\255\255\255\255\255\090\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\109\000\255\255\109\000\109\000\109\000\109\000\
    \255\255\255\255\255\255\109\000\109\000\255\255\109\000\109\000\
    \109\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\109\000\255\255\109\000\109\000\109\000\
    \109\000\109\000\255\255\255\255\090\000\255\255\255\255\255\255\
    \255\255\255\255\090\000\111\000\255\255\111\000\111\000\111\000\
    \111\000\255\255\255\255\255\255\111\000\111\000\090\000\111\000\
    \111\000\111\000\090\000\255\255\090\000\109\000\255\255\109\000\
    \090\000\255\255\255\255\255\255\111\000\255\255\111\000\111\000\
    \111\000\111\000\111\000\255\255\255\255\255\255\112\000\255\255\
    \112\000\112\000\112\000\112\000\255\255\255\255\255\255\112\000\
    \112\000\255\255\112\000\112\000\112\000\109\000\255\255\109\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\111\000\112\000\
    \111\000\112\000\112\000\112\000\112\000\112\000\255\255\255\255\
    \255\255\121\000\255\255\121\000\121\000\121\000\121\000\255\255\
    \255\255\255\255\121\000\121\000\255\255\121\000\121\000\121\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\111\000\255\255\
    \111\000\112\000\121\000\112\000\121\000\121\000\121\000\121\000\
    \121\000\255\255\255\255\255\255\122\000\255\255\122\000\122\000\
    \122\000\122\000\255\255\255\255\255\255\122\000\122\000\255\255\
    \122\000\122\000\122\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\112\000\255\255\112\000\121\000\122\000\121\000\122\000\
    \122\000\122\000\122\000\122\000\255\255\255\255\255\255\123\000\
    \255\255\123\000\123\000\123\000\123\000\255\255\255\255\255\255\
    \123\000\123\000\255\255\123\000\123\000\123\000\255\255\255\255\
    \090\000\255\255\255\255\255\255\121\000\255\255\121\000\122\000\
    \123\000\122\000\123\000\123\000\123\000\123\000\123\000\255\255\
    \255\255\255\255\130\000\255\255\130\000\130\000\130\000\130\000\
    \255\255\255\255\255\255\130\000\130\000\255\255\130\000\130\000\
    \130\000\255\255\255\255\255\255\255\255\255\255\255\255\122\000\
    \255\255\122\000\123\000\130\000\123\000\130\000\130\000\130\000\
    \130\000\130\000\255\255\255\255\255\255\131\000\255\255\131\000\
    \131\000\131\000\131\000\255\255\255\255\255\255\131\000\131\000\
    \255\255\131\000\131\000\131\000\255\255\255\255\178\000\255\255\
    \255\255\178\000\123\000\255\255\123\000\130\000\131\000\130\000\
    \131\000\131\000\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\178\000\255\255\
    \255\255\255\255\255\255\178\000\255\255\255\255\178\000\255\255\
    \255\255\255\255\255\255\178\000\255\255\130\000\255\255\130\000\
    \131\000\255\255\131\000\255\255\255\255\255\255\255\255\255\255\
    \132\000\255\255\132\000\132\000\132\000\132\000\255\255\255\255\
    \255\255\132\000\132\000\255\255\132\000\132\000\132\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \131\000\132\000\131\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\255\255\132\000\255\255\132\000\132\000\178\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\133\000\132\000\255\255\132\000\255\255\255\255\
    \255\255\133\000\255\255\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\255\255\255\255\
    \255\255\255\255\133\000\255\255\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\134\000\255\255\
    \134\000\134\000\134\000\134\000\255\255\255\255\255\255\134\000\
    \134\000\255\255\134\000\134\000\134\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\178\000\255\255\255\255\134\000\
    \255\255\134\000\134\000\134\000\134\000\134\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\135\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\135\000\255\255\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \255\255\134\000\255\255\134\000\135\000\255\255\255\255\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\134\000\255\255\134\000\255\255\135\000\255\255\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\137\000\255\255\137\000\137\000\137\000\137\000\255\255\
    \255\255\255\255\137\000\137\000\255\255\137\000\137\000\137\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\137\000\255\255\137\000\137\000\137\000\137\000\
    \137\000\255\255\255\255\255\255\138\000\255\255\138\000\138\000\
    \138\000\138\000\255\255\255\255\255\255\138\000\138\000\255\255\
    \138\000\138\000\138\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\137\000\138\000\137\000\138\000\
    \138\000\138\000\138\000\138\000\255\255\255\255\255\255\147\000\
    \255\255\147\000\147\000\147\000\147\000\255\255\255\255\255\255\
    \147\000\147\000\255\255\147\000\147\000\147\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\137\000\255\255\137\000\138\000\
    \147\000\138\000\147\000\147\000\147\000\147\000\147\000\255\255\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\199\000\199\000\199\000\199\000\199\000\199\000\138\000\
    \255\255\138\000\147\000\255\255\147\000\255\255\255\255\255\255\
    \255\255\255\255\148\000\255\255\148\000\148\000\148\000\148\000\
    \255\255\255\255\255\255\148\000\148\000\255\255\148\000\148\000\
    \148\000\199\000\199\000\199\000\199\000\199\000\199\000\255\255\
    \255\255\255\255\147\000\148\000\147\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\255\255\148\000\255\255\148\000\
    \148\000\255\255\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\148\000\148\000\148\000\148\000\
    \148\000\148\000\148\000\148\000\255\255\148\000\151\000\148\000\
    \151\000\151\000\151\000\151\000\255\255\255\255\255\255\151\000\
    \151\000\255\255\151\000\151\000\151\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\151\000\
    \255\255\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \255\255\151\000\255\255\151\000\151\000\255\255\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \255\255\151\000\153\000\151\000\153\000\153\000\153\000\153\000\
    \255\255\255\255\255\255\153\000\153\000\255\255\153\000\153\000\
    \153\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\153\000\255\255\153\000\153\000\153\000\
    \153\000\153\000\255\255\255\255\255\255\154\000\255\255\154\000\
    \154\000\154\000\154\000\255\255\255\255\255\255\154\000\154\000\
    \255\255\154\000\154\000\154\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\153\000\154\000\153\000\
    \154\000\154\000\154\000\154\000\154\000\255\255\255\255\255\255\
    \255\255\255\255\156\000\255\255\156\000\156\000\156\000\156\000\
    \255\255\255\255\255\255\156\000\156\000\255\255\156\000\156\000\
    \156\000\255\255\255\255\255\255\255\255\153\000\255\255\153\000\
    \154\000\154\000\154\000\156\000\255\255\156\000\156\000\156\000\
    \156\000\156\000\255\255\255\255\255\255\157\000\255\255\157\000\
    \157\000\157\000\157\000\255\255\255\255\255\255\157\000\157\000\
    \255\255\157\000\157\000\157\000\255\255\255\255\255\255\255\255\
    \154\000\255\255\154\000\255\255\255\255\156\000\157\000\156\000\
    \157\000\157\000\157\000\157\000\157\000\255\255\255\255\255\255\
    \158\000\255\255\158\000\158\000\158\000\158\000\255\255\255\255\
    \255\255\158\000\158\000\255\255\158\000\158\000\158\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\156\000\255\255\156\000\
    \157\000\158\000\157\000\158\000\158\000\158\000\158\000\158\000\
    \255\255\255\255\255\255\159\000\255\255\159\000\159\000\159\000\
    \159\000\255\255\255\255\255\255\159\000\159\000\255\255\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \157\000\255\255\157\000\158\000\159\000\158\000\159\000\159\000\
    \159\000\159\000\159\000\255\255\255\255\255\255\160\000\255\255\
    \160\000\160\000\160\000\160\000\255\255\255\255\255\255\160\000\
    \160\000\255\255\160\000\160\000\160\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\158\000\255\255\158\000\159\000\160\000\
    \159\000\160\000\160\000\160\000\160\000\160\000\255\255\255\255\
    \255\255\163\000\255\255\163\000\163\000\163\000\163\000\255\255\
    \255\255\255\255\163\000\163\000\255\255\163\000\163\000\163\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\159\000\255\255\
    \159\000\160\000\163\000\160\000\163\000\163\000\163\000\163\000\
    \163\000\255\255\255\255\255\255\166\000\255\255\166\000\166\000\
    \166\000\166\000\255\255\255\255\255\255\166\000\166\000\255\255\
    \166\000\166\000\166\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\160\000\255\255\160\000\163\000\166\000\163\000\166\000\
    \166\000\166\000\166\000\166\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\163\000\255\255\163\000\166\000\
    \184\000\166\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\184\000\184\000\184\000\184\000\
    \184\000\184\000\184\000\184\000\255\255\184\000\255\255\166\000\
    \191\000\166\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\192\000\191\000\192\000\255\255\
    \255\255\255\255\255\255\192\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \192\000\255\255\255\255\255\255\255\255\255\255\192\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\255\255\255\255\255\255\
    \255\255\255\255\192\000\255\255\255\255\255\255\192\000\215\000\
    \192\000\255\255\215\000\255\255\192\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\206\000\
    \206\000\206\000\206\000\206\000\206\000\215\000\255\255\215\000\
    \255\255\255\255\255\255\255\255\215\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\215\000\255\255\255\255\255\255\255\255\255\255\215\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\255\255\255\255\
    \255\255\255\255\255\255\215\000\255\255\255\255\255\255\215\000\
    \255\255\215\000\255\255\255\255\255\255\215\000\226\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\255\255\226\000\
    \226\000\226\000\226\000\226\000\226\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\226\000\
    \226\000\226\000\226\000\226\000\226\000\231\000\255\255\231\000\
    \231\000\231\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\255\255\237\000\231\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\255\255\255\255\
    \237\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\215\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\010\000\036\000\000\000\000\000\000\000\
    \000\000\012\000\000\000\000\000\000\000\002\000\000\000\027\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\000\004\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\039\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\019\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\036\000\036\000\000\000\036\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\022\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\001\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\026\000\116\000\215\000\222\000\116\000\223\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \026\000\255\255\116\000\000\000\117\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\107\000\108\000\255\255\255\255\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\108\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \116\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\004\255\255\005\255\255\007\255\006\255\255\003\255\000\004\
    \001\005\255\007\255\255\006\255\007\255\255\000\004\001\005\003\
    \006\002\007\255\001\255\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 8 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 404 "src/reason-parser/reason_lexer.mll"
                 (
      match !preprocessor with
      | None ->
        raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      | Some _ ->
        update_loc lexbuf None 1 false 0;
        token lexbuf )
# 2152 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 413 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        match !preprocessor with
        | None -> token lexbuf
        | Some _ -> EOL
      )
# 2161 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 419 "src/reason-parser/reason_lexer.mll"
      ( token lexbuf )
# 2166 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 421 "src/reason-parser/reason_lexer.mll"
      ( UNDERSCORE )
# 2171 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 423 "src/reason-parser/reason_lexer.mll"
      ( TILDE )
# 2176 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 425 "src/reason-parser/reason_lexer.mll"
      ( QUESTION )
# 2181 "src/reason-parser/reason_lexer.ml"

  | 6 ->
# 427 "src/reason-parser/reason_lexer.mll"
      ( set_lexeme_length lexbuf 1; EQUAL )
# 2186 "src/reason-parser/reason_lexer.ml"

  | 7 ->
# 429 "src/reason-parser/reason_lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        try Hashtbl.find keyword_table s
        with Not_found -> LIDENT s )
# 2193 "src/reason-parser/reason_lexer.ml"

  | 8 ->
# 433 "src/reason-parser/reason_lexer.mll"
      ( warn_latin1 lexbuf; LIDENT (Lexing.lexeme lexbuf) )
# 2198 "src/reason-parser/reason_lexer.ml"

  | 9 ->
# 435 "src/reason-parser/reason_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        LABEL_WITH_EQUAL(String.sub l 1 (String.length l - 2)) )
# 2204 "src/reason-parser/reason_lexer.ml"

  | 10 ->
# 438 "src/reason-parser/reason_lexer.mll"
      ( UIDENT(Lexing.lexeme lexbuf) )
# 2209 "src/reason-parser/reason_lexer.ml"

  | 11 ->
# 440 "src/reason-parser/reason_lexer.mll"
      ( warn_latin1 lexbuf; UIDENT(Lexing.lexeme lexbuf) )
# 2214 "src/reason-parser/reason_lexer.ml"

  | 12 ->
# 441 "src/reason-parser/reason_lexer.mll"
                ( INT (Lexing.lexeme lexbuf, None) )
# 2219 "src/reason-parser/reason_lexer.ml"

  | 13 ->
let
# 442 "src/reason-parser/reason_lexer.mll"
                    lit
# 2225 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 442 "src/reason-parser/reason_lexer.mll"
                                              modif
# 2230 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 443 "src/reason-parser/reason_lexer.mll"
      ( INT (lit, Some modif) )
# 2234 "src/reason-parser/reason_lexer.ml"

  | 14 ->
# 445 "src/reason-parser/reason_lexer.mll"
      ( FLOAT (Lexing.lexeme lexbuf, None) )
# 2239 "src/reason-parser/reason_lexer.ml"

  | 15 ->
let
# 446 "src/reason-parser/reason_lexer.mll"
                                            lit
# 2245 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 446 "src/reason-parser/reason_lexer.mll"
                                                                      modif
# 2250 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 447 "src/reason-parser/reason_lexer.mll"
      ( FLOAT (lit, Some modif) )
# 2254 "src/reason-parser/reason_lexer.ml"

  | 16 ->
# 449 "src/reason-parser/reason_lexer.mll"
      ( raise (Error(Invalid_literal (Lexing.lexeme lexbuf),
                     Location.curr lexbuf)) )
# 2260 "src/reason-parser/reason_lexer.ml"

  | 17 ->
# 452 "src/reason-parser/reason_lexer.mll"
      ( reset_string_buffer();
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        string lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string(), None) )
# 2272 "src/reason-parser/reason_lexer.ml"

  | 18 ->
# 461 "src/reason-parser/reason_lexer.mll"
      ( reset_string_buffer();
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        quoted_string delim lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string(), Some delim) )
# 2286 "src/reason-parser/reason_lexer.ml"

  | 19 ->
# 472 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        CHAR (Lexing.lexeme_char lexbuf 1) )
# 2292 "src/reason-parser/reason_lexer.ml"

  | 20 ->
# 475 "src/reason-parser/reason_lexer.mll"
      ( CHAR(Lexing.lexeme_char lexbuf 1) )
# 2297 "src/reason-parser/reason_lexer.ml"

  | 21 ->
# 477 "src/reason-parser/reason_lexer.mll"
      ( CHAR(char_for_backslash (Lexing.lexeme_char lexbuf 2)) )
# 2302 "src/reason-parser/reason_lexer.ml"

  | 22 ->
# 479 "src/reason-parser/reason_lexer.mll"
      ( CHAR(char_for_decimal_code lexbuf 2) )
# 2307 "src/reason-parser/reason_lexer.ml"

  | 23 ->
# 481 "src/reason-parser/reason_lexer.mll"
      ( CHAR(char_for_hexadecimal_code lexbuf 3) )
# 2312 "src/reason-parser/reason_lexer.ml"

  | 24 ->
# 483 "src/reason-parser/reason_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let esc = String.sub l 1 (String.length l - 1) in
        raise (Error(Illegal_escape esc, Location.curr lexbuf))
      )
# 2320 "src/reason-parser/reason_lexer.ml"

  | 25 ->
# 487 "src/reason-parser/reason_lexer.mll"
          (
    (* Allow parsing of foo#=<bar /> *)
    set_lexeme_length lexbuf 2;
    SHARPOP("#=")
  )
# 2329 "src/reason-parser/reason_lexer.ml"

  | 26 ->
# 493 "src/reason-parser/reason_lexer.mll"
      ( SHARPOP(lexeme_operator lexbuf) )
# 2334 "src/reason-parser/reason_lexer.ml"

  | 27 ->
let
# 494 "src/reason-parser/reason_lexer.mll"
                                   num
# 2340 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 495 "src/reason-parser/reason_lexer.mll"
                                           name
# 2345 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 497 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf name (int_of_string num) true 0;
        token lexbuf
      )
# 2351 "src/reason-parser/reason_lexer.ml"

  | 28 ->
# 500 "src/reason-parser/reason_lexer.mll"
         ( AMPERSAND )
# 2356 "src/reason-parser/reason_lexer.ml"

  | 29 ->
# 501 "src/reason-parser/reason_lexer.mll"
         ( AMPERAMPER )
# 2361 "src/reason-parser/reason_lexer.ml"

  | 30 ->
# 502 "src/reason-parser/reason_lexer.mll"
         ( BACKQUOTE )
# 2366 "src/reason-parser/reason_lexer.ml"

  | 31 ->
# 503 "src/reason-parser/reason_lexer.mll"
         ( QUOTE )
# 2371 "src/reason-parser/reason_lexer.ml"

  | 32 ->
# 504 "src/reason-parser/reason_lexer.mll"
         ( LPAREN )
# 2376 "src/reason-parser/reason_lexer.ml"

  | 33 ->
# 505 "src/reason-parser/reason_lexer.mll"
         ( RPAREN )
# 2381 "src/reason-parser/reason_lexer.ml"

  | 34 ->
# 506 "src/reason-parser/reason_lexer.mll"
         ( STAR )
# 2386 "src/reason-parser/reason_lexer.ml"

  | 35 ->
# 507 "src/reason-parser/reason_lexer.mll"
         ( COMMA )
# 2391 "src/reason-parser/reason_lexer.ml"

  | 36 ->
# 508 "src/reason-parser/reason_lexer.mll"
         ( MINUSGREATER )
# 2396 "src/reason-parser/reason_lexer.ml"

  | 37 ->
# 509 "src/reason-parser/reason_lexer.mll"
         ( EQUALGREATER )
# 2401 "src/reason-parser/reason_lexer.ml"

  | 38 ->
# 510 "src/reason-parser/reason_lexer.mll"
         ( SHARP )
# 2406 "src/reason-parser/reason_lexer.ml"

  | 39 ->
# 511 "src/reason-parser/reason_lexer.mll"
         ( DOT )
# 2411 "src/reason-parser/reason_lexer.ml"

  | 40 ->
# 512 "src/reason-parser/reason_lexer.mll"
         ( DOTDOT )
# 2416 "src/reason-parser/reason_lexer.ml"

  | 41 ->
# 513 "src/reason-parser/reason_lexer.mll"
         ( DOTDOTDOT )
# 2421 "src/reason-parser/reason_lexer.ml"

  | 42 ->
# 514 "src/reason-parser/reason_lexer.mll"
         ( COLON )
# 2426 "src/reason-parser/reason_lexer.ml"

  | 43 ->
# 515 "src/reason-parser/reason_lexer.mll"
         ( COLONCOLON )
# 2431 "src/reason-parser/reason_lexer.ml"

  | 44 ->
# 516 "src/reason-parser/reason_lexer.mll"
         ( COLONEQUAL )
# 2436 "src/reason-parser/reason_lexer.ml"

  | 45 ->
# 517 "src/reason-parser/reason_lexer.mll"
         ( COLONGREATER )
# 2441 "src/reason-parser/reason_lexer.ml"

  | 46 ->
# 518 "src/reason-parser/reason_lexer.mll"
         ( SEMI )
# 2446 "src/reason-parser/reason_lexer.ml"

  | 47 ->
# 519 "src/reason-parser/reason_lexer.mll"
         ( SEMISEMI )
# 2451 "src/reason-parser/reason_lexer.ml"

  | 48 ->
# 520 "src/reason-parser/reason_lexer.mll"
         ( LESS )
# 2456 "src/reason-parser/reason_lexer.ml"

  | 49 ->
# 521 "src/reason-parser/reason_lexer.mll"
         ( EQUAL )
# 2461 "src/reason-parser/reason_lexer.ml"

  | 50 ->
# 522 "src/reason-parser/reason_lexer.mll"
         ( LBRACKET )
# 2466 "src/reason-parser/reason_lexer.ml"

  | 51 ->
# 523 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETBAR )
# 2471 "src/reason-parser/reason_lexer.ml"

  | 52 ->
# 524 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETLESS )
# 2476 "src/reason-parser/reason_lexer.ml"

  | 53 ->
# 525 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETGREATER )
# 2481 "src/reason-parser/reason_lexer.ml"

  | 54 ->
# 526 "src/reason-parser/reason_lexer.mll"
                                                   (
    let buf = Lexing.lexeme lexbuf in
    LESSIDENT (String.sub buf 1 (String.length buf - 1))
  )
# 2489 "src/reason-parser/reason_lexer.ml"

  | 55 ->
# 530 "src/reason-parser/reason_lexer.mll"
                                                    (
    (* allows parsing of `{<Text` in <Description term={<Text text="Age" />}> as correct jsx *)
    set_lexeme_length lexbuf 1;
    LBRACE
  )
# 2498 "src/reason-parser/reason_lexer.ml"

  | 56 ->
# 535 "src/reason-parser/reason_lexer.mll"
                                                        (
    let buf = Lexing.lexeme lexbuf in
    LESSSLASHIDENTGREATER (String.sub buf 2 (String.length buf - 2 - 1))
  )
# 2506 "src/reason-parser/reason_lexer.ml"

  | 57 ->
# 539 "src/reason-parser/reason_lexer.mll"
         ( RBRACKET )
# 2511 "src/reason-parser/reason_lexer.ml"

  | 58 ->
# 540 "src/reason-parser/reason_lexer.mll"
         ( LBRACE )
# 2516 "src/reason-parser/reason_lexer.ml"

  | 59 ->
# 541 "src/reason-parser/reason_lexer.mll"
         ( LBRACELESS )
# 2521 "src/reason-parser/reason_lexer.ml"

  | 60 ->
# 542 "src/reason-parser/reason_lexer.mll"
         ( BAR )
# 2526 "src/reason-parser/reason_lexer.ml"

  | 61 ->
# 543 "src/reason-parser/reason_lexer.mll"
         ( BARBAR )
# 2531 "src/reason-parser/reason_lexer.ml"

  | 62 ->
# 544 "src/reason-parser/reason_lexer.mll"
         ( BARRBRACKET )
# 2536 "src/reason-parser/reason_lexer.ml"

  | 63 ->
# 545 "src/reason-parser/reason_lexer.mll"
         ( GREATER )
# 2541 "src/reason-parser/reason_lexer.ml"

  | 64 ->
# 551 "src/reason-parser/reason_lexer.mll"
         ( RBRACE )
# 2546 "src/reason-parser/reason_lexer.ml"

  | 65 ->
# 552 "src/reason-parser/reason_lexer.mll"
         ( GREATERRBRACE )
# 2551 "src/reason-parser/reason_lexer.ml"

  | 66 ->
# 553 "src/reason-parser/reason_lexer.mll"
                                 (
    (* allow `let x=<div />;` *)
    set_lexeme_length lexbuf 1;
    EQUAL
  )
# 2560 "src/reason-parser/reason_lexer.ml"

  | 67 ->
# 559 "src/reason-parser/reason_lexer.mll"
           (
    set_lexeme_length lexbuf 2;
    SLASHGREATER
  )
# 2568 "src/reason-parser/reason_lexer.ml"

  | 68 ->
# 563 "src/reason-parser/reason_lexer.mll"
          (
    set_lexeme_length lexbuf 2;
    LBRACKETBAR
  )
# 2576 "src/reason-parser/reason_lexer.ml"

  | 69 ->
# 567 "src/reason-parser/reason_lexer.mll"
                                  (
    (* allow parsing of <div asd=1></div> *)
    set_lexeme_length lexbuf 1;
    GREATER
  )
# 2585 "src/reason-parser/reason_lexer.ml"

  | 70 ->
# 572 "src/reason-parser/reason_lexer.mll"
                                 (
    (* allow parsing of <div><span> *)
    set_lexeme_length lexbuf 1;
    GREATER
  )
# 2594 "src/reason-parser/reason_lexer.ml"

  | 71 ->
# 577 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETAT )
# 2599 "src/reason-parser/reason_lexer.ml"

  | 72 ->
# 578 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETPERCENT )
# 2604 "src/reason-parser/reason_lexer.ml"

  | 73 ->
# 579 "src/reason-parser/reason_lexer.mll"
          ( LBRACKETPERCENTPERCENT )
# 2609 "src/reason-parser/reason_lexer.ml"

  | 74 ->
# 580 "src/reason-parser/reason_lexer.mll"
         ( BANG )
# 2614 "src/reason-parser/reason_lexer.ml"

  | 75 ->
# 581 "src/reason-parser/reason_lexer.mll"
         ( INFIXOP0 "!=" )
# 2619 "src/reason-parser/reason_lexer.ml"

  | 76 ->
# 582 "src/reason-parser/reason_lexer.mll"
          ( INFIXOP0 "!==" )
# 2624 "src/reason-parser/reason_lexer.ml"

  | 77 ->
# 583 "src/reason-parser/reason_lexer.mll"
           ( INFIXOP0 "!=" )
# 2629 "src/reason-parser/reason_lexer.ml"

  | 78 ->
# 584 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP0 "!==" )
# 2634 "src/reason-parser/reason_lexer.ml"

  | 79 ->
# 585 "src/reason-parser/reason_lexer.mll"
         ( PLUS )
# 2639 "src/reason-parser/reason_lexer.ml"

  | 80 ->
# 586 "src/reason-parser/reason_lexer.mll"
         ( PLUSDOT )
# 2644 "src/reason-parser/reason_lexer.ml"

  | 81 ->
# 587 "src/reason-parser/reason_lexer.mll"
         ( PLUSEQ )
# 2649 "src/reason-parser/reason_lexer.ml"

  | 82 ->
# 588 "src/reason-parser/reason_lexer.mll"
         ( MINUS )
# 2654 "src/reason-parser/reason_lexer.ml"

  | 83 ->
# 589 "src/reason-parser/reason_lexer.mll"
         ( MINUSDOT )
# 2659 "src/reason-parser/reason_lexer.ml"

  | 84 ->
# 590 "src/reason-parser/reason_lexer.mll"
         ( LESSGREATER )
# 2664 "src/reason-parser/reason_lexer.ml"

  | 85 ->
# 591 "src/reason-parser/reason_lexer.mll"
          ( LESSSLASHGREATER )
# 2669 "src/reason-parser/reason_lexer.ml"

  | 86 ->
# 592 "src/reason-parser/reason_lexer.mll"
           ( LESSDOTDOTGREATER )
# 2674 "src/reason-parser/reason_lexer.ml"

  | 87 ->
# 594 "src/reason-parser/reason_lexer.mll"
            ( PREFIXOP(lexeme_operator lexbuf) )
# 2679 "src/reason-parser/reason_lexer.ml"

  | 88 ->
# 596 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP0(lexeme_operator lexbuf) )
# 2684 "src/reason-parser/reason_lexer.ml"

  | 89 ->
# 598 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP1(lexeme_operator lexbuf) )
# 2689 "src/reason-parser/reason_lexer.ml"

  | 90 ->
# 600 "src/reason-parser/reason_lexer.mll"
            ( match lexeme_without_comment lexbuf with
              | "^." -> set_lexeme_length lexbuf 1; POSTFIXOP("^")
              | "^" -> POSTFIXOP("^")
              | op -> INFIXOP1(unescape_operator op) )
# 2697 "src/reason-parser/reason_lexer.ml"

  | 91 ->
# 605 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP1(lexeme_operator lexbuf) )
# 2702 "src/reason-parser/reason_lexer.ml"

  | 92 ->
# 607 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP2(lexeme_operator lexbuf) )
# 2707 "src/reason-parser/reason_lexer.ml"

  | 93 ->
# 609 "src/reason-parser/reason_lexer.mll"
         ( SLASHGREATER )
# 2712 "src/reason-parser/reason_lexer.ml"

  | 94 ->
# 620 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP4(lexeme_operator lexbuf) )
# 2717 "src/reason-parser/reason_lexer.ml"

  | 95 ->
# 621 "src/reason-parser/reason_lexer.mll"
            ( PERCENT )
# 2722 "src/reason-parser/reason_lexer.ml"

  | 96 ->
# 623 "src/reason-parser/reason_lexer.mll"
            ( match lexeme_operator lexbuf with
              | "" ->
                  (* If the operator is empty, it means the lexeme is beginning
                   * by a comment sequence: we let the comment lexer handle
                   * the case. *)
                  enter_comment lexbuf
              | op -> INFIXOP3 op )
# 2733 "src/reason-parser/reason_lexer.ml"

  | 97 ->
# 631 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP3(lexeme_operator lexbuf) )
# 2738 "src/reason-parser/reason_lexer.ml"

  | 98 ->
# 632 "src/reason-parser/reason_lexer.mll"
        ( EOF )
# 2743 "src/reason-parser/reason_lexer.ml"

  | 99 ->
# 634 "src/reason-parser/reason_lexer.mll"
      ( raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      )
# 2750 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and enter_comment lexbuf =
    __ocaml_lex_enter_comment_rec lexbuf 168
and __ocaml_lex_enter_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 640 "src/reason-parser/reason_lexer.mll"
      ( set_lexeme_length lexbuf 2;
        let start_loc = Location.curr lexbuf  in
        comment_start_loc := [start_loc];
        reset_string_buffer ();
        let {Location. loc_end; _} = comment lexbuf in
        let s = get_stored_string () in
        reset_string_buffer ();
        COMMENT (s, { start_loc with Location.loc_end })
      )
# 2770 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 650 "src/reason-parser/reason_lexer.mll"
      ( let start_p = lexbuf.Lexing.lex_start_p in
        let start_loc = Location.curr lexbuf in
        comment_start_loc := [start_loc];
        reset_string_buffer ();
        let _ = comment lexbuf in
        let s = get_stored_string () in
        reset_string_buffer ();
        lexbuf.Lexing.lex_start_p <- start_p;
        DOCSTRING s
      )
# 2784 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 661 "src/reason-parser/reason_lexer.mll"
      ( DOCSTRING "" )
# 2789 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 663 "src/reason-parser/reason_lexer.mll"
      ( let loc = Location.curr lexbuf  in
        if !print_warnings then
          Location.prerr_warning loc Warnings.Comment_start;
        comment_start_loc := [loc];
        reset_string_buffer ();
        let {Location. loc_end; _} = comment lexbuf in
        let s = get_stored_string () in
        reset_string_buffer ();
        COMMENT (s, { loc with Location.loc_end })
      )
# 2803 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 674 "src/reason-parser/reason_lexer.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_not_end;
        set_lexeme_length lexbuf 1;
        STAR
      )
# 2812 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 679 "src/reason-parser/reason_lexer.mll"
      ( assert false )
# 2817 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_enter_comment_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 178
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 683 "src/reason-parser/reason_lexer.mll"
      ( comment_start_loc := (Location.curr lexbuf) :: !comment_start_loc;
        store_lexeme lexbuf;
        comment lexbuf;
      )
# 2832 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 688 "src/reason-parser/reason_lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | [_] -> comment_start_loc := []; Location.curr lexbuf
        | _ :: l -> comment_start_loc := l;
                  store_lexeme lexbuf;
                  comment lexbuf;
       )
# 2843 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 696 "src/reason-parser/reason_lexer.mll"
      (
        string_start_loc := Location.curr lexbuf;
        store_string_char '"';
        is_in_string := true;
        begin try string lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '"';
        comment lexbuf )
# 2864 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 714 "src/reason-parser/reason_lexer.mll"
      (
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        string_start_loc := Location.curr lexbuf;
        store_lexeme lexbuf;
        is_in_string := true;
        begin try quoted_string delim lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '|';
        store_string delim;
        store_string_char '}';
        comment lexbuf )
# 2889 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 737 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2894 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 739 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2902 "src/reason-parser/reason_lexer.ml"

  | 6 ->
# 744 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2907 "src/reason-parser/reason_lexer.ml"

  | 7 ->
# 746 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2912 "src/reason-parser/reason_lexer.ml"

  | 8 ->
# 748 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2917 "src/reason-parser/reason_lexer.ml"

  | 9 ->
# 750 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2922 "src/reason-parser/reason_lexer.ml"

  | 10 ->
# 752 "src/reason-parser/reason_lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | loc :: _ ->
          let start = List.hd (List.rev !comment_start_loc) in
          comment_start_loc := [];
          raise (Error (Unterminated_comment start, loc))
      )
# 2933 "src/reason-parser/reason_lexer.ml"

  | 11 ->
# 760 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2941 "src/reason-parser/reason_lexer.ml"

  | 12 ->
# 765 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2946 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1) ;   __ocaml_lex_string_rec lexbuf 210
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 769 "src/reason-parser/reason_lexer.mll"
      ( () )
# 2958 "src/reason-parser/reason_lexer.ml"

  | 1 ->
let
# 770 "src/reason-parser/reason_lexer.mll"
                                  space
# 2964 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 771 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        if in_comment () then store_lexeme lexbuf;
        string lexbuf
      )
# 2971 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 776 "src/reason-parser/reason_lexer.mll"
      ( if in_comment () then store_lexeme lexbuf
        else store_string_char(char_for_backslash(Lexing.lexeme_char lexbuf 1));
        string lexbuf )
# 2978 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 780 "src/reason-parser/reason_lexer.mll"
      ( if in_comment () then store_lexeme lexbuf
        else store_string_char(char_for_decimal_code lexbuf 1);
        string lexbuf )
# 2985 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 784 "src/reason-parser/reason_lexer.mll"
      ( if in_comment () then store_lexeme lexbuf
        else store_string_char(char_for_hexadecimal_code lexbuf 2);
        string lexbuf )
# 2992 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 788 "src/reason-parser/reason_lexer.mll"
      ( if in_comment ()
        then string lexbuf
        else begin
(*  Should be an error, but we are very lax.
          raise (Error (Illegal_escape (Lexing.lexeme lexbuf),
                        Location.curr lexbuf))
*)
          let loc = Location.curr lexbuf in
          Location.prerr_warning loc Warnings.Illegal_backslash;
          store_string_char (Lexing.lexeme_char lexbuf 0);
          store_string_char (Lexing.lexeme_char lexbuf 1);
          string lexbuf
        end
      )
# 3010 "src/reason-parser/reason_lexer.ml"

  | 6 ->
# 803 "src/reason-parser/reason_lexer.mll"
      ( if not (in_comment ()) then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Eol_in_string;
        update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        string lexbuf
      )
# 3020 "src/reason-parser/reason_lexer.ml"

  | 7 ->
# 810 "src/reason-parser/reason_lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 3026 "src/reason-parser/reason_lexer.ml"

  | 8 ->
# 813 "src/reason-parser/reason_lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        string lexbuf )
# 3032 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and quoted_string delim lexbuf =
    __ocaml_lex_quoted_string_rec delim lexbuf 229
and __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 818 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        quoted_string delim lexbuf
      )
# 3047 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 823 "src/reason-parser/reason_lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 3053 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 826 "src/reason-parser/reason_lexer.mll"
      (
        let edelim = Lexing.lexeme lexbuf in
        let edelim = String.sub edelim 1 (String.length edelim - 2) in
        if delim = edelim then ()
        else (store_lexeme lexbuf; quoted_string delim lexbuf)
      )
# 3063 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 833 "src/reason-parser/reason_lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        quoted_string delim lexbuf )
# 3069 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state

and skip_sharp_bang lexbuf =
    __ocaml_lex_skip_sharp_bang_rec lexbuf 238
and __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 838 "src/reason-parser/reason_lexer.mll"
       ( update_loc lexbuf None 3 false 0 )
# 3081 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 840 "src/reason-parser/reason_lexer.mll"
       ( update_loc lexbuf None 1 false 0 )
# 3086 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 841 "src/reason-parser/reason_lexer.mll"
       ( () )
# 3091 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state

;;

# 843 "src/reason-parser/reason_lexer.mll"
 

  (* Filter commnets *)

  let token_with_comments lexbuf =
    match !preprocessor with
    | None -> token lexbuf
    | Some (_init, preprocess) -> preprocess token lexbuf

  let last_comments = ref []

  let rec token lexbuf =
    match token_with_comments lexbuf with
        COMMENT (s, comment_loc) ->
          last_comments := (s, comment_loc) :: !last_comments;
          token lexbuf
      | tok -> tok

  let add_invalid_docstring text loc =
    let open Location in
    let rec aux = function
      | ((_, loc') as x) :: xs
        when loc'.loc_start.pos_cnum > loc.loc_start.pos_cnum ->
        x :: aux xs
      | xs -> (text, loc) :: xs
    in
    last_comments := aux !last_comments

  let comments () = List.rev !last_comments

  (* Routines for manipulating lexer state *)

  let save_triple lexbuf tok =
    (tok, lexbuf.lex_start_p, lexbuf.lex_curr_p)

  let load_triple lexbuf (tok, p1, p2) = (
    lexbuf.lex_start_p <- p1;
    lexbuf.lex_curr_p <- p2;
    tok
  )

  let fake_triple t (_, pos, _) =
    (t, pos, pos)

  (* insert ES6_FUN *)

  exception Lex_balanced_failed of (token * position * position) list *
                                   (exn * position * position) option

  let closing_of = function
    | LPAREN -> RPAREN
    | LBRACE -> RBRACE
    | _ -> assert false

  let inject_es6_fun = function
    | tok :: acc ->
      tok :: fake_triple ES6_FUN tok :: acc
    | _ -> assert false

  let is_triggering_token = function
    | EQUALGREATER | COLON -> true
    | _ -> false

  let rec lex_balanced_step closing lexbuf acc tok =
    let acc = save_triple lexbuf tok :: acc in
    match tok, closing with
    | (RPAREN, RPAREN) | (RBRACE, RBRACE) | (RBRACKET, RBRACKET) ->
      acc
    | ((RPAREN | RBRACE | RBRACKET | EOF), _) ->
      raise (Lex_balanced_failed (acc, None))
    | (( LBRACKET | LBRACKETLESS | LBRACKETGREATER
       | LBRACKETAT
       | LBRACKETPERCENT | LBRACKETPERCENTPERCENT ), _) ->
      lex_balanced closing lexbuf (lex_balanced RBRACKET lexbuf acc)
    | ((LPAREN | LBRACE), _) ->
      let rparen =
        try lex_balanced (closing_of tok) lexbuf []
        with (Lex_balanced_failed (rparen, None)) ->
          raise (Lex_balanced_failed (rparen @ acc, None))
      in
      begin match token lexbuf with
      | exception exn ->
        raise (Lex_balanced_failed (rparen @ acc, Some (save_triple lexbuf exn)))
      | tok' ->
        let acc = if is_triggering_token tok' then inject_es6_fun acc else acc in
        lex_balanced_step closing lexbuf (rparen @ acc) tok'
      end
    | (LIDENT _, _) ->
      begin match token lexbuf with
      | exception exn ->
        raise (Lex_balanced_failed (acc, Some (save_triple lexbuf exn)))
      | tok' ->
        let acc = if is_triggering_token tok' then inject_es6_fun acc else acc in
        lex_balanced_step closing lexbuf acc tok'
      end
    | _ -> lex_balanced closing lexbuf acc

  and lex_balanced closing lexbuf acc =
    match token lexbuf with
    | exception exn ->
      raise (Lex_balanced_failed (acc, Some (save_triple lexbuf exn)))
    | tok -> lex_balanced_step closing lexbuf acc tok

  let queued_tokens = ref []
  let queued_exn = ref None

  let lookahead_esfun lexbuf (tok, _, _ as lparen) =
    let triple =
      match lex_balanced (closing_of tok) lexbuf [] with
      | exception (Lex_balanced_failed (tokens, exn)) ->
        queued_tokens := List.rev tokens;
        queued_exn := exn;
        lparen
      | tokens ->
        begin match token lexbuf with
          | exception exn ->
            queued_tokens := List.rev tokens;
            queued_exn := Some (save_triple lexbuf exn);
            lparen
          | token ->
            let tokens = save_triple lexbuf token :: tokens in
            if is_triggering_token token then (
              queued_tokens := lparen :: List.rev tokens;
              fake_triple ES6_FUN lparen
            ) else (
              queued_tokens := List.rev tokens;
              lparen
            )
        end
    in
    load_triple lexbuf triple

  let token lexbuf =
    match !queued_tokens, !queued_exn with
    | [], Some exn ->
      queued_exn := None;
      raise (load_triple lexbuf exn)
    | [(LPAREN, _, _) as lparen], None ->
      let _ = load_triple lexbuf lparen in
      lookahead_esfun lexbuf lparen
    | [(LBRACE, _, _) as lparen], None ->
      let _ = load_triple lexbuf lparen in
      lookahead_esfun lexbuf lparen
    | [], None ->
      begin match token lexbuf with
      | LPAREN | LBRACE as tok ->
          lookahead_esfun lexbuf (save_triple lexbuf tok)
      | LIDENT _ as tok ->
          let tok = save_triple lexbuf tok in
          begin match token lexbuf with
          | exception exn ->
              queued_exn := Some (save_triple lexbuf exn);
              load_triple lexbuf tok
          | tok' ->
            if is_triggering_token tok' then (
              queued_tokens := [tok; save_triple lexbuf tok'];
              load_triple lexbuf (fake_triple ES6_FUN tok)
            ) else (
              queued_tokens := [save_triple lexbuf tok'];
              load_triple lexbuf tok
            )
          end
      | token -> token
      end
    | x :: xs, _ -> queued_tokens := xs; load_triple lexbuf x

  let init () =
    is_in_string := false;
    last_comments := [];
    comment_start_loc := [];
    queued_tokens := [];
    queued_exn := None;
    match !preprocessor with
    | None -> ()
    | Some (init, _preprocess) -> init ()

  let set_preprocessor init preprocess =
    preprocessor := Some (init, preprocess)


# 3279 "src/reason-parser/reason_lexer.ml"
