# 68 "src/reason-parser/reason_lexer.mll"
 
open Lexing
open Reason_parser
open Lexer_warning

type error =
  | Illegal_character of char
  | Illegal_escape of string
  | Unterminated_comment of Location.t
  | Unterminated_string
  | Unterminated_string_in_comment of Location.t * Location.t
  | Keyword_as_label of string
  | Literal_overflow of string
  | Invalid_literal of string
;;

exception Error of error * Location.t;;

(* The table of keywords *)

let keyword_table =
  let create_hashtable n l =
    let t = Hashtbl.create n in
    List.iter (fun (k, v) -> Hashtbl.add t k v) l;
    t
  in
  create_hashtable 149 [
    "and", AND;
    "as", AS;
    "assert", ASSERT;
    "begin", BEGIN;
    "class", CLASS;
    "constraint", CONSTRAINT;
    "do", DO;
    "done", DONE;
    "downto", DOWNTO;
    "else", ELSE;
    "end", END;
    "exception", EXCEPTION;
    "external", EXTERNAL;
    "false", FALSE;
    "for", FOR;
    "fun", FUN;
    "esfun", ES6_FUN;
    "function", FUNCTION;
    "functor", FUNCTOR;
    "if", IF;
    "in", IN;
    "include", INCLUDE;
    "inherit", INHERIT;
    "initializer", INITIALIZER;
    "lazy", LAZY;
    "let", LET;
    "switch", SWITCH;
    "module", MODULE;
    "pub", PUB;
    "mutable", MUTABLE;
    "new", NEW;
    "nonrec", NONREC;
    "object", OBJECT;
    "of", OF;
    "open", OPEN;
    "or", OR;
(*  "parser", PARSER; *)
    "pri", PRI;
    "rec", REC;
    "sig", SIG;
    "struct", STRUCT;
    "then", THEN;
    "to", TO;
    "true", TRUE;
    "try", TRY;
    "type", TYPE;
    "val", VAL;
    "virtual", VIRTUAL;
    "when", WHEN;
    "while", WHILE;
    "with", WITH;

    "mod", INFIXOP3("mod");
    "land", INFIXOP3("land");
    "lor", INFIXOP3("lor");
    "lxor", INFIXOP3("lxor");
    "lsl", INFIXOP4("lsl");
    "lsr", INFIXOP4("lsr");
    "asr", INFIXOP4("asr")
]

(* To buffer string literals *)

let string_buffer = Buffer.create 256
let raw_buffer = Buffer.create 256

let reset_string_buffer () =
  Buffer.reset string_buffer;
  Buffer.reset raw_buffer

let store_string_char c =
  Buffer.add_char string_buffer c

let store_string s =
  Buffer.add_string string_buffer s

let store_lexeme buffer lexbuf =
  Buffer.add_string buffer (Lexing.lexeme lexbuf)

let get_stored_string () =
  let str = Buffer.contents string_buffer in
  let raw =
    if Buffer.length raw_buffer = 0
    then None
    else Some (Buffer.contents raw_buffer)
  in
  Buffer.reset string_buffer;
  Buffer.reset raw_buffer;
  (str, raw)

(* To store the position of the beginning of a string and comment *)
let string_start_loc = ref Location.none;;
let comment_start_loc = ref [];;
let in_comment () = !comment_start_loc <> [];;
let is_in_string = ref false
let in_string () = !is_in_string
let print_warnings = ref true

(* To "unlex" a few characters *)
let set_lexeme_length buf n = (
  let open Lexing in
  if n < 0 then
    invalid_arg "set_lexeme_length: offset should be positive";
  if n > buf.lex_curr_pos - buf.lex_start_pos then
    invalid_arg "set_lexeme_length: offset larger than lexeme";
  buf.lex_curr_pos <- buf.lex_start_pos + n;
  buf.lex_curr_p <- {buf.lex_start_p
                     with pos_cnum = buf.lex_abs_pos + buf.lex_curr_pos};
)

(* This cut comment characters of the current buffer.
 * Operators (including "/*" and "*/") are lexed with the same rule, and this
 * function cuts the lexeme at the beginning of an operator. *)
let lexeme_without_comment buf = (
  let lexeme = Lexing.lexeme buf in
  let i = ref 0 and len = String.length lexeme - 1 in
  let found = ref (-1) in
  while !i < len && !found = -1 do
    begin match lexeme.[!i], lexeme.[!i+1] with
      | ('/', '*') | ('*', '/') ->
        found := !i;
      | _ -> ()
    end;
    incr i
  done;
  match !found with
  | -1 -> lexeme
  | n ->
      set_lexeme_length buf n;
      String.sub lexeme 0 n
)

(* Operators that could conflict with comments (those containing /*, */ and //)
 * are escaped in the source. The lexer removes the escapes so that the
 * identifier looks like OCaml ones.
 * An escape in first position is kept to distinguish "verbatim" operators
 * (\=== for instance). *)
let unescape_operator str =
  if (str <> "" && String.contains_from str 1 '\\') then (
    let b = Buffer.create (String.length str) in
    Buffer.add_char b str.[0];
    for i = 1 to String.length str - 1 do
      let c = str.[i] in
      if c <> '\\' then Buffer.add_char b c
    done;
    Buffer.contents b
  ) else str

let lexeme_operator lexbuf =
  unescape_operator (lexeme_without_comment lexbuf)

(* To translate escape sequences *)

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let char_for_decimal_code lexbuf i =
  let c = 100 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           10 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
                (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  if (c < 0 || c > 255) then
    if in_comment ()
    then 'x'
    else raise (Error(Illegal_escape (Lexing.lexeme lexbuf),
                      Location.curr lexbuf))
  else Char.chr c

let char_for_hexadecimal_code lexbuf i =
  let d1 = Char.code (Lexing.lexeme_char lexbuf i) in
  let val1 = if d1 >= 97 then d1 - 87
             else if d1 >= 65 then d1 - 55
             else d1 - 48
  in
  let d2 = Char.code (Lexing.lexeme_char lexbuf (i+1)) in
  let val2 = if d2 >= 97 then d2 - 87
             else if d2 >= 65 then d2 - 55
             else d2 - 48
  in
  Char.chr (val1 * 16 + val2)

(* To convert integer literals, allowing max_int + 1 (PR#4210) *)

let cvt_int_literal s =
  - int_of_string ("-" ^ s)
let cvt_int32_literal s =
  Int32.neg (Int32.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_int64_literal s =
  Int64.neg (Int64.of_string ("-" ^ String.sub s 0 (String.length s - 1)))
let cvt_nativeint_literal s =
  Nativeint.neg (Nativeint.of_string ("-" ^ String.sub s 0
                                                       (String.length s - 1)))

(* Remove underscores from float literals *)

let remove_underscores s =
  let l = String.length s in
  let b = Bytes.create l in
  let rec remove src dst =
    if src >= l then
      if dst >= l then s else Bytes.sub_string b 0 dst
    else
      match s.[src] with
        '_' -> remove (src + 1) dst
      |  c  -> Bytes.set b dst c; remove (src + 1) (dst + 1)
  in remove 0 0

(* Update the current location with file name and line number. *)

let update_loc lexbuf file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = match file with
                 | None -> pos.pos_fname
                 | Some s -> s
  in
  lexbuf.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  }
;;

let preprocessor = ref None

(* Error report *)

open Format

let report_error ppf = function
  | Illegal_character c ->
      fprintf ppf "Illegal character (%s)" (Char.escaped c)
  | Illegal_escape s ->
      fprintf ppf "Illegal backslash escape in string or character (%s)" s
  | Unterminated_comment _ ->
      fprintf ppf "Comment not terminated"
  | Unterminated_string ->
      fprintf ppf "String literal not terminated"
  | Unterminated_string_in_comment (_, loc) ->
      fprintf ppf "This comment contains an unterminated string literal@.\
                   %aString literal begins here"
              Location.print_error loc
  | Keyword_as_label kwd ->
      fprintf ppf "`%s' is a keyword, it cannot be used as label name" kwd
  | Literal_overflow ty ->
      fprintf ppf "Integer literal exceeds the range of representable \
                   integers of type %s" ty
  | Invalid_literal s ->
      fprintf ppf "Invalid literal %s" s

let () =
  Location.register_error_of_exn
    (function
      | Error (err, loc) ->
          Some (Location.error_of_printer loc report_error err)
      | _ ->
          None
    )

(* To "unlex" a few characters *)
let set_lexeme_length buf n = (
  let open Lexing in
  if n < 0 then
    invalid_arg "set_lexeme_length: offset should be positive";
  if n > buf.lex_curr_pos - buf.lex_start_pos then
    invalid_arg "set_lexeme_length: offset larger than lexeme";
  buf.lex_curr_pos <- buf.lex_start_pos + n;
  buf.lex_curr_p <- {buf.lex_start_p
                     with pos_cnum = buf.lex_abs_pos + buf.lex_curr_pos};
)


# 304 "src/reason-parser/reason_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\155\255\156\255\224\000\003\001\038\001\073\001\108\001\
    \143\001\178\001\191\255\213\001\248\001\198\255\091\000\059\002\
    \068\000\071\000\084\000\151\002\220\255\186\002\222\255\223\255\
    \225\255\221\002\060\003\012\001\090\003\238\255\174\003\002\004\
    \086\004\038\005\246\005\198\006\165\007\200\007\235\007\063\008\
    \122\000\254\255\001\000\053\009\088\009\123\009\158\009\193\009\
    \228\009\255\255\005\000\007\010\093\000\042\010\077\010\094\000\
    \112\010\095\000\096\000\097\000\006\000\174\010\246\255\103\000\
    \008\011\100\011\135\011\144\004\165\011\249\011\077\012\161\012\
    \245\012\073\013\157\013\241\013\069\014\153\014\237\014\108\000\
    \065\015\149\015\233\015\061\016\145\016\118\000\096\005\237\255\
    \058\003\229\016\064\017\107\000\108\000\007\000\236\255\235\255\
    \231\255\010\002\128\000\109\000\234\255\151\002\110\000\233\255\
    \238\002\113\000\232\255\104\006\114\006\089\017\144\000\130\017\
    \165\017\228\255\011\000\012\000\008\001\024\001\016\000\228\255\
    \017\000\200\017\235\017\014\018\141\000\214\255\210\255\211\255\
    \212\255\208\255\049\018\084\018\151\018\235\018\063\019\008\007\
    \153\019\186\019\199\255\020\020\055\020\151\000\183\255\202\255\
    \203\255\129\000\187\255\181\255\193\255\090\020\157\020\190\255\
    \121\008\249\020\055\021\145\021\180\021\215\021\188\255\026\022\
    \088\022\178\022\213\022\248\022\027\023\062\023\216\000\233\000\
    \097\023\040\001\005\001\132\023\012\001\070\001\250\255\143\000\
    \205\000\030\001\252\255\046\001\253\255\222\000\251\255\124\018\
    \243\255\244\255\018\000\245\255\253\001\130\023\253\255\218\000\
    \231\000\255\255\254\255\252\255\162\023\253\023\237\000\240\000\
    \111\001\251\255\250\255\249\255\108\020\077\003\244\000\248\255\
    \087\003\252\000\247\255\233\021\004\001\246\255\112\001\217\002\
    \247\255\248\255\249\255\113\001\024\024\255\255\250\255\097\024\
    \219\004\253\255\046\001\129\001\139\001\171\005\252\255\120\024\
    \251\255\146\001\220\001\252\255\128\024\254\255\255\255\147\001\
    \148\001\253\255\157\024\009\001\020\001\047\001\069\001\035\001\
    \093\001\034\001\013\000\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\096\000\091\000\090\000\089\000\080\000\
    \075\000\097\000\255\255\063\000\060\000\255\255\050\000\048\000\
    \046\000\042\000\039\000\083\000\255\255\034\000\255\255\255\255\
    \255\255\028\000\038\000\031\000\058\000\255\255\012\000\012\000\
    \011\000\010\000\008\000\007\000\049\000\005\000\004\000\003\000\
    \002\000\255\255\100\000\100\000\097\000\097\000\093\000\255\255\
    \255\255\255\255\255\255\088\000\255\255\078\000\079\000\255\255\
    \095\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \089\000\037\000\006\000\066\000\016\000\016\000\014\000\013\000\
    \016\000\013\000\013\000\012\000\014\000\013\000\014\000\255\255\
    \015\000\015\000\012\000\012\000\014\000\255\255\059\000\255\255\
    \255\255\055\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\024\000\024\000\024\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\026\000\255\255\026\000\
    \025\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\029\000\084\000\036\000\040\000\255\255\255\255\255\255\
    \255\255\255\255\089\000\085\000\089\000\054\000\255\255\255\255\
    \086\000\255\255\255\255\089\000\087\000\073\000\255\255\255\255\
    \255\255\051\000\255\255\255\255\255\255\061\000\089\000\255\255\
    \071\000\089\000\070\000\094\000\097\000\097\000\255\255\097\000\
    \069\000\076\000\077\000\092\000\082\000\081\000\255\255\255\255\
    \091\000\255\255\255\255\098\000\255\255\255\255\255\255\005\000\
    \005\000\000\000\255\255\001\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\012\000\255\255\012\000\012\000\255\255\012\000\
    \012\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\008\000\008\000\255\255\255\255\005\000\
    \005\000\255\255\001\000\005\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\255\255\003\000\
    \255\255\255\255\255\255\002\000\255\255\255\255\001\000\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\000\000\
    \000\000\255\255\255\255\091\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\096\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \255\255\255\255\000\000\115\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\115\000\116\000\115\000\118\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\174\000\000\000\255\255\
    \255\255\255\255\000\000\255\255\000\000\255\255\000\000\184\000\
    \000\000\000\000\255\255\000\000\198\000\255\255\000\000\255\255\
    \255\255\000\000\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\255\255\255\255\255\255\000\000\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\216\000\
    \000\000\000\000\000\000\255\255\222\000\000\000\000\000\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \000\000\255\255\235\000\000\000\255\255\000\000\000\000\255\255\
    \255\255\000\000\255\255\255\255\255\255\245\000\248\000\255\255\
    \248\000\255\255\255\255\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\040\000\041\000\041\000\040\000\042\000\060\000\049\000\
    \041\000\092\000\050\000\060\000\093\000\113\000\113\000\251\000\
    \114\000\114\000\119\000\119\000\185\000\120\000\120\000\214\000\
    \040\000\008\000\029\000\026\000\006\000\003\000\025\000\027\000\
    \023\000\022\000\021\000\007\000\020\000\019\000\018\000\009\000\
    \031\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\017\000\016\000\015\000\036\000\011\000\037\000\
    \005\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\014\000\043\000\013\000\004\000\039\000\
    \024\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\028\000\012\000\010\000\038\000\129\000\
    \141\000\128\000\124\000\040\000\127\000\126\000\040\000\051\000\
    \046\000\056\000\056\000\044\000\051\000\046\000\044\000\056\000\
    \044\000\006\000\095\000\094\000\100\000\103\000\006\000\144\000\
    \106\000\143\000\040\000\142\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \101\000\101\000\109\000\125\000\147\000\146\000\182\000\109\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\145\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\177\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \002\000\171\000\163\000\171\000\171\000\171\000\171\000\163\000\
    \181\000\194\000\171\000\171\000\255\255\171\000\171\000\171\000\
    \255\255\193\000\113\000\005\000\203\000\114\000\092\000\202\000\
    \005\000\093\000\171\000\207\000\171\000\171\000\171\000\171\000\
    \171\000\117\000\113\000\210\000\168\000\114\000\168\000\168\000\
    \168\000\168\000\118\000\213\000\244\000\168\000\168\000\168\000\
    \168\000\168\000\168\000\255\255\168\000\245\000\171\000\226\000\
    \117\000\246\000\116\000\171\000\172\000\168\000\171\000\168\000\
    \168\000\168\000\168\000\168\000\249\000\250\000\000\000\005\000\
    \179\000\005\000\005\000\005\000\005\000\178\000\226\000\247\000\
    \005\000\005\000\168\000\005\000\005\000\005\000\168\000\168\000\
    \181\000\000\000\000\000\000\000\171\000\180\000\171\000\169\000\
    \005\000\168\000\005\000\005\000\005\000\005\000\005\000\247\000\
    \090\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \175\000\000\000\000\000\006\000\006\000\176\000\006\000\006\000\
    \006\000\199\000\185\000\218\000\200\000\214\000\233\000\168\000\
    \000\000\168\000\167\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\226\000\000\000\046\000\228\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\226\000\046\000\163\000\
    \228\000\046\000\165\000\046\000\218\000\238\000\238\000\233\000\
    \240\000\240\000\005\000\000\000\005\000\063\000\046\000\006\000\
    \046\000\164\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\051\000\051\000\000\000\051\000\051\000\051\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \055\000\051\000\046\000\051\000\161\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\044\000\000\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\044\000\044\000\000\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\000\000\238\000\000\000\
    \046\000\239\000\046\000\052\000\044\000\051\000\044\000\044\000\
    \155\000\044\000\044\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\199\000\
    \255\255\000\000\200\000\051\000\255\255\051\000\059\000\006\000\
    \044\000\150\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \255\255\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\201\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\044\000\255\255\
    \044\000\063\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\000\000\255\255\255\255\000\000\
    \000\000\000\000\000\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\006\000\151\000\006\000\063\000\148\000\006\000\000\000\
    \236\000\197\000\000\000\000\000\006\000\255\255\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\006\000\000\000\
    \006\000\130\000\132\000\104\000\104\000\104\000\104\000\104\000\
    \104\000\000\000\000\000\000\000\149\000\006\000\006\000\006\000\
    \006\000\131\000\006\000\006\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\000\000\063\000\
    \000\000\006\000\133\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\006\000\
    \046\000\006\000\046\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\046\000\046\000\000\000\046\000\122\000\046\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\046\000\000\000\046\000\046\000\123\000\046\000\046\000\
    \000\000\000\000\000\000\044\000\237\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\218\000\056\000\044\000\219\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\055\000\044\000\046\000\044\000\044\000\
    \044\000\044\000\044\000\221\000\000\000\255\255\006\000\000\000\
    \006\000\006\000\006\000\121\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\046\000\000\000\046\000\057\000\006\000\
    \044\000\006\000\006\000\006\000\006\000\006\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\220\000\044\000\000\000\
    \044\000\063\000\000\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\108\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\108\000\109\000\000\000\109\000\
    \109\000\109\000\109\000\000\000\000\000\000\000\109\000\109\000\
    \000\000\109\000\109\000\109\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\109\000\000\000\
    \109\000\111\000\109\000\109\000\109\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\000\000\000\000\000\000\000\000\000\000\086\000\000\000\
    \110\000\088\000\109\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\000\000\087\000\000\000\
    \109\000\088\000\109\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\068\000\087\000\000\000\
    \000\000\217\000\000\000\000\000\070\000\000\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \068\000\068\000\068\000\069\000\068\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\000\000\000\000\000\000\000\000\030\000\000\000\068\000\
    \068\000\068\000\068\000\069\000\068\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\068\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \070\000\000\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\068\000\072\000\068\000\068\000\069\000\
    \068\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\073\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\074\000\071\000\071\000\000\000\000\000\000\000\
    \000\000\030\000\000\000\068\000\072\000\068\000\068\000\069\000\
    \068\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\073\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\074\000\071\000\071\000\032\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\000\000\000\000\000\000\000\000\032\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\000\000\000\000\000\000\067\000\
    \000\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\229\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\033\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\000\000\000\000\000\000\033\000\000\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\000\000\000\000\000\000\000\000\089\000\
    \000\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\230\000\230\000\230\000\230\000\230\000\
    \230\000\230\000\230\000\230\000\230\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\034\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\000\000\000\000\000\000\000\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\117\000\113\000\000\000\000\000\114\000\000\000\000\000\
    \000\000\000\000\000\000\108\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \117\000\000\000\116\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\108\000\000\000\000\000\000\000\000\000\000\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\035\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\000\000\035\000\000\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\000\000\000\000\000\000\000\000\133\000\
    \255\255\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\000\000\000\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\000\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\006\000\
    \000\000\064\000\006\000\065\000\066\000\006\000\000\000\000\000\
    \000\000\051\000\000\000\051\000\051\000\051\000\051\000\000\000\
    \000\000\000\000\051\000\051\000\000\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\051\000\006\000\051\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\051\000\000\000\051\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\051\000\051\000\000\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\052\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\051\000\000\000\051\000\052\000\
    \000\000\051\000\061\000\000\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\035\000\051\000\
    \000\000\051\000\000\000\000\000\000\000\000\000\000\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\000\000\000\000\000\000\000\000\
    \152\000\000\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\049\000\
    \000\000\000\000\050\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\048\000\000\000\
    \000\000\006\000\000\000\006\000\000\000\000\000\000\000\045\000\
    \046\000\000\000\046\000\000\000\044\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\006\000\006\000\047\000\005\000\000\000\000\000\
    \000\000\044\000\000\000\044\000\044\000\044\000\044\000\000\000\
    \000\000\000\000\044\000\044\000\000\000\044\000\044\000\044\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\044\000\004\000\044\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\044\000\000\000\044\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\056\000\044\000\000\000\
    \044\000\044\000\044\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\047\000\059\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \046\000\046\000\000\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\044\000\000\000\044\000\057\000\
    \046\000\044\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \000\000\000\000\051\000\000\000\051\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\051\000\051\000\000\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\000\000\000\000\000\000\044\000\
    \000\000\044\000\055\000\051\000\046\000\051\000\051\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\051\000\000\000\051\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\051\000\051\000\
    \000\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\046\000\000\000\046\000\052\000\051\000\051\000\
    \051\000\053\000\051\000\051\000\051\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\051\000\051\000\000\000\051\000\051\000\051\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\051\000\000\000\051\000\
    \052\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \000\000\000\000\000\000\051\000\000\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\051\000\051\000\000\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\000\000\051\000\052\000\051\000\051\000\051\000\054\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\051\000\
    \051\000\000\000\051\000\051\000\051\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\051\000\000\000\051\000\052\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\056\000\000\000\056\000\056\000\056\000\056\000\000\000\
    \000\000\000\000\056\000\056\000\000\000\056\000\056\000\056\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\052\000\056\000\051\000\056\000\056\000\056\000\056\000\
    \056\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\051\000\000\000\051\000\058\000\000\000\056\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \000\000\000\000\000\000\062\000\056\000\000\000\056\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\000\000\000\000\000\000\000\000\061\000\000\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \006\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\063\000\000\000\006\000\067\000\
    \000\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\000\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \063\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\000\000\
    \006\000\000\000\006\000\063\000\000\000\006\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\006\000\068\000\006\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\085\000\000\000\085\000\000\000\
    \000\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\069\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \000\000\000\000\000\000\000\000\070\000\000\000\068\000\068\000\
    \068\000\068\000\069\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\083\000\083\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\076\000\
    \000\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\077\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\000\000\000\000\000\000\000\000\
    \075\000\000\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\077\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\081\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \000\000\000\000\000\000\000\000\076\000\000\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\081\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \068\000\000\000\000\000\000\000\079\000\000\000\079\000\000\000\
    \000\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \000\000\000\000\000\000\000\000\078\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\000\000\000\000\000\000\000\000\
    \068\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\000\000\000\000\000\000\
    \079\000\000\000\079\000\000\000\000\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \000\000\000\000\000\000\000\000\068\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\000\000\000\000\000\000\000\000\
    \082\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\083\000\083\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \000\000\000\000\000\000\000\000\083\000\000\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\000\000\000\000\000\000\000\000\
    \084\000\000\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\089\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\089\000\000\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \000\000\000\000\000\000\000\000\089\000\000\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \099\000\000\000\099\000\000\000\000\000\000\000\000\000\099\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\098\000\109\000\000\000\109\000\109\000\109\000\109\000\
    \000\000\000\000\000\000\109\000\109\000\000\000\109\000\109\000\
    \109\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\109\000\000\000\109\000\109\000\109\000\
    \109\000\109\000\000\000\000\000\099\000\000\000\000\000\000\000\
    \000\000\000\000\099\000\109\000\000\000\109\000\109\000\109\000\
    \109\000\000\000\000\000\000\000\109\000\109\000\099\000\109\000\
    \109\000\109\000\099\000\000\000\099\000\110\000\000\000\109\000\
    \097\000\000\000\000\000\000\000\109\000\000\000\112\000\109\000\
    \109\000\109\000\109\000\000\000\000\000\000\000\109\000\000\000\
    \109\000\109\000\109\000\109\000\000\000\000\000\000\000\109\000\
    \109\000\000\000\109\000\109\000\109\000\109\000\000\000\109\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\110\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\109\000\000\000\000\000\
    \000\000\006\000\000\000\006\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\006\000\006\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\109\000\000\000\
    \109\000\110\000\006\000\109\000\006\000\006\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\046\000\000\000\046\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\109\000\000\000\109\000\063\000\046\000\006\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \046\000\046\000\000\000\046\000\046\000\046\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\006\000\000\000\006\000\055\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \000\000\000\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\006\000\000\000\006\000\139\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\055\000\006\000\046\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\185\000\000\000\
    \000\000\186\000\046\000\000\000\046\000\063\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\190\000\000\000\
    \000\000\000\000\000\000\188\000\000\000\000\000\191\000\000\000\
    \000\000\000\000\000\000\192\000\000\000\006\000\000\000\006\000\
    \063\000\000\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\136\000\006\000\006\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\000\000\063\000\000\000\006\000\137\000\189\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\133\000\006\000\000\000\006\000\000\000\000\000\
    \000\000\000\000\000\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\000\000\000\000\
    \000\000\000\000\133\000\000\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\134\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\135\000\000\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\000\000\000\000\000\000\187\000\000\000\000\000\000\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\000\000\000\000\000\000\000\000\134\000\000\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\006\000\000\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\006\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\000\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\137\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \137\000\000\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\000\000\063\000\000\000\006\000\
    \138\000\000\000\000\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\006\000\000\000\006\000\
    \000\000\137\000\000\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\006\000\000\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\006\000\140\000\006\000\006\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \063\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\000\000\006\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\006\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\063\000\006\000\006\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\211\000\211\000\211\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\006\000\000\000\006\000\063\000\000\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\006\000\
    \006\000\000\000\006\000\006\000\153\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\000\000\000\000\000\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\006\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \000\000\063\000\000\000\006\000\152\000\000\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\006\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\000\000\006\000\006\000\006\000\
    \006\000\006\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\000\000\063\000\000\000\006\000\
    \154\000\000\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\000\000\006\000\000\000\006\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\000\000\000\000\000\000\000\000\154\000\000\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\044\000\000\000\044\000\044\000\044\000\044\000\
    \000\000\000\000\000\000\044\000\044\000\000\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\044\000\000\000\156\000\044\000\044\000\
    \044\000\044\000\000\000\000\000\000\000\044\000\000\000\044\000\
    \044\000\044\000\044\000\000\000\000\000\000\000\044\000\044\000\
    \000\000\044\000\044\000\159\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\059\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\044\000\000\000\000\000\000\000\
    \044\000\000\000\044\000\044\000\044\000\044\000\000\000\000\000\
    \000\000\044\000\044\000\000\000\044\000\044\000\044\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\157\000\000\000\044\000\
    \059\000\044\000\044\000\044\000\044\000\044\000\044\000\044\000\
    \000\000\212\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \212\000\212\000\212\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \044\000\000\000\044\000\059\000\158\000\044\000\000\000\000\000\
    \000\000\000\000\000\000\044\000\000\000\044\000\044\000\044\000\
    \044\000\000\000\000\000\000\000\044\000\044\000\000\000\044\000\
    \044\000\044\000\212\000\212\000\212\000\212\000\212\000\212\000\
    \000\000\000\000\000\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\000\000\059\000\000\000\
    \044\000\160\000\000\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\000\000\044\000\000\000\
    \044\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\000\000\000\000\000\000\000\000\160\000\
    \000\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\051\000\000\000\051\000\051\000\051\000\
    \051\000\000\000\000\000\000\000\051\000\051\000\000\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\051\000\000\000\051\000\162\000\
    \051\000\051\000\051\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\051\000\
    \051\000\000\000\051\000\051\000\051\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\052\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\000\000\
    \000\000\163\000\000\000\163\000\163\000\163\000\163\000\000\000\
    \000\000\000\000\163\000\163\000\000\000\163\000\163\000\163\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\052\000\163\000\051\000\163\000\163\000\163\000\163\000\
    \163\000\000\000\000\000\000\000\046\000\000\000\046\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\051\000\000\000\051\000\166\000\046\000\163\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \046\000\046\000\000\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\163\000\000\000\163\000\055\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \000\000\000\000\168\000\000\000\168\000\168\000\168\000\168\000\
    \000\000\000\000\000\000\168\000\168\000\000\000\168\000\168\000\
    \168\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\055\000\168\000\046\000\168\000\168\000\168\000\
    \168\000\168\000\000\000\000\000\000\000\171\000\000\000\171\000\
    \171\000\171\000\171\000\000\000\000\000\000\000\171\000\171\000\
    \000\000\171\000\171\000\171\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\046\000\000\000\046\000\170\000\171\000\168\000\
    \171\000\171\000\171\000\171\000\171\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\168\000\000\000\168\000\
    \172\000\196\000\171\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\000\000\195\000\000\000\
    \171\000\196\000\171\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\206\000\195\000\206\000\
    \000\000\000\000\226\000\000\000\206\000\227\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\000\000\
    \225\000\000\000\225\000\000\000\000\000\000\000\000\000\225\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\206\000\000\000\000\000\000\000\000\000\000\000\206\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\206\000\000\000\000\000\000\000\206\000\
    \000\000\206\000\000\000\000\000\225\000\204\000\000\000\000\000\
    \000\000\000\000\225\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\225\000\000\000\
    \000\000\000\000\225\000\000\000\225\000\000\000\000\000\000\000\
    \223\000\231\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\231\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \232\000\232\000\232\000\232\000\232\000\232\000\232\000\232\000\
    \232\000\232\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\232\000\232\000\232\000\232\000\232\000\232\000\000\000\
    \000\000\000\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\232\000\232\000\232\000\232\000\232\000\232\000\242\000\
    \000\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\000\000\242\000\241\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \255\255\000\000\241\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\042\000\000\000\000\000\042\000\050\000\
    \060\000\093\000\050\000\060\000\093\000\114\000\115\000\250\000\
    \114\000\115\000\118\000\120\000\186\000\118\000\120\000\186\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\016\000\
    \014\000\017\000\018\000\040\000\017\000\017\000\040\000\052\000\
    \055\000\057\000\058\000\059\000\052\000\055\000\057\000\058\000\
    \059\000\063\000\091\000\092\000\099\000\102\000\063\000\014\000\
    \105\000\014\000\040\000\014\000\079\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \098\000\098\000\098\000\098\000\098\000\098\000\098\000\098\000\
    \098\000\098\000\110\000\124\000\141\000\145\000\175\000\110\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\014\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\176\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\166\000\003\000\003\000\003\000\003\000\166\000\
    \181\000\191\000\003\000\003\000\115\000\003\000\003\000\003\000\
    \118\000\192\000\116\000\167\000\198\000\116\000\027\000\199\000\
    \167\000\027\000\003\000\206\000\003\000\003\000\003\000\003\000\
    \003\000\117\000\117\000\209\000\004\000\117\000\004\000\004\000\
    \004\000\004\000\116\000\212\000\243\000\004\000\004\000\170\000\
    \004\000\004\000\004\000\027\000\170\000\244\000\172\000\226\000\
    \117\000\245\000\117\000\172\000\003\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\247\000\249\000\255\255\005\000\
    \177\000\005\000\005\000\005\000\005\000\177\000\226\000\246\000\
    \005\000\005\000\169\000\005\000\005\000\005\000\169\000\169\000\
    \179\000\255\255\255\255\255\255\003\000\179\000\003\000\004\000\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\248\000\
    \027\000\255\255\006\000\255\255\006\000\006\000\006\000\006\000\
    \173\000\255\255\255\255\006\000\006\000\173\000\006\000\006\000\
    \006\000\200\000\214\000\219\000\200\000\214\000\219\000\004\000\
    \255\255\004\000\005\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\227\000\255\255\007\000\227\000\007\000\
    \007\000\007\000\007\000\255\255\255\255\228\000\007\000\007\000\
    \228\000\007\000\007\000\007\000\233\000\239\000\240\000\233\000\
    \239\000\240\000\005\000\255\255\005\000\006\000\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \008\000\255\255\008\000\008\000\008\000\008\000\255\255\255\255\
    \255\255\008\000\008\000\255\255\008\000\008\000\008\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\006\000\
    \007\000\008\000\007\000\008\000\008\000\008\000\008\000\008\000\
    \255\255\255\255\255\255\009\000\255\255\009\000\009\000\009\000\
    \009\000\255\255\255\255\255\255\009\000\009\000\255\255\009\000\
    \009\000\009\000\255\255\255\255\255\255\255\255\234\000\255\255\
    \007\000\234\000\007\000\008\000\009\000\008\000\009\000\009\000\
    \009\000\009\000\009\000\255\255\255\255\255\255\011\000\255\255\
    \011\000\011\000\011\000\011\000\255\255\255\255\255\255\011\000\
    \011\000\255\255\011\000\011\000\011\000\255\255\255\255\188\000\
    \116\000\255\255\188\000\008\000\027\000\008\000\009\000\011\000\
    \009\000\011\000\011\000\011\000\011\000\011\000\255\255\255\255\
    \117\000\012\000\255\255\012\000\012\000\012\000\012\000\255\255\
    \255\255\255\255\012\000\012\000\188\000\012\000\012\000\012\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\009\000\245\000\
    \009\000\011\000\012\000\011\000\012\000\012\000\012\000\012\000\
    \012\000\255\255\097\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\255\255\246\000\173\000\255\255\
    \255\255\255\255\255\255\097\000\097\000\097\000\097\000\097\000\
    \097\000\011\000\011\000\011\000\012\000\012\000\012\000\255\255\
    \234\000\188\000\255\255\255\255\015\000\248\000\015\000\015\000\
    \015\000\015\000\255\255\255\255\255\255\015\000\015\000\255\255\
    \015\000\015\000\015\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\255\255\255\255\255\255\012\000\015\000\012\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\255\255\015\000\
    \255\255\015\000\015\000\255\255\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\255\255\015\000\
    \019\000\015\000\019\000\019\000\019\000\019\000\255\255\255\255\
    \255\255\019\000\019\000\255\255\019\000\019\000\019\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \101\000\019\000\255\255\019\000\019\000\019\000\019\000\019\000\
    \255\255\255\255\255\255\021\000\234\000\021\000\021\000\021\000\
    \021\000\255\255\255\255\215\000\021\000\021\000\215\000\021\000\
    \021\000\021\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\019\000\021\000\019\000\021\000\021\000\
    \021\000\021\000\021\000\215\000\255\255\188\000\025\000\255\255\
    \025\000\025\000\025\000\025\000\255\255\255\255\255\255\025\000\
    \025\000\255\255\025\000\025\000\025\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\019\000\255\255\019\000\021\000\025\000\
    \021\000\025\000\025\000\025\000\025\000\025\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\104\000\
    \104\000\104\000\104\000\104\000\104\000\215\000\021\000\255\255\
    \021\000\025\000\255\255\025\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\026\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\104\000\
    \104\000\104\000\104\000\104\000\104\000\255\255\255\255\255\255\
    \255\255\025\000\255\255\025\000\026\000\026\000\255\255\026\000\
    \026\000\026\000\026\000\255\255\255\255\255\255\026\000\026\000\
    \255\255\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \026\000\026\000\026\000\026\000\026\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\208\000\
    \208\000\255\255\255\255\255\255\255\255\255\255\028\000\255\255\
    \026\000\088\000\026\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\255\255\088\000\255\255\
    \026\000\028\000\026\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\030\000\028\000\255\255\
    \255\255\215\000\255\255\255\255\030\000\255\255\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\255\255\255\255\255\255\255\255\030\000\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\031\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\255\255\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\255\255\255\255\
    \255\255\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\032\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\255\255\255\255\255\255\032\000\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\255\255\255\255\255\255\255\255\067\000\
    \255\255\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\033\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\255\255\255\255\255\255\255\255\033\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\255\255\255\255\255\255\255\255\086\000\
    \255\255\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\229\000\229\000\229\000\229\000\229\000\
    \229\000\229\000\229\000\229\000\229\000\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\034\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\255\255\255\255\255\255\255\255\034\000\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\107\000\107\000\255\255\255\255\107\000\255\255\255\255\
    \255\255\255\255\255\255\108\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \107\000\255\255\107\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\108\000\255\255\255\255\255\255\255\255\255\255\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\108\000\108\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\035\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\255\255\255\255\255\255\255\255\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\255\255\255\255\255\255\255\255\135\000\
    \107\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\135\000\135\000\135\000\135\000\135\000\
    \135\000\135\000\135\000\255\255\255\255\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\036\000\255\255\
    \036\000\036\000\036\000\036\000\255\255\255\255\255\255\036\000\
    \036\000\255\255\036\000\036\000\036\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\036\000\
    \255\255\036\000\036\000\036\000\036\000\036\000\255\255\255\255\
    \255\255\037\000\255\255\037\000\037\000\037\000\037\000\255\255\
    \255\255\255\255\037\000\037\000\255\255\037\000\037\000\037\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\036\000\037\000\036\000\037\000\037\000\037\000\037\000\
    \037\000\255\255\255\255\255\255\038\000\255\255\038\000\038\000\
    \038\000\038\000\255\255\255\255\255\255\038\000\038\000\255\255\
    \038\000\038\000\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\036\000\255\255\036\000\037\000\038\000\037\000\038\000\
    \038\000\038\000\038\000\038\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\255\255\037\000\038\000\
    \255\255\038\000\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\039\000\038\000\
    \255\255\038\000\255\255\255\255\255\255\255\255\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\255\255\255\255\255\255\255\255\039\000\255\255\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\255\255\255\255\255\255\255\255\
    \152\000\255\255\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\255\255\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\043\000\
    \255\255\255\255\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\043\000\255\255\
    \255\255\043\000\255\255\043\000\255\255\255\255\255\255\043\000\
    \043\000\255\255\043\000\255\255\043\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\043\000\043\000\043\000\043\000\255\255\255\255\
    \255\255\044\000\255\255\044\000\044\000\044\000\044\000\255\255\
    \255\255\255\255\044\000\044\000\255\255\044\000\044\000\044\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\043\000\044\000\044\000\044\000\044\000\
    \044\000\255\255\255\255\255\255\045\000\255\255\045\000\045\000\
    \045\000\045\000\255\255\255\255\255\255\045\000\045\000\255\255\
    \045\000\045\000\045\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\255\255\043\000\044\000\045\000\044\000\045\000\
    \045\000\045\000\045\000\045\000\255\255\255\255\255\255\046\000\
    \255\255\046\000\046\000\046\000\046\000\255\255\255\255\255\255\
    \046\000\046\000\255\255\046\000\046\000\046\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\044\000\045\000\
    \046\000\045\000\046\000\046\000\046\000\046\000\046\000\255\255\
    \255\255\255\255\047\000\255\255\047\000\047\000\047\000\047\000\
    \255\255\255\255\255\255\047\000\047\000\255\255\047\000\047\000\
    \047\000\255\255\255\255\255\255\255\255\255\255\255\255\045\000\
    \255\255\045\000\046\000\047\000\046\000\047\000\047\000\047\000\
    \047\000\047\000\255\255\255\255\255\255\048\000\255\255\048\000\
    \048\000\048\000\048\000\255\255\255\255\255\255\048\000\048\000\
    \255\255\048\000\048\000\048\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\046\000\255\255\046\000\047\000\048\000\047\000\
    \048\000\048\000\048\000\048\000\048\000\255\255\255\255\255\255\
    \051\000\255\255\051\000\051\000\051\000\051\000\255\255\255\255\
    \255\255\051\000\051\000\255\255\051\000\051\000\051\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\047\000\255\255\047\000\
    \048\000\051\000\048\000\051\000\051\000\051\000\051\000\051\000\
    \255\255\255\255\255\255\053\000\255\255\053\000\053\000\053\000\
    \053\000\255\255\255\255\255\255\053\000\053\000\255\255\053\000\
    \053\000\053\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \048\000\255\255\048\000\051\000\053\000\051\000\053\000\053\000\
    \053\000\053\000\053\000\255\255\255\255\255\255\054\000\255\255\
    \054\000\054\000\054\000\054\000\255\255\255\255\255\255\054\000\
    \054\000\255\255\054\000\054\000\054\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\051\000\255\255\051\000\053\000\054\000\
    \053\000\054\000\054\000\054\000\054\000\054\000\255\255\255\255\
    \255\255\056\000\255\255\056\000\056\000\056\000\056\000\255\255\
    \255\255\255\255\056\000\056\000\255\255\056\000\056\000\056\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\053\000\255\255\
    \053\000\054\000\056\000\054\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\054\000\255\255\054\000\056\000\255\255\056\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\061\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \255\255\255\255\255\255\061\000\056\000\255\255\056\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\255\255\255\255\255\255\255\255\061\000\255\255\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\064\000\255\255\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\064\000\064\000\255\255\064\000\064\000\064\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\064\000\255\255\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\255\255\064\000\255\255\064\000\064\000\
    \255\255\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\255\255\064\000\065\000\064\000\065\000\
    \065\000\065\000\065\000\255\255\255\255\255\255\065\000\065\000\
    \255\255\065\000\065\000\065\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\065\000\255\255\
    \065\000\065\000\065\000\065\000\065\000\255\255\255\255\255\255\
    \066\000\255\255\066\000\066\000\066\000\066\000\255\255\255\255\
    \255\255\066\000\066\000\255\255\066\000\066\000\066\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \065\000\066\000\065\000\066\000\066\000\066\000\066\000\066\000\
    \255\255\255\255\255\255\255\255\068\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\255\255\
    \065\000\255\255\065\000\066\000\255\255\066\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \255\255\255\255\255\255\066\000\068\000\066\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \069\000\255\255\255\255\255\255\069\000\255\255\069\000\255\255\
    \255\255\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\255\255\255\255\255\255\255\255\
    \069\000\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\070\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \255\255\255\255\255\255\255\255\070\000\255\255\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \071\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\255\255\255\255\255\255\255\255\
    \071\000\255\255\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\072\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \255\255\255\255\255\255\255\255\072\000\255\255\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \073\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\255\255\255\255\255\255\255\255\
    \073\000\255\255\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\074\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \255\255\255\255\255\255\255\255\074\000\255\255\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \075\000\255\255\255\255\255\255\255\255\255\255\255\255\075\000\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\255\255\255\255\255\255\255\255\
    \075\000\255\255\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\076\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \255\255\255\255\255\255\255\255\076\000\255\255\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \077\000\255\255\255\255\255\255\077\000\255\255\077\000\255\255\
    \255\255\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\255\255\255\255\255\255\255\255\
    \077\000\255\255\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\078\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \255\255\255\255\255\255\255\255\078\000\255\255\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \080\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\255\255\255\255\255\255\255\255\
    \080\000\255\255\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\080\000\080\000\080\000\080\000\
    \080\000\080\000\080\000\080\000\081\000\255\255\255\255\255\255\
    \081\000\255\255\081\000\255\255\255\255\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \255\255\255\255\255\255\255\255\081\000\255\255\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \082\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\255\255\255\255\255\255\255\255\
    \082\000\255\255\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\083\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \255\255\255\255\255\255\255\255\083\000\255\255\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \084\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\255\255\255\255\255\255\255\255\
    \084\000\255\255\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\089\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\089\000\255\255\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \255\255\255\255\255\255\255\255\089\000\255\255\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \090\000\255\255\090\000\255\255\255\255\255\255\255\255\090\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\109\000\255\255\109\000\109\000\109\000\109\000\
    \255\255\255\255\255\255\109\000\109\000\255\255\109\000\109\000\
    \109\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\109\000\255\255\109\000\109\000\109\000\
    \109\000\109\000\255\255\255\255\090\000\255\255\255\255\255\255\
    \255\255\255\255\090\000\111\000\255\255\111\000\111\000\111\000\
    \111\000\255\255\255\255\255\255\111\000\111\000\090\000\111\000\
    \111\000\111\000\090\000\255\255\090\000\109\000\255\255\109\000\
    \090\000\255\255\255\255\255\255\111\000\255\255\111\000\111\000\
    \111\000\111\000\111\000\255\255\255\255\255\255\112\000\255\255\
    \112\000\112\000\112\000\112\000\255\255\255\255\255\255\112\000\
    \112\000\255\255\112\000\112\000\112\000\109\000\255\255\109\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\111\000\112\000\
    \111\000\112\000\112\000\112\000\112\000\112\000\255\255\255\255\
    \255\255\121\000\255\255\121\000\121\000\121\000\121\000\255\255\
    \255\255\255\255\121\000\121\000\255\255\121\000\121\000\121\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\111\000\255\255\
    \111\000\112\000\121\000\112\000\121\000\121\000\121\000\121\000\
    \121\000\255\255\255\255\255\255\122\000\255\255\122\000\122\000\
    \122\000\122\000\255\255\255\255\255\255\122\000\122\000\255\255\
    \122\000\122\000\122\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\112\000\255\255\112\000\121\000\122\000\121\000\122\000\
    \122\000\122\000\122\000\122\000\255\255\255\255\255\255\123\000\
    \255\255\123\000\123\000\123\000\123\000\255\255\255\255\255\255\
    \123\000\123\000\255\255\123\000\123\000\123\000\255\255\255\255\
    \090\000\255\255\255\255\255\255\121\000\255\255\121\000\122\000\
    \123\000\122\000\123\000\123\000\123\000\123\000\123\000\255\255\
    \255\255\255\255\130\000\255\255\130\000\130\000\130\000\130\000\
    \255\255\255\255\255\255\130\000\130\000\255\255\130\000\130\000\
    \130\000\255\255\255\255\255\255\255\255\255\255\255\255\122\000\
    \255\255\122\000\123\000\130\000\123\000\130\000\130\000\130\000\
    \130\000\130\000\255\255\255\255\255\255\131\000\255\255\131\000\
    \131\000\131\000\131\000\255\255\255\255\255\255\131\000\131\000\
    \255\255\131\000\131\000\131\000\255\255\255\255\183\000\255\255\
    \255\255\183\000\123\000\255\255\123\000\130\000\131\000\130\000\
    \131\000\131\000\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\183\000\255\255\
    \255\255\255\255\255\255\183\000\255\255\255\255\183\000\255\255\
    \255\255\255\255\255\255\183\000\255\255\130\000\255\255\130\000\
    \131\000\255\255\131\000\255\255\255\255\255\255\255\255\255\255\
    \132\000\255\255\132\000\132\000\132\000\132\000\255\255\255\255\
    \255\255\132\000\132\000\255\255\132\000\132\000\132\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \131\000\132\000\131\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\255\255\132\000\255\255\132\000\132\000\183\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\133\000\132\000\255\255\132\000\255\255\255\255\
    \255\255\255\255\255\255\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\255\255\255\255\
    \255\255\255\255\133\000\255\255\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\133\000\133\000\134\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\134\000\255\255\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\255\255\255\255\255\255\183\000\255\255\255\255\255\255\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\255\255\255\255\255\255\255\255\134\000\255\255\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\136\000\255\255\136\000\136\000\136\000\136\000\
    \255\255\255\255\255\255\136\000\136\000\255\255\136\000\136\000\
    \136\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\136\000\255\255\136\000\136\000\136\000\
    \136\000\136\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\137\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \137\000\255\255\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\255\255\136\000\255\255\136\000\
    \137\000\255\255\255\255\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\136\000\255\255\136\000\
    \255\255\137\000\255\255\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\139\000\255\255\139\000\
    \139\000\139\000\139\000\255\255\255\255\255\255\139\000\139\000\
    \255\255\139\000\139\000\139\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\139\000\255\255\
    \139\000\139\000\139\000\139\000\139\000\255\255\255\255\255\255\
    \140\000\255\255\140\000\140\000\140\000\140\000\255\255\255\255\
    \255\255\140\000\140\000\255\255\140\000\140\000\140\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \139\000\140\000\139\000\140\000\140\000\140\000\140\000\140\000\
    \255\255\255\255\255\255\149\000\255\255\149\000\149\000\149\000\
    \149\000\255\255\255\255\255\255\149\000\149\000\255\255\149\000\
    \149\000\149\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \139\000\255\255\139\000\140\000\149\000\140\000\149\000\149\000\
    \149\000\149\000\149\000\255\255\204\000\204\000\204\000\204\000\
    \204\000\204\000\204\000\204\000\204\000\204\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\204\000\204\000\204\000\
    \204\000\204\000\204\000\140\000\255\255\140\000\149\000\255\255\
    \149\000\255\255\255\255\255\255\255\255\255\255\150\000\255\255\
    \150\000\150\000\150\000\150\000\255\255\255\255\255\255\150\000\
    \150\000\255\255\150\000\150\000\150\000\204\000\204\000\204\000\
    \204\000\204\000\204\000\255\255\255\255\255\255\149\000\150\000\
    \149\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \255\255\150\000\255\255\150\000\150\000\255\255\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \150\000\150\000\150\000\150\000\150\000\150\000\150\000\150\000\
    \255\255\150\000\153\000\150\000\153\000\153\000\153\000\153\000\
    \255\255\255\255\255\255\153\000\153\000\255\255\153\000\153\000\
    \153\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\153\000\255\255\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\255\255\153\000\255\255\153\000\
    \153\000\255\255\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\255\255\153\000\255\255\153\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\255\255\255\255\255\255\255\255\154\000\255\255\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\155\000\255\255\155\000\155\000\155\000\155\000\
    \255\255\255\255\255\255\155\000\155\000\255\255\155\000\155\000\
    \155\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\155\000\255\255\155\000\155\000\155\000\
    \155\000\155\000\255\255\255\255\255\255\156\000\255\255\156\000\
    \156\000\156\000\156\000\255\255\255\255\255\255\156\000\156\000\
    \255\255\156\000\156\000\156\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\155\000\156\000\155\000\
    \156\000\156\000\156\000\156\000\156\000\255\255\255\255\255\255\
    \157\000\255\255\157\000\157\000\157\000\157\000\255\255\255\255\
    \255\255\157\000\157\000\255\255\157\000\157\000\157\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\155\000\255\255\155\000\
    \156\000\157\000\156\000\157\000\157\000\157\000\157\000\157\000\
    \255\255\211\000\211\000\211\000\211\000\211\000\211\000\211\000\
    \211\000\211\000\211\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\211\000\211\000\211\000\211\000\211\000\211\000\
    \156\000\255\255\156\000\157\000\157\000\157\000\255\255\255\255\
    \255\255\255\255\255\255\159\000\255\255\159\000\159\000\159\000\
    \159\000\255\255\255\255\255\255\159\000\159\000\255\255\159\000\
    \159\000\159\000\211\000\211\000\211\000\211\000\211\000\211\000\
    \255\255\255\255\255\255\157\000\159\000\157\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\255\255\159\000\255\255\
    \159\000\159\000\255\255\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\255\255\159\000\255\255\
    \159\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\255\255\255\255\255\255\255\255\160\000\
    \255\255\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\161\000\255\255\161\000\161\000\161\000\
    \161\000\255\255\255\255\255\255\161\000\161\000\255\255\161\000\
    \161\000\161\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\161\000\255\255\161\000\161\000\
    \161\000\161\000\161\000\255\255\255\255\255\255\162\000\255\255\
    \162\000\162\000\162\000\162\000\255\255\255\255\255\255\162\000\
    \162\000\255\255\162\000\162\000\162\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\161\000\162\000\
    \161\000\162\000\162\000\162\000\162\000\162\000\255\255\255\255\
    \255\255\163\000\255\255\163\000\163\000\163\000\163\000\255\255\
    \255\255\255\255\163\000\163\000\255\255\163\000\163\000\163\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\161\000\255\255\
    \161\000\162\000\163\000\162\000\163\000\163\000\163\000\163\000\
    \163\000\255\255\255\255\255\255\164\000\255\255\164\000\164\000\
    \164\000\164\000\255\255\255\255\255\255\164\000\164\000\255\255\
    \164\000\164\000\164\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\162\000\255\255\162\000\163\000\164\000\163\000\164\000\
    \164\000\164\000\164\000\164\000\255\255\255\255\255\255\165\000\
    \255\255\165\000\165\000\165\000\165\000\255\255\255\255\255\255\
    \165\000\165\000\255\255\165\000\165\000\165\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\163\000\255\255\163\000\164\000\
    \165\000\164\000\165\000\165\000\165\000\165\000\165\000\255\255\
    \255\255\255\255\168\000\255\255\168\000\168\000\168\000\168\000\
    \255\255\255\255\255\255\168\000\168\000\255\255\168\000\168\000\
    \168\000\255\255\255\255\255\255\255\255\255\255\255\255\164\000\
    \255\255\164\000\165\000\168\000\165\000\168\000\168\000\168\000\
    \168\000\168\000\255\255\255\255\255\255\171\000\255\255\171\000\
    \171\000\171\000\171\000\255\255\255\255\255\255\171\000\171\000\
    \255\255\171\000\171\000\171\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\165\000\255\255\165\000\168\000\171\000\168\000\
    \171\000\171\000\171\000\171\000\171\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\168\000\255\255\168\000\
    \171\000\189\000\171\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\255\255\189\000\255\255\
    \171\000\196\000\171\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\197\000\196\000\197\000\
    \255\255\255\255\220\000\255\255\197\000\220\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\197\000\197\000\197\000\
    \197\000\197\000\197\000\197\000\197\000\197\000\197\000\255\255\
    \220\000\255\255\220\000\255\255\255\255\255\255\255\255\220\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\197\000\255\255\255\255\255\255\255\255\255\255\197\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\197\000\255\255\255\255\255\255\197\000\
    \255\255\197\000\255\255\255\255\220\000\197\000\255\255\255\255\
    \255\255\255\255\220\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\220\000\255\255\
    \255\255\255\255\220\000\255\255\220\000\255\255\255\255\255\255\
    \220\000\223\000\223\000\223\000\223\000\223\000\223\000\223\000\
    \223\000\223\000\223\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\223\000\223\000\223\000\223\000\223\000\223\000\
    \231\000\231\000\231\000\231\000\231\000\231\000\231\000\231\000\
    \231\000\231\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\231\000\231\000\231\000\231\000\231\000\231\000\255\255\
    \255\255\255\255\223\000\223\000\223\000\223\000\223\000\223\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\231\000\231\000\231\000\231\000\231\000\231\000\236\000\
    \255\255\236\000\236\000\236\000\236\000\236\000\236\000\236\000\
    \236\000\236\000\236\000\236\000\236\000\236\000\236\000\236\000\
    \236\000\236\000\236\000\236\000\236\000\236\000\236\000\236\000\
    \236\000\236\000\236\000\255\255\242\000\236\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \220\000\255\255\242\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\010\000\036\000\000\000\000\000\000\000\
    \000\000\012\000\000\000\000\000\000\000\002\000\000\000\027\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\039\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\019\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\036\000\036\000\000\000\036\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\001\000\022\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\001\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\001\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\026\000\116\000\220\000\227\000\116\000\228\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \026\000\255\255\116\000\000\000\117\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\107\000\108\000\255\255\255\255\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\108\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \116\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\004\255\255\005\255\255\007\255\006\255\255\003\255\000\004\
    \001\005\255\007\255\255\006\255\007\255\255\000\004\001\005\003\
    \006\002\007\255\001\255\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 8 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 412 "src/reason-parser/reason_lexer.mll"
                 (
      match !preprocessor with
      | None ->
        raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      | Some _ ->
        update_loc lexbuf None 1 false 0;
        token lexbuf )
# 2242 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 421 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        match !preprocessor with
        | None -> token lexbuf
        | Some _ -> EOL
      )
# 2251 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 427 "src/reason-parser/reason_lexer.mll"
      ( token lexbuf )
# 2256 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 429 "src/reason-parser/reason_lexer.mll"
      ( UNDERSCORE )
# 2261 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 431 "src/reason-parser/reason_lexer.mll"
      ( TILDE )
# 2266 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 433 "src/reason-parser/reason_lexer.mll"
      ( QUESTION )
# 2271 "src/reason-parser/reason_lexer.ml"

  | 6 ->
# 435 "src/reason-parser/reason_lexer.mll"
      ( set_lexeme_length lexbuf 1; EQUAL )
# 2276 "src/reason-parser/reason_lexer.ml"

  | 7 ->
# 437 "src/reason-parser/reason_lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        try Hashtbl.find keyword_table s
        with Not_found -> LIDENT s )
# 2283 "src/reason-parser/reason_lexer.ml"

  | 8 ->
# 441 "src/reason-parser/reason_lexer.mll"
      ( warn_latin1 lexbuf; LIDENT (Lexing.lexeme lexbuf) )
# 2288 "src/reason-parser/reason_lexer.ml"

  | 9 ->
# 443 "src/reason-parser/reason_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        LABEL_WITH_EQUAL(String.sub l 1 (String.length l - 2)) )
# 2294 "src/reason-parser/reason_lexer.ml"

  | 10 ->
# 446 "src/reason-parser/reason_lexer.mll"
      ( UIDENT(Lexing.lexeme lexbuf) )
# 2299 "src/reason-parser/reason_lexer.ml"

  | 11 ->
# 448 "src/reason-parser/reason_lexer.mll"
      ( warn_latin1 lexbuf; UIDENT(Lexing.lexeme lexbuf) )
# 2304 "src/reason-parser/reason_lexer.ml"

  | 12 ->
# 449 "src/reason-parser/reason_lexer.mll"
                ( INT (Lexing.lexeme lexbuf, None) )
# 2309 "src/reason-parser/reason_lexer.ml"

  | 13 ->
let
# 450 "src/reason-parser/reason_lexer.mll"
                    lit
# 2315 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 450 "src/reason-parser/reason_lexer.mll"
                                              modif
# 2320 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 451 "src/reason-parser/reason_lexer.mll"
      ( INT (lit, Some modif) )
# 2324 "src/reason-parser/reason_lexer.ml"

  | 14 ->
# 453 "src/reason-parser/reason_lexer.mll"
      ( FLOAT (Lexing.lexeme lexbuf, None) )
# 2329 "src/reason-parser/reason_lexer.ml"

  | 15 ->
let
# 454 "src/reason-parser/reason_lexer.mll"
                                            lit
# 2335 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 454 "src/reason-parser/reason_lexer.mll"
                                                                      modif
# 2340 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 455 "src/reason-parser/reason_lexer.mll"
      ( FLOAT (lit, Some modif) )
# 2344 "src/reason-parser/reason_lexer.ml"

  | 16 ->
# 457 "src/reason-parser/reason_lexer.mll"
      ( raise (Error(Invalid_literal (Lexing.lexeme lexbuf),
                     Location.curr lexbuf)) )
# 2350 "src/reason-parser/reason_lexer.ml"

  | 17 ->
# 460 "src/reason-parser/reason_lexer.mll"
      ( reset_string_buffer();
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        string raw_buffer lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        let text, raw = get_stored_string() in
        STRING (text, raw, None) )
# 2363 "src/reason-parser/reason_lexer.ml"

  | 18 ->
# 470 "src/reason-parser/reason_lexer.mll"
      ( reset_string_buffer();
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        quoted_string delim lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (fst (get_stored_string()), None, Some delim) )
# 2377 "src/reason-parser/reason_lexer.ml"

  | 19 ->
# 481 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        CHAR (Lexing.lexeme_char lexbuf 1) )
# 2383 "src/reason-parser/reason_lexer.ml"

  | 20 ->
# 484 "src/reason-parser/reason_lexer.mll"
      ( CHAR(Lexing.lexeme_char lexbuf 1) )
# 2388 "src/reason-parser/reason_lexer.ml"

  | 21 ->
# 486 "src/reason-parser/reason_lexer.mll"
      ( CHAR(char_for_backslash (Lexing.lexeme_char lexbuf 2)) )
# 2393 "src/reason-parser/reason_lexer.ml"

  | 22 ->
# 488 "src/reason-parser/reason_lexer.mll"
      ( CHAR(char_for_decimal_code lexbuf 2) )
# 2398 "src/reason-parser/reason_lexer.ml"

  | 23 ->
# 490 "src/reason-parser/reason_lexer.mll"
      ( CHAR(char_for_hexadecimal_code lexbuf 3) )
# 2403 "src/reason-parser/reason_lexer.ml"

  | 24 ->
# 492 "src/reason-parser/reason_lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let esc = String.sub l 1 (String.length l - 1) in
        raise (Error(Illegal_escape esc, Location.curr lexbuf))
      )
# 2411 "src/reason-parser/reason_lexer.ml"

  | 25 ->
# 496 "src/reason-parser/reason_lexer.mll"
          (
    (* Allow parsing of foo#=<bar /> *)
    set_lexeme_length lexbuf 2;
    SHARPOP("#=")
  )
# 2420 "src/reason-parser/reason_lexer.ml"

  | 26 ->
# 502 "src/reason-parser/reason_lexer.mll"
      ( SHARPOP(lexeme_operator lexbuf) )
# 2425 "src/reason-parser/reason_lexer.ml"

  | 27 ->
let
# 503 "src/reason-parser/reason_lexer.mll"
                                   num
# 2431 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 504 "src/reason-parser/reason_lexer.mll"
                                           name
# 2436 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2) in
# 506 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf name (int_of_string num) true 0;
        token lexbuf
      )
# 2442 "src/reason-parser/reason_lexer.ml"

  | 28 ->
# 509 "src/reason-parser/reason_lexer.mll"
         ( AMPERSAND )
# 2447 "src/reason-parser/reason_lexer.ml"

  | 29 ->
# 510 "src/reason-parser/reason_lexer.mll"
         ( AMPERAMPER )
# 2452 "src/reason-parser/reason_lexer.ml"

  | 30 ->
# 511 "src/reason-parser/reason_lexer.mll"
         ( BACKQUOTE )
# 2457 "src/reason-parser/reason_lexer.ml"

  | 31 ->
# 512 "src/reason-parser/reason_lexer.mll"
         ( QUOTE )
# 2462 "src/reason-parser/reason_lexer.ml"

  | 32 ->
# 513 "src/reason-parser/reason_lexer.mll"
         ( LPAREN )
# 2467 "src/reason-parser/reason_lexer.ml"

  | 33 ->
# 514 "src/reason-parser/reason_lexer.mll"
         ( RPAREN )
# 2472 "src/reason-parser/reason_lexer.ml"

  | 34 ->
# 515 "src/reason-parser/reason_lexer.mll"
         ( STAR )
# 2477 "src/reason-parser/reason_lexer.ml"

  | 35 ->
# 516 "src/reason-parser/reason_lexer.mll"
         ( COMMA )
# 2482 "src/reason-parser/reason_lexer.ml"

  | 36 ->
# 517 "src/reason-parser/reason_lexer.mll"
         ( MINUSGREATER )
# 2487 "src/reason-parser/reason_lexer.ml"

  | 37 ->
# 518 "src/reason-parser/reason_lexer.mll"
         ( EQUALGREATER )
# 2492 "src/reason-parser/reason_lexer.ml"

  | 38 ->
# 519 "src/reason-parser/reason_lexer.mll"
         ( SHARP )
# 2497 "src/reason-parser/reason_lexer.ml"

  | 39 ->
# 520 "src/reason-parser/reason_lexer.mll"
         ( DOT )
# 2502 "src/reason-parser/reason_lexer.ml"

  | 40 ->
# 521 "src/reason-parser/reason_lexer.mll"
         ( DOTDOT )
# 2507 "src/reason-parser/reason_lexer.ml"

  | 41 ->
# 522 "src/reason-parser/reason_lexer.mll"
         ( DOTDOTDOT )
# 2512 "src/reason-parser/reason_lexer.ml"

  | 42 ->
# 523 "src/reason-parser/reason_lexer.mll"
         ( COLON )
# 2517 "src/reason-parser/reason_lexer.ml"

  | 43 ->
# 524 "src/reason-parser/reason_lexer.mll"
         ( COLONCOLON )
# 2522 "src/reason-parser/reason_lexer.ml"

  | 44 ->
# 525 "src/reason-parser/reason_lexer.mll"
         ( COLONEQUAL )
# 2527 "src/reason-parser/reason_lexer.ml"

  | 45 ->
# 526 "src/reason-parser/reason_lexer.mll"
         ( COLONGREATER )
# 2532 "src/reason-parser/reason_lexer.ml"

  | 46 ->
# 527 "src/reason-parser/reason_lexer.mll"
         ( SEMI )
# 2537 "src/reason-parser/reason_lexer.ml"

  | 47 ->
# 528 "src/reason-parser/reason_lexer.mll"
         ( SEMISEMI )
# 2542 "src/reason-parser/reason_lexer.ml"

  | 48 ->
# 529 "src/reason-parser/reason_lexer.mll"
         ( LESS )
# 2547 "src/reason-parser/reason_lexer.ml"

  | 49 ->
# 530 "src/reason-parser/reason_lexer.mll"
         ( EQUAL )
# 2552 "src/reason-parser/reason_lexer.ml"

  | 50 ->
# 531 "src/reason-parser/reason_lexer.mll"
         ( LBRACKET )
# 2557 "src/reason-parser/reason_lexer.ml"

  | 51 ->
# 532 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETBAR )
# 2562 "src/reason-parser/reason_lexer.ml"

  | 52 ->
# 533 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETLESS )
# 2567 "src/reason-parser/reason_lexer.ml"

  | 53 ->
# 534 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETGREATER )
# 2572 "src/reason-parser/reason_lexer.ml"

  | 54 ->
# 535 "src/reason-parser/reason_lexer.mll"
                                                         (
    let buf = Lexing.lexeme lexbuf in
    LESSIDENT (String.sub buf 1 (String.length buf - 1))
  )
# 2580 "src/reason-parser/reason_lexer.ml"

  | 55 ->
# 539 "src/reason-parser/reason_lexer.mll"
                                                    (
    (* allows parsing of `{<Text` in <Description term={<Text text="Age" />}> as correct jsx *)
    set_lexeme_length lexbuf 1;
    LBRACE
  )
# 2589 "src/reason-parser/reason_lexer.ml"

  | 56 ->
# 544 "src/reason-parser/reason_lexer.mll"
                                                        (
    let buf = Lexing.lexeme lexbuf in
    LESSSLASHIDENTGREATER (String.sub buf 2 (String.length buf - 2 - 1))
  )
# 2597 "src/reason-parser/reason_lexer.ml"

  | 57 ->
# 548 "src/reason-parser/reason_lexer.mll"
         ( RBRACKET )
# 2602 "src/reason-parser/reason_lexer.ml"

  | 58 ->
# 549 "src/reason-parser/reason_lexer.mll"
         ( LBRACE )
# 2607 "src/reason-parser/reason_lexer.ml"

  | 59 ->
# 550 "src/reason-parser/reason_lexer.mll"
         ( LBRACELESS )
# 2612 "src/reason-parser/reason_lexer.ml"

  | 60 ->
# 551 "src/reason-parser/reason_lexer.mll"
         ( BAR )
# 2617 "src/reason-parser/reason_lexer.ml"

  | 61 ->
# 552 "src/reason-parser/reason_lexer.mll"
         ( BARBAR )
# 2622 "src/reason-parser/reason_lexer.ml"

  | 62 ->
# 553 "src/reason-parser/reason_lexer.mll"
         ( BARRBRACKET )
# 2627 "src/reason-parser/reason_lexer.ml"

  | 63 ->
# 554 "src/reason-parser/reason_lexer.mll"
         ( GREATER )
# 2632 "src/reason-parser/reason_lexer.ml"

  | 64 ->
# 560 "src/reason-parser/reason_lexer.mll"
         ( RBRACE )
# 2637 "src/reason-parser/reason_lexer.ml"

  | 65 ->
# 561 "src/reason-parser/reason_lexer.mll"
         ( GREATERRBRACE )
# 2642 "src/reason-parser/reason_lexer.ml"

  | 66 ->
# 562 "src/reason-parser/reason_lexer.mll"
                                 (
    (* allow `let x=<div />;` *)
    set_lexeme_length lexbuf 1;
    EQUAL
  )
# 2651 "src/reason-parser/reason_lexer.ml"

  | 67 ->
# 568 "src/reason-parser/reason_lexer.mll"
           (
    set_lexeme_length lexbuf 2;
    SLASHGREATER
  )
# 2659 "src/reason-parser/reason_lexer.ml"

  | 68 ->
# 572 "src/reason-parser/reason_lexer.mll"
          (
    set_lexeme_length lexbuf 2;
    LBRACKETBAR
  )
# 2667 "src/reason-parser/reason_lexer.ml"

  | 69 ->
# 577 "src/reason-parser/reason_lexer.mll"
                                   (
    (* allow parsing of <div asd=1></div> *)
    set_lexeme_length lexbuf 2;
    SLASHGREATER
  )
# 2676 "src/reason-parser/reason_lexer.ml"

  | 70 ->
# 582 "src/reason-parser/reason_lexer.mll"
                                  (
    (* allow parsing of <div asd=1></div> *)
    set_lexeme_length lexbuf 1;
    GREATER
  )
# 2685 "src/reason-parser/reason_lexer.ml"

  | 71 ->
# 587 "src/reason-parser/reason_lexer.mll"
                                 (
    (* allow parsing of <div><span> *)
    set_lexeme_length lexbuf 1;
    GREATER
  )
# 2694 "src/reason-parser/reason_lexer.ml"

  | 72 ->
# 592 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETAT )
# 2699 "src/reason-parser/reason_lexer.ml"

  | 73 ->
# 593 "src/reason-parser/reason_lexer.mll"
         ( LBRACKETPERCENT )
# 2704 "src/reason-parser/reason_lexer.ml"

  | 74 ->
# 594 "src/reason-parser/reason_lexer.mll"
          ( LBRACKETPERCENTPERCENT )
# 2709 "src/reason-parser/reason_lexer.ml"

  | 75 ->
# 595 "src/reason-parser/reason_lexer.mll"
         ( BANG )
# 2714 "src/reason-parser/reason_lexer.ml"

  | 76 ->
# 596 "src/reason-parser/reason_lexer.mll"
         ( INFIXOP0 "!=" )
# 2719 "src/reason-parser/reason_lexer.ml"

  | 77 ->
# 597 "src/reason-parser/reason_lexer.mll"
          ( INFIXOP0 "!==" )
# 2724 "src/reason-parser/reason_lexer.ml"

  | 78 ->
# 598 "src/reason-parser/reason_lexer.mll"
           ( INFIXOP0 "!=" )
# 2729 "src/reason-parser/reason_lexer.ml"

  | 79 ->
# 599 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP0 "!==" )
# 2734 "src/reason-parser/reason_lexer.ml"

  | 80 ->
# 600 "src/reason-parser/reason_lexer.mll"
         ( PLUS )
# 2739 "src/reason-parser/reason_lexer.ml"

  | 81 ->
# 601 "src/reason-parser/reason_lexer.mll"
         ( PLUSDOT )
# 2744 "src/reason-parser/reason_lexer.ml"

  | 82 ->
# 602 "src/reason-parser/reason_lexer.mll"
         ( PLUSEQ )
# 2749 "src/reason-parser/reason_lexer.ml"

  | 83 ->
# 603 "src/reason-parser/reason_lexer.mll"
         ( MINUS )
# 2754 "src/reason-parser/reason_lexer.ml"

  | 84 ->
# 604 "src/reason-parser/reason_lexer.mll"
         ( MINUSDOT )
# 2759 "src/reason-parser/reason_lexer.ml"

  | 85 ->
# 605 "src/reason-parser/reason_lexer.mll"
         ( LESSGREATER )
# 2764 "src/reason-parser/reason_lexer.ml"

  | 86 ->
# 606 "src/reason-parser/reason_lexer.mll"
          ( LESSSLASHGREATER )
# 2769 "src/reason-parser/reason_lexer.ml"

  | 87 ->
# 607 "src/reason-parser/reason_lexer.mll"
           ( LESSDOTDOTGREATER )
# 2774 "src/reason-parser/reason_lexer.ml"

  | 88 ->
# 609 "src/reason-parser/reason_lexer.mll"
            ( PREFIXOP(lexeme_operator lexbuf) )
# 2779 "src/reason-parser/reason_lexer.ml"

  | 89 ->
# 611 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP0(lexeme_operator lexbuf) )
# 2784 "src/reason-parser/reason_lexer.ml"

  | 90 ->
# 613 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP1(lexeme_operator lexbuf) )
# 2789 "src/reason-parser/reason_lexer.ml"

  | 91 ->
# 615 "src/reason-parser/reason_lexer.mll"
            ( match lexeme_without_comment lexbuf with
              | "^." -> set_lexeme_length lexbuf 1; POSTFIXOP("^")
              | "^" -> POSTFIXOP("^")
              | op -> INFIXOP1(unescape_operator op) )
# 2797 "src/reason-parser/reason_lexer.ml"

  | 92 ->
# 620 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP1(lexeme_operator lexbuf) )
# 2802 "src/reason-parser/reason_lexer.ml"

  | 93 ->
# 622 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP2(lexeme_operator lexbuf) )
# 2807 "src/reason-parser/reason_lexer.ml"

  | 94 ->
# 624 "src/reason-parser/reason_lexer.mll"
         ( SLASHGREATER )
# 2812 "src/reason-parser/reason_lexer.ml"

  | 95 ->
# 635 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP4(lexeme_operator lexbuf) )
# 2817 "src/reason-parser/reason_lexer.ml"

  | 96 ->
# 636 "src/reason-parser/reason_lexer.mll"
            ( PERCENT )
# 2822 "src/reason-parser/reason_lexer.ml"

  | 97 ->
# 638 "src/reason-parser/reason_lexer.mll"
            ( match lexeme_operator lexbuf with
              | "" ->
                  (* If the operator is empty, it means the lexeme is beginning
                   * by a comment sequence: we let the comment lexer handle
                   * the case. *)
                  enter_comment lexbuf
              | op -> INFIXOP3 op )
# 2833 "src/reason-parser/reason_lexer.ml"

  | 98 ->
# 646 "src/reason-parser/reason_lexer.mll"
            ( INFIXOP3(lexeme_operator lexbuf) )
# 2838 "src/reason-parser/reason_lexer.ml"

  | 99 ->
# 647 "src/reason-parser/reason_lexer.mll"
        ( EOF )
# 2843 "src/reason-parser/reason_lexer.ml"

  | 100 ->
# 649 "src/reason-parser/reason_lexer.mll"
      ( raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      )
# 2850 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and enter_comment lexbuf =
    __ocaml_lex_enter_comment_rec lexbuf 173
and __ocaml_lex_enter_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 655 "src/reason-parser/reason_lexer.mll"
      ( set_lexeme_length lexbuf 2;
        let start_loc = Location.curr lexbuf  in
        comment_start_loc := [start_loc];
        reset_string_buffer ();
        let {Location. loc_end; _} = comment lexbuf in
        let s, _ = get_stored_string () in
        reset_string_buffer ();
        COMMENT (s, { start_loc with Location.loc_end })
      )
# 2870 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 665 "src/reason-parser/reason_lexer.mll"
      ( let start_p = lexbuf.Lexing.lex_start_p in
        let start_loc = Location.curr lexbuf in
        comment_start_loc := [start_loc];
        reset_string_buffer ();
        let _ = comment lexbuf in
        let s, _ = get_stored_string () in
        reset_string_buffer ();
        lexbuf.Lexing.lex_start_p <- start_p;
        DOCSTRING s
      )
# 2884 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 676 "src/reason-parser/reason_lexer.mll"
      ( DOCSTRING "" )
# 2889 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 678 "src/reason-parser/reason_lexer.mll"
      ( let loc = Location.curr lexbuf  in
        if !print_warnings then
          Location.prerr_warning loc Warnings.Comment_start;
        comment_start_loc := [loc];
        reset_string_buffer ();
        let {Location. loc_end; _} = comment lexbuf in
        let s, _ = get_stored_string () in
        reset_string_buffer ();
        COMMENT (s, { loc with Location.loc_end })
      )
# 2903 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 689 "src/reason-parser/reason_lexer.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_not_end;
        set_lexeme_length lexbuf 1;
        STAR
      )
# 2912 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 694 "src/reason-parser/reason_lexer.mll"
      ( assert false )
# 2917 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_enter_comment_rec lexbuf __ocaml_lex_state

and comment lexbuf =
    __ocaml_lex_comment_rec lexbuf 183
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 698 "src/reason-parser/reason_lexer.mll"
      ( comment_start_loc := (Location.curr lexbuf) :: !comment_start_loc;
        store_lexeme string_buffer lexbuf;
        comment lexbuf;
      )
# 2932 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 703 "src/reason-parser/reason_lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | [_] -> comment_start_loc := []; Location.curr lexbuf
        | _ :: l -> comment_start_loc := l;
                  store_lexeme string_buffer lexbuf;
                  comment lexbuf;
       )
# 2943 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 711 "src/reason-parser/reason_lexer.mll"
      (
        string_start_loc := Location.curr lexbuf;
        store_string_char '"';
        is_in_string := true;
        begin try string string_buffer lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '"';
        comment lexbuf )
# 2964 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 729 "src/reason-parser/reason_lexer.mll"
      (
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        string_start_loc := Location.curr lexbuf;
        store_lexeme string_buffer lexbuf;
        is_in_string := true;
        begin try quoted_string delim lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '|';
        store_string delim;
        store_string_char '}';
        comment lexbuf )
# 2989 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 752 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme string_buffer lexbuf; comment lexbuf )
# 2994 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 754 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        store_lexeme string_buffer lexbuf;
        comment lexbuf
      )
# 3002 "src/reason-parser/reason_lexer.ml"

  | 6 ->
# 759 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme string_buffer lexbuf; comment lexbuf )
# 3007 "src/reason-parser/reason_lexer.ml"

  | 7 ->
# 761 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme string_buffer lexbuf; comment lexbuf )
# 3012 "src/reason-parser/reason_lexer.ml"

  | 8 ->
# 763 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme string_buffer lexbuf; comment lexbuf )
# 3017 "src/reason-parser/reason_lexer.ml"

  | 9 ->
# 765 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme string_buffer lexbuf; comment lexbuf )
# 3022 "src/reason-parser/reason_lexer.ml"

  | 10 ->
# 767 "src/reason-parser/reason_lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | loc :: _ ->
          let start = List.hd (List.rev !comment_start_loc) in
          comment_start_loc := [];
          raise (Error (Unterminated_comment start, loc))
      )
# 3033 "src/reason-parser/reason_lexer.ml"

  | 11 ->
# 775 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme string_buffer lexbuf;
        comment lexbuf
      )
# 3041 "src/reason-parser/reason_lexer.ml"

  | 12 ->
# 780 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme string_buffer lexbuf; comment lexbuf )
# 3046 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string rawbuffer lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1) ;   __ocaml_lex_string_rec rawbuffer lexbuf 215
and __ocaml_lex_string_rec rawbuffer lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 784 "src/reason-parser/reason_lexer.mll"
      ( () )
# 3058 "src/reason-parser/reason_lexer.ml"

  | 1 ->
let
# 785 "src/reason-parser/reason_lexer.mll"
                                  space
# 3064 "src/reason-parser/reason_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 786 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        store_lexeme rawbuffer lexbuf;
        string rawbuffer lexbuf
      )
# 3071 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 791 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme rawbuffer lexbuf;
        if not (in_comment ()) then
          store_string_char(char_for_backslash(Lexing.lexeme_char lexbuf 1));
        string rawbuffer lexbuf )
# 3079 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 796 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme rawbuffer lexbuf;
        if not (in_comment ()) then
          store_string_char(char_for_decimal_code lexbuf 1);
        string rawbuffer lexbuf )
# 3087 "src/reason-parser/reason_lexer.ml"

  | 4 ->
# 801 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme rawbuffer lexbuf;
        if not (in_comment ()) then
          store_string_char(char_for_hexadecimal_code lexbuf 2);
        string rawbuffer lexbuf )
# 3095 "src/reason-parser/reason_lexer.ml"

  | 5 ->
# 806 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme rawbuffer lexbuf;
        if not (in_comment ()) then
        begin
(*  Should be an error, but we are very lax.
          raise (Error (Illegal_escape (Lexing.lexeme lexbuf),
                        Location.curr lexbuf))
*)
          let loc = Location.curr lexbuf in
          Location.prerr_warning loc Warnings.Illegal_backslash;
          store_string_char (Lexing.lexeme_char lexbuf 0);
          store_string_char (Lexing.lexeme_char lexbuf 1);
        end;
        string rawbuffer lexbuf
      )
# 3113 "src/reason-parser/reason_lexer.ml"

  | 6 ->
# 821 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme rawbuffer lexbuf;
        if not (in_comment ()) then (
          store_lexeme string_buffer lexbuf;
          Location.prerr_warning (Location.curr lexbuf) Warnings.Eol_in_string
        );
        update_loc lexbuf None 1 false 0;
        string rawbuffer lexbuf
      )
# 3125 "src/reason-parser/reason_lexer.ml"

  | 7 ->
# 830 "src/reason-parser/reason_lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 3131 "src/reason-parser/reason_lexer.ml"

  | 8 ->
# 833 "src/reason-parser/reason_lexer.mll"
      ( store_lexeme rawbuffer lexbuf;
        if not (in_comment ()) then
          store_string_char(Lexing.lexeme_char lexbuf 0);
        string rawbuffer lexbuf )
# 3139 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_string_rec rawbuffer lexbuf __ocaml_lex_state

and quoted_string delim lexbuf =
    __ocaml_lex_quoted_string_rec delim lexbuf 234
and __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 840 "src/reason-parser/reason_lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme string_buffer lexbuf;
        quoted_string delim lexbuf
      )
# 3154 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 845 "src/reason-parser/reason_lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 3160 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 848 "src/reason-parser/reason_lexer.mll"
      (
        let edelim = Lexing.lexeme lexbuf in
        let edelim = String.sub edelim 1 (String.length edelim - 2) in
        if delim = edelim then ()
        else (store_lexeme string_buffer lexbuf;
              quoted_string delim lexbuf)
      )
# 3171 "src/reason-parser/reason_lexer.ml"

  | 3 ->
# 856 "src/reason-parser/reason_lexer.mll"
      ( store_string_char (Lexing.lexeme_char lexbuf 0);
        quoted_string delim lexbuf )
# 3177 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state

and skip_sharp_bang lexbuf =
    __ocaml_lex_skip_sharp_bang_rec lexbuf 243
and __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 861 "src/reason-parser/reason_lexer.mll"
       ( update_loc lexbuf None 3 false 0 )
# 3189 "src/reason-parser/reason_lexer.ml"

  | 1 ->
# 863 "src/reason-parser/reason_lexer.mll"
       ( update_loc lexbuf None 1 false 0 )
# 3194 "src/reason-parser/reason_lexer.ml"

  | 2 ->
# 864 "src/reason-parser/reason_lexer.mll"
       ( () )
# 3199 "src/reason-parser/reason_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; 
      __ocaml_lex_skip_sharp_bang_rec lexbuf __ocaml_lex_state

;;

# 866 "src/reason-parser/reason_lexer.mll"
 

  (* Filter commnets *)

  let token_with_comments lexbuf =
    match !preprocessor with
    | None -> token lexbuf
    | Some (_init, preprocess) -> preprocess token lexbuf

  let last_comments = ref []

  let rec token lexbuf =
    match token_with_comments lexbuf with
        COMMENT (s, comment_loc) ->
          last_comments := (s, comment_loc) :: !last_comments;
          token lexbuf
      | tok -> tok

  let add_invalid_docstring text loc =
    let open Location in
    let rec aux = function
      | ((_, loc') as x) :: xs
        when loc'.loc_start.pos_cnum > loc.loc_start.pos_cnum ->
        x :: aux xs
      | xs -> (text, loc) :: xs
    in
    last_comments := aux !last_comments

  let comments () = List.rev !last_comments

  (* Routines for manipulating lexer state *)

  let save_triple lexbuf tok =
    (tok, lexbuf.lex_start_p, lexbuf.lex_curr_p)

  let load_triple lexbuf (tok, p1, p2) = (
    lexbuf.lex_start_p <- p1;
    lexbuf.lex_curr_p <- p2;
    tok
  )

  let fake_triple t (_, pos, _) =
    (t, pos, pos)

  (* insert ES6_FUN *)

  exception Lex_balanced_failed of (token * position * position) list *
                                   (exn * position * position) option

  let closing_of = function
    | LPAREN -> RPAREN
    | LBRACE -> RBRACE
    | _ -> assert false

  let inject_es6_fun = function
    | tok :: acc ->
      tok :: fake_triple ES6_FUN tok :: acc
    | _ -> assert false

  let is_triggering_token = function
    | EQUALGREATER | COLON -> true
    | _ -> false

  let rec lex_balanced_step closing lexbuf acc tok =
    let acc = save_triple lexbuf tok :: acc in
    match tok, closing with
    | (RPAREN, RPAREN) | (RBRACE, RBRACE) | (RBRACKET, RBRACKET) ->
      acc
    | ((RPAREN | RBRACE | RBRACKET | EOF), _) ->
      raise (Lex_balanced_failed (acc, None))
    | (( LBRACKET | LBRACKETLESS | LBRACKETGREATER
       | LBRACKETAT
       | LBRACKETPERCENT | LBRACKETPERCENTPERCENT ), _) ->
      lex_balanced closing lexbuf (lex_balanced RBRACKET lexbuf acc)
    | ((LPAREN | LBRACE), _) ->
      let rparen =
        try lex_balanced (closing_of tok) lexbuf []
        with (Lex_balanced_failed (rparen, None)) ->
          raise (Lex_balanced_failed (rparen @ acc, None))
      in
      begin match token lexbuf with
      | exception exn ->
        raise (Lex_balanced_failed (rparen @ acc, Some (save_triple lexbuf exn)))
      | tok' ->
        let acc = if is_triggering_token tok' then inject_es6_fun acc else acc in
        lex_balanced_step closing lexbuf (rparen @ acc) tok'
      end
    | ((LIDENT _ | UNDERSCORE), _) ->
      begin match token lexbuf with
      | exception exn ->
        raise (Lex_balanced_failed (acc, Some (save_triple lexbuf exn)))
      | tok' ->
        let acc = if is_triggering_token tok' then inject_es6_fun acc else acc in
        lex_balanced_step closing lexbuf acc tok'
      end
    | _ -> lex_balanced closing lexbuf acc

  and lex_balanced closing lexbuf acc =
    match token lexbuf with
    | exception exn ->
      raise (Lex_balanced_failed (acc, Some (save_triple lexbuf exn)))
    | tok -> lex_balanced_step closing lexbuf acc tok

  let queued_tokens = ref []
  let queued_exn = ref None

  let lookahead_esfun lexbuf (tok, _, _ as lparen) =
    let triple =
      match lex_balanced (closing_of tok) lexbuf [] with
      | exception (Lex_balanced_failed (tokens, exn)) ->
        queued_tokens := List.rev tokens;
        queued_exn := exn;
        lparen
      | tokens ->
        begin match token lexbuf with
          | exception exn ->
            queued_tokens := List.rev tokens;
            queued_exn := Some (save_triple lexbuf exn);
            lparen
          | token ->
            let tokens = save_triple lexbuf token :: tokens in
            if is_triggering_token token then (
              queued_tokens := lparen :: List.rev tokens;
              fake_triple ES6_FUN lparen
            ) else (
              queued_tokens := List.rev tokens;
              lparen
            )
        end
    in
    load_triple lexbuf triple

  let token lexbuf =
    match !queued_tokens, !queued_exn with
    | [], Some exn ->
      queued_exn := None;
      raise (load_triple lexbuf exn)
    | [(LPAREN, _, _) as lparen], None ->
      let _ = load_triple lexbuf lparen in
      lookahead_esfun lexbuf lparen
    | [(LBRACE, _, _) as lparen], None ->
      let _ = load_triple lexbuf lparen in
      lookahead_esfun lexbuf lparen
    | [], None ->
      begin match token lexbuf with
      | LPAREN | LBRACE as tok ->
          lookahead_esfun lexbuf (save_triple lexbuf tok)
      | (LIDENT _ | UNDERSCORE) as tok ->
          let tok = save_triple lexbuf tok in
          begin match token lexbuf with
          | exception exn ->
              queued_exn := Some (save_triple lexbuf exn);
              load_triple lexbuf tok
          | tok' ->
            if is_triggering_token tok' then (
              queued_tokens := [tok; save_triple lexbuf tok'];
              load_triple lexbuf (fake_triple ES6_FUN tok)
            ) else (
              queued_tokens := [save_triple lexbuf tok'];
              load_triple lexbuf tok
            )
          end
      | token -> token
      end
    | x :: xs, _ -> queued_tokens := xs; load_triple lexbuf x

  let completion_ident_offset = ref min_int
  let completion_ident_pos = ref Lexing.dummy_pos

  let token lexbuf =
    let before = lexbuf.Lexing.lex_curr_p.Lexing.pos_cnum in
    let token = token lexbuf in
    let after = lexbuf.Lexing.lex_start_p.Lexing.pos_cnum in
    if !completion_ident_offset > min_int &&
       before <= !completion_ident_offset &&
       after >= !completion_ident_offset then (
      match token with
      | LIDENT _ | UIDENT _ when after = !completion_ident_offset -> token
      | _ ->
        queued_tokens := save_triple lexbuf token :: !queued_tokens;
        completion_ident_offset := min_int;
        load_triple lexbuf
          (LIDENT "_", !completion_ident_pos, !completion_ident_pos)
    ) else
      token

  let init ?insert_completion_ident () =
    is_in_string := false;
    last_comments := [];
    comment_start_loc := [];
    queued_tokens := [];
    queued_exn := None;
    begin match insert_completion_ident with
      | None ->
        completion_ident_offset := min_int;
      | Some pos ->
        completion_ident_offset := pos.Lexing.pos_cnum;
        completion_ident_pos := pos
    end;
    match !preprocessor with
    | None -> ()
    | Some (init, _preprocess) -> init ()

  let set_preprocessor init preprocess =
    preprocessor := Some (init, preprocess)


# 3414 "src/reason-parser/reason_lexer.ml"
